/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.tasks;

import com.tyr.EnemyManager;
import com.tyr.OrderedExpands;
import com.tyr.PositionUtil;
import com.tyr.Tyr;
import com.tyr.agents.Agent;
import com.tyr.unitgroups.AttackGroup;
import bwapi.Color;
import bwapi.Game;
import bwapi.Player;
import bwapi.Position;
import bwapi.Unit;
import bwapi.UnitType;

/**
 * This class implements a solution for constantly pushing the opponent.
 */
public class DragoonSquadSolution extends Solution
{
	/**
	 * The OrderedExpands object for our clockwise scouting observer.
	 */
	private OrderedExpands cwExpands = new OrderedExpands(false, false, false, true);
	
	private AttackGroup attackGroup;
	
	private Agent observer;
	
	/**
	 * This class implements a solution for constantly pushing the opponent.
	 * @param task The task that started this solution.
	 */
	public DragoonSquadSolution(Task task) 
	{
		super(task);
	}
	
	@Override
	public void onFrame(Game game, Player self, Tyr bot)
	{
		final Position target = getTarget();
		if (target == null)
			return;
		
		if (attackGroup == null)
			 attackGroup = new AttackGroup(target);
		
		attackGroup.setTarget(target);
		attackGroup.cleanup();
		attackGroup.onFrame(game, self, bot);
		
		if (observer != null && observer.isDead())
			observer = null;
		
		if (observer != null)
		{
			Position observerPos = null;
			int dist = Integer.MAX_VALUE;
			for (Agent agent : attackGroup.units)
			{
				final int newDist = agent.distanceSquared(target); 
				if (newDist < dist)
				{
					dist = newDist;
					observerPos = agent.unit.getPosition();
				}
			}
			
			if (observerPos != null)
			{
				observer.move(observerPos);
				game.drawLineMap(observerPos, observer.unit.getPosition(), Color.Green);
			}
		}
	}
	
	/**
	 * Add an agent to the attack group with the attacking agents.
	 * @param unit The agent to be added.
	 */
	public void add(Agent agent)
	{
		if (attackGroup == null)
			 attackGroup = new AttackGroup(null);
		
		if (agent.unit.getType() == UnitType.Protoss_Observer)
		{
			if (observer == null || observer.isDead())
				observer = agent;
			else
				Tyr.bot.hobos.add(agent);
		}
		else
			attackGroup.add(agent);
	}
	
	/**
	 * Gets us the next target to harass.
	 */
	private Position getTarget()
	{
		if (!cwExpands.tryInitialize())
			return null;

		final Position current = cwExpands.getPos();
		
		if (current == null)
			cwExpands.next();
		else
		{
			if (!Tyr.game.isVisible(Tyr.positionToTile(current)))
				return current;
			for (Unit unit : EnemyManager.getEnemyUnits())
			{
				if (!unit.getType().isWorker() && !unit.getType().isResourceDepot())
					continue;
				if (PositionUtil.distanceSq(unit, current) <= 200 * 200)
					return current;
			}
			cwExpands.next();
		}
		
		return cwExpands.getPos();
	}
	
	public int count()
	{
		if (attackGroup == null)
			return 0;
		return attackGroup.units.size();
	}
}
