/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.tasks;

import java.util.ArrayList;
import java.util.List;

import com.tyr.EnemyManager;
import com.tyr.PositionUtil;
import com.tyr.Tyr;
import com.tyr.agents.Agent;
import com.tyr.unitgroups.MineralWorkers;

import bwapi.Color;
import bwapi.Game;
import bwapi.Player;
import bwapi.Position;
import bwapi.Unit;
import bwapi.UnitType;

/**
 * This class implements a solution for defending our bases using air units.
 */
public class AirDefenseSolution extends Solution
{
	private List<Agent> units = new ArrayList<Agent>();
	
	private Unit target = null;
	
	/**
	 * This class implements a solution for defending our bases using air units.
	 * @param task The task that started this solution.
	 */
	public AirDefenseSolution(Task task) 
	{
		super(task);
	}
	
	@Override
	public void onFrame(Game game, Player self, Tyr bot)
	{
		for (int i=units.size() - 1; i >= 0; i--)
			if (units.get(i) == null || units.get(i).isDead())
				units.remove(i);
		
		if (target != null &&
				(target.getHitPoints() <= 0 || target.getRemoveTimer() > 0 || !target.exists()))
		{
			Position lastSeen = EnemyManager.getManager().getLastPosition(target);
			
			if (lastSeen == null)
				target = null;
			else
			{
				for (Agent agent : units)
				{
					if (agent.distanceSquared(lastSeen) <= 100 * 100)
					{
						target = null;
						break;
					}
				}
			}
		}
		
		if (target == null)
		{
			for (MineralWorkers base : bot.workForce.mineralWorkers)
			{
				if (base.resourceDepot == null)
					continue;
				
				for (Unit unit : EnemyManager.getEnemyUnits())
					if ((unit.getType() == UnitType.Protoss_Reaver || unit.getType() == UnitType.Protoss_Shuttle || unit.getType() == UnitType.Protoss_High_Templar)
							&& PositionUtil.distanceSq(unit, base.resourceDepot) <= 400 * 400)
					{
						target = unit;
						break;
					}
				if (target != null)
					break;
			}
		}
		
		if (target != null)
			Tyr.drawCircle(target.getPosition(), Color.Red, 16);
		
		for (Agent agent : units)
		{
			if (target != null && (target.getHitPoints() <= 0 || target.getRemoveTimer() > 0 || !target.exists()))
				agent.unit.move(EnemyManager.getManager().getLastPosition(target));
			else if (target != null)
				agent.attack(target);
			else
				agent.unit.move(Tyr.tileToPosition(self.getStartLocation()));
		}
	}
	
	/**
	 * Add an agent to the list of defending units.
	 * @param unit The agent to be added.
	 */
	public void add(Agent agent)
	{
		units.add(agent);
	}

	/**
	 * Returns the number if agents sent out by this solution.
	 * @return Returns the number if agents sent out by this solution.
	 */
	public int size() 
	{
		return units.size();
	}
	
	public boolean done() 
	{
		return size() == 0;
	}
}
