/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.builds;

import com.tyr.BWTAProxy;
import com.tyr.DebugMessages;
import com.tyr.Settings;
import com.tyr.StopWatch;
import com.tyr.Tyr;
import com.tyr.UnitTracker;
import com.tyr.agents.Agent;
import com.tyr.buildingplacement.DefensiveStructures;
import com.tyr.tasks.LayMinesTask;
import com.tyr.tasks.VultureHarassTask;

import bwapi.Game;
import bwapi.Player;
import bwapi.Race;
import bwapi.TechType;
import bwapi.UnitType;
import bwapi.UpgradeType;


public class Mech extends CompositeBuildOrder
{
	/**
	 * If this is true we only use a limited number of bunkers for defense.
	 */
	private boolean limitedBunkers = false;
	
	public Mech()
	{}
	
	public Mech(boolean limitedBunkers)
	{
		this.limitedBunkers = limitedBunkers;
	}
	
	@Override
	public void initialize(Game game, Player self, Tyr bot)
	{
		Settings.setRequiredSize(20);
		Settings.setMaximumSize(40);
		
		Settings.setWorkersPerGas(1);
		
		
		this.add(new SupplyDepotPart());
		this.add(new WorkerScoutPart(1600));
		this.add(new ExpandPart());
		
		bot.taskManager.potentialTasks.add(new LayMinesTask());
		bot.taskManager.potentialTasks.add(new VultureHarassTask());
		
		super.initialize(game, self, bot);
	}
	
	@Override
	public void onFrame(Game game, Player self, Tyr bot)
	{
		DebugMessages.addMessage("Going mech.");
		
		StopWatch watch = new StopWatch();
		long time;
		watch.start();
		
		super.onFrame(game, self, bot);
		
		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Mech super time: " + time);
		
		
		if(UnitTracker.count(UnitType.Terran_Factory) >= 2)
			Settings.setWorkersPerGas(2);
		else
			Settings.setWorkersPerGas(1);
		
		//if we've the resources to build a barracks ...
		if(bot.getAvailableMinerals()>= 150 && (UnitTracker.count(UnitType.Terran_Barracks) == 0)) 
		{
			bot.spaceManager.build(UnitType.Terran_Barracks);
		}
		
		if(UnitTracker.count(UnitType.Terran_Refinery) < UnitTracker.count(UnitType.Terran_Command_Center) 
				&& bot.getAvailableMinerals()>= 100 
				&& UnitTracker.getGeyserCount() > 0 
				&& UnitTracker.count(UnitType.Terran_Barracks) >= 1) 
		{
			bot.spaceManager.build(UnitType.Terran_Refinery);
		}
		
		if(bot.getAvailableMinerals() >= 200 && bot.getAvailableGas() >= 100
				&& UnitTracker.count(UnitType.Terran_Barracks) >= 1 && UnitTracker.count(UnitType.Terran_Factory) < Math.min(5, 2*UnitTracker.count(UnitType.Terran_Command_Center)-1))
		{
			bot.spaceManager.build(UnitType.Terran_Factory);
		}

		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Mech tier 1 time: " + time);
		
		if (bot.getAvailableMinerals() >= 100 && UnitTracker.count(UnitType.Terran_Barracks) >= 1)
		{
			for(DefensiveStructures structures : bot.defensiveStructures)
			{
				if (limitedBunkers)
				{
					if (!BWTAProxy.initialized)
						continue;
				
					// If we only use limited bunkers, we do not build bunkers for our main.
					if (Tyr.tileToPosition(Tyr.self.getStartLocation()).getDistance(structures.defendedPosition) < 200)
						continue;
				}
				
				int count = structures.getUnitCount(UnitType.Terran_Bunker);
				
				if(count < ((game.enemy().getRace() == Race.Protoss && !limitedBunkers)?2:1))
				{
					bot.spaceManager.buildDefensive(UnitType.Terran_Bunker, structures);
		  			break;
				}
			}
		}

		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Mech bunker time: " + time);
		
		if (bot.getAvailableMinerals() >= 150 
				&& UnitTracker.count(UnitType.Terran_Barracks) != 0 && UnitTracker.count(UnitType.Terran_Refinery) != 0
				&& UnitTracker.count(UnitType.Terran_Academy) == 0
				&& bot.homeGroup.units.size() >= 10)
		{
			bot.spaceManager.build(UnitType.Terran_Academy);
		}
		
		if (UnitTracker.count(UnitType.Terran_Siege_Tank_Siege_Mode) + UnitTracker.count(UnitType.Terran_Siege_Tank_Tank_Mode) >= 4
				&& UnitTracker.count(UnitType.Terran_Armory) < 2
				&& bot.getAvailableGas() >= 50
				&& bot.getAvailableMinerals() >= 100)
		{
			bot.spaceManager.build(UnitType.Terran_Armory);
		}
		
		if(bot.getAvailableMinerals()>= 125 && UnitTracker.count(UnitType.Terran_Engineering_Bay) < 1 && UnitTracker.count(UnitType.Terran_Vulture) >= 2) 
		{
			bot.spaceManager.build(UnitType.Terran_Engineering_Bay);
		}
		
		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Mech tier 2 time: " + time);

		if(UnitTracker.count(UnitType.Terran_Engineering_Bay) >= 1 && bot.getAvailableMinerals() >= 75)
		{
			for(DefensiveStructures structures : bot.defensiveStructures)
			{
				int count = structures.getUnitCount(UnitType.Terran_Missile_Turret);
				
				if(count == 0 && structures.defences.size() > 0)
				{
					bot.spaceManager.buildDefensive(UnitType.Terran_Missile_Turret, structures);
		  			break;
				}
			}
		}

		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Mech turret time: " + time);
	}

	@Override
	public boolean overrideStructureOrder(Game game, Player self, Tyr bot, Agent agent)
	{
		if(super.overrideStructureOrder(game, self, bot, agent))
			return true;
		
		if (agent.unit.getType() == UnitType.Terran_Factory)
		{
			if (agent.unit.getAddon() != null)
			{
				if (UnitTracker.count(UnitType.Terran_Factory) <= 1 
						&& UnitTracker.count(UnitType.Terran_Siege_Tank_Siege_Mode) + UnitTracker.count(UnitType.Terran_Siege_Tank_Tank_Mode) >= 2)
					return false;
				return !self.isResearching(TechType.Tank_Siege_Mode) && !self.hasResearched(TechType.Tank_Siege_Mode);
			}

			boolean addonWanted = UnitTracker.usableAddOns(UnitType.Terran_Machine_Shop)*3 + 1 < self.completedUnitCount(UnitType.Terran_Factory)
					|| self.completedUnitCount(UnitType.Terran_Factory) <= 1;
			
			if(UnitTracker.count(UnitType.Terran_Armory) >= 1 && !agent.unit.isTraining()
					&& UnitTracker.count(UnitType.Terran_Siege_Tank_Siege_Mode) + UnitTracker.count(UnitType.Terran_Siege_Tank_Tank_Mode) 
						>= 5 + (UnitTracker.count(UnitType.Terran_Goliath)/2) 
					&& bot.getAvailableMinerals() >= 100 && bot.getAvailableGas() >= 50)
			{
				agent.unit.train(UnitType.Terran_Goliath);
				return true;
			}
			else if (bot.getAvailableMinerals() >= 75 && !agent.unit.isTraining() && 
					UnitTracker.count(UnitType.Terran_Vulture) <= 2*(UnitTracker.count(UnitType.Terran_Siege_Tank_Siege_Mode) + UnitTracker.count(UnitType.Terran_Siege_Tank_Tank_Mode)) - 1)
			{
				agent.unit.train(UnitType.Terran_Vulture);
				return true;
			}
			
			return !addonWanted;
		}
		else if(agent.unit.getType() == UnitType.Terran_Armory)
		{
			if(bot.getAvailableMinerals() >= UpgradeType.Terran_Vehicle_Weapons.mineralPrice()
					&& bot.getAvailableGas() >= UpgradeType.Terran_Vehicle_Weapons.gasPrice())
				agent.unit.upgrade(UpgradeType.Terran_Vehicle_Weapons);
			
			if(bot.getAvailableMinerals() >= UpgradeType.Terran_Vehicle_Plating.mineralPrice()
					&& bot.getAvailableGas() >= UpgradeType.Terran_Vehicle_Plating.gasPrice())
				agent.unit.upgrade(UpgradeType.Terran_Vehicle_Plating);
			
			return true;
		}
		else if (agent.unit.getType() == UnitType.Terran_Engineering_Bay)
		{
			return true;
		}
		else if (agent.unit.getType() == UnitType.Terran_Machine_Shop)
		{
			if (!agent.unit.isResearching() && self.hasResearched(TechType.Tank_Siege_Mode)
					&& bot.getAvailableMinerals() >= 100
					&& bot.getAvailableGas() >= 100)
				agent.unit.research(TechType.Spider_Mines);
		}
		return false;
	}
}
