/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.builds;

import bwapi.Game;
import bwapi.Player;
import bwapi.Position;
import bwapi.Unit;
import bwapi.UnitType;
import bwapi.UpgradeType;
import bwta.BWTA;

import com.tyr.DebugMessages;
import com.tyr.EnemyManager;
import com.tyr.Settings;
import com.tyr.Tyr;
import com.tyr.UnitTracker;
import com.tyr.agents.Agent;
import com.tyr.agents.Attack;
import com.tyr.buildingplacement.SpaceManager;
import com.tyr.requirements.ConjRequirement;
import com.tyr.requirements.CostRequirement;
import com.tyr.requirements.UnitRequirement;
import com.tyr.tasks.BuildAtLocationSolution;
import com.tyr.tasks.BuildAtLocationTask;
import com.tyr.tasks.ClearExpandsTask;
import com.tyr.tasks.ConstantPushSolution;
import com.tyr.tasks.ConstantPushTask;
import com.tyr.tasks.LocalDefenseTask;

public class DTHarass extends CompositeBuildOrder
{
	private ConstantPushTask constantPushTask;
	private LocalDefenseTask localDefenseTask;
	private LocalDefenseTask thirdDefenseTask;
	private ClearExpandsTask dtHarassTask;
	public boolean attackStarted;
	
	private int armySize = 25;
	private int retreatSize = 20;
	
	private boolean detectionNeeded = false;

	private boolean localDefense = false;
	
	private boolean counterZealotRush = false;
	
	private boolean zealotEncountered = false;
	private boolean counterDragoonRush = false;
	
	private boolean defensesInitialized;
	
	private boolean buildingCannons = false;
	
	@Override
	public void initialize(Game game, Player self, Tyr bot)
	{
		Settings.setRequiredSize(100);
		Settings.setMaximumSize(100);
		Settings.setMaximumWorkers(60);
		Settings.setSmallInvasionDist(480);
		Settings.setLargeInvasionDist(768);
		Settings.setWorkersPerGas(2);
		
		Attack.dontWaitAtCannon = true;
		
		this.add(new ExpandPart(true));
		if (game.mapFileName().contains("Alchemist"))
			this.add(new WorkerScoutPart(100));
		else
			this.add(new WorkerScoutPart(1600));
		ExpandPart.maximumCcs = 2;
		
		localDefenseTask = new LocalDefenseTask(BWTA.getRegion(self.getStartLocation()), 0);
		bot.taskManager.potentialTasks.add(localDefenseTask);
		
		thirdDefenseTask = new LocalDefenseTask(null, 0);
		bot.taskManager.potentialTasks.add(thirdDefenseTask);
		constantPushTask = new ConstantPushTask(null, ConstantPushSolution.ATTACK);
		constantPushTask.stop = true;
		bot.taskManager.potentialTasks.add(constantPushTask);
		
		ConstantPushTask.prioritizeExpands = true;
		
		dtHarassTask = new ClearExpandsTask(true);
		dtHarassTask.setActive(true);
		bot.taskManager.potentialTasks.add(dtHarassTask);
		ClearExpandsTask.allowFullAssault = false;
		
		super.initialize(game, self, bot);
	}
	
	@Override
	public void onFrame(Game game, Player self, Tyr bot) 
	{
		int max = 0;
		for (Unit unit : self.getUnits())
		{
			game.drawTextMap(unit.getPosition(), unit.getID() + "");
			max = Math.max(max, unit.getID());
		}
		DebugMessages.addMessage("Max ID: " + max);
		
		DebugMessages.addMessage("Performing PvT build.");
		super.onFrame(game, self, bot);
		
		if (gas() >= 400)
			Settings.setWorkersPerGas(2);
		else if ((gas() < 250 && count(GATEWAY) >= 12) 
				|| gas() + 400 < minerals()
				|| count(CITADEL) > 0)
			Settings.setWorkersPerGas(3);

		if (game.getFrameCount() > 8000)
			counterZealotRush = false;
		else if (EnemyManager.getManager().getAllCount(ZEALOT) >= 4)
			counterZealotRush = true;
		
		if (EnemyManager.getManager().getAllCount(DARK_TEMPLAR) 
				+ EnemyManager.getManager().getAllCount(CITADEL)
				+ EnemyManager.getManager().getAllCount(TEMPLAR_ARCHIVES) > 0)
			detectionNeeded = true;
		
		if (game.getFrameCount() >= 7500 
				&& EnemyManager.getManager().getAllCount(DARK_TEMPLAR)
					+ EnemyManager.getManager().getAllCount(TEMPLAR_ARCHIVES) > 0)
		{
			armySize = 15;
			retreatSize = 10;
			buildingCannons = true;
		}
		
		if (completed(CANNON) >= 2 && completed(OBSERVER) > 0)
		{
			armySize = 25;
			retreatSize = 20;
		}
		
		if (EnemyManager.getManager().getAllCount(ZEALOT) > 0)
			zealotEncountered = true;
		counterDragoonRush = !zealotEncountered 
				&& EnemyManager.getManager().getAllCount(DRAGOON) 
				+ EnemyManager.getManager().getAllCount(CYBERNETICS_CORE) > 0;
		
		
		if (constantPushTask != null && completed(ZEALOT) + completed(DRAGOON) <= retreatSize)
			constantPushTask.stop = true;
		if (constantPushTask != null && completed(ZEALOT) + completed(DRAGOON) >= armySize)
		{
			ExpandPart.maximumCcs = 100;
			constantPushTask.stop = false;
			attackStarted = true;
			if (localDefense)
				localDefenseTask.setDefendersNeeded(3);
			if (EnemyManager.getManager().getAllCount(DARK_TEMPLAR)
					+ EnemyManager.getManager().getAllCount(TEMPLAR_ARCHIVES) == 0)
				armySize = Math.max(armySize, 30);
			detectionNeeded = true;
		}
		
		if (!defensesInitialized && buildingCannons && !Tyr.game.mapFileName().contains("Alchemist"))
		{
			defensesInitialized = true;

			final Position startLocation = Tyr.tileToPosition(Tyr.self.getStartLocation());
			final Position mainExit = SpaceManager.getMainExit();
			final Position defensivePosition = new Position((startLocation.getX() + mainExit.getX()*3)/4, (startLocation.getY() + mainExit.getY()*3)/4);
			//final Position defensivePosition = SpaceManager.getNaturalDefensePos();
					
			final BuildAtLocationTask buildTask = new BuildAtLocationTask(defensivePosition, true);
			BuildAtLocationSolution.buildingDistance = 300;
			
			buildTask.addBuilding(UnitType.Protoss_Pylon, new UnitRequirement(UnitType.Protoss_Probe, 7));

			for (int i=0; i<2; i++)
			{
				final ConjRequirement requirements = new ConjRequirement()
						.addRequirement(new UnitRequirement(UnitType.Protoss_Forge, 1, true))
						.addRequirement(new CostRequirement(50, 0));
				buildTask.addBuilding(UnitType.Protoss_Photon_Cannon, requirements);
			}
			
			bot.taskManager.potentialTasks.add(buildTask);
		}
		
		if (localDefense && thirdDefenseTask.getDefendersNeeded() == 0 && Tyr.bot.workForce.mineralWorkers.size() >= 3)
		{
			thirdDefenseTask.setDefendersNeeded(3);
			thirdDefenseTask.setDefendedPosition(Tyr.bot.workForce.mineralWorkers.get(2).resourceDepot.getPosition());
		}
		
		//if we're running out of supply and have enough minerals ...
		if ((self.supplyTotal() + UnitTracker.getSupplyConstructing() - self.supplyUsed() 
					<= count(GATEWAY)*3 + UnitTracker.getCcCount() * 3)
				&& (minerals() >= 100)
				&& self.supplyTotal() + UnitTracker.getSupplyConstructing() < 400)
		{
			bot.spaceManager.build(PYLON);
		}
		
		//if we've the resources to build a Gateway ...
		if ((count(NEXUS) >= 2)
				&& minerals() >= 150
				&& count(GATEWAY) == 0
				&& completed(PYLON) > 0
				&& count(PYLON) > 0)
		{
			bot.spaceManager.build(GATEWAY);
		}

		if ((count(NEXUS) >= 2)
				&& minerals() >= 150
				&& count(GATEWAY) < 2
				&& count(CYBERNETICS_CORE) > 0)
		{
			bot.spaceManager.build(GATEWAY);
		}
		
		if (minerals() >= 150
				&& count(GATEWAY) < 2)
		{
			bot.spaceManager.build(GATEWAY);
		}
		
		if (!pauseProduction())
		{
			if ((count(NEXUS) >= 2)
					&& minerals() >= 150
					&& count(GATEWAY) < 5
					&& (count(CYBERNETICS_CORE) > 0))
			{
				bot.spaceManager.build(GATEWAY);
			}
			
			if (count(NEXUS) >= 2
					&& minerals() >= 250
					&& (count(CYBERNETICS_CORE) > 0)
					&& count(GATEWAY) < 8)
			{
				bot.spaceManager.build(GATEWAY);
			}
			
			if (count(NEXUS) >= 2
					&& minerals() >= 150
					&& completed(GATEWAY) >= 8
					&& count(GATEWAY) < 12)
			{
				bot.spaceManager.build(GATEWAY);
			}
		}
		
		if(minerals() >= 100 
				&& UnitTracker.getGeyserCount() > 0
				&& count(GATEWAY) > 0
				&& (count(NEXUS) >= 2)
				&& (!counterZealotRush || count(ZEALOT) >= 6))
		{
			bot.spaceManager.build(ASSIMILATOR);
		}
		
		if (minerals() >= 200 
				&& count(CYBERNETICS_CORE) < 1
				&& completed(GATEWAY) > 0
				&& (count(NEXUS) >= 2)
				&& (!counterZealotRush || count(ZEALOT) >= 6)
				&& (!buildingCannons || count(CANNON) >= 2))
		{
			bot.spaceManager.build(CYBERNETICS_CORE);
		}
		
		if (minerals() >= 150
				&& count(FORGE) < 1
				&& (count(DRAGOON) >= 2 || buildingCannons)
				&& (!counterZealotRush || count(ZEALOT) >= 6 || buildingCannons)
				&& (!counterDragoonRush || count(DRAGOON) > 15 || buildingCannons)
				)
		{
			bot.spaceManager.build(FORGE);
		}
		
		if (minerals() >= 150
				&& gas() >= 100
				&& count(CITADEL) == 0
				&& completed(CYBERNETICS_CORE) > 0
				&& count(NEXUS) >= 2
				&& (!counterDragoonRush || count(DRAGOON) > 15)
				&& (!buildingCannons || count(CANNON) >= 2))
		{
			bot.spaceManager.build(CITADEL);
		}
		
		if (minerals() >= 150
				&& gas() >= 200 
				&& count(TEMPLAR_ARCHIVES) < 1
				&& completed(CITADEL) > 0
				&& count(NEXUS) >= 2
				&& (!buildingCannons || count(CANNON) >= 2))
		{
			build(TEMPLAR_ARCHIVES);
		}
		
		if (gas() >= 200 && minerals() >= 200
				&& detectionNeeded
				&& count(ROBOTICS_FACILITY) == 0
				&& (!buildingCannons || count(CANNON) >= 2))
		{
			bot.spaceManager.build(ROBOTICS_FACILITY);
		}
		
		if (gas() >= 50 && minerals() >= 100
				&& completed(ROBOTICS_FACILITY) > 0
				&& count(OBSERVATORY) == 0
				&& detectionNeeded
				&& (!buildingCannons || count(CANNON) >= 2))
		{
			bot.spaceManager.build(OBSERVATORY);
		}
	}
	
	private boolean pauseProduction()
	{
		if (gas() > 250 && minerals() > 400)
			return false;
		if (counterZealotRush && count(ZEALOT) < 6)
			return false;

		if (buildingCannons && count(FORGE) == 0)
			return true;
		if (buildingCannons && completed(FORGE) > 0
				&& count(CANNON) < 2 
				&& minerals() < 200)
			return true;
		if (count(DRAGOON) + count(ZEALOT) >= 10 && count(CITADEL) == 0
				&& (!counterDragoonRush || count(DRAGOON) > 15))
			return true;
		if (completed(CITADEL) > 0
				&& count(TEMPLAR_ARCHIVES) == 0
				&& (!counterDragoonRush || count(DRAGOON) > 15))
			return true;
		if (detectionNeeded && count(DRAGOON) >= 5
				&& completed(ROBOTICS_FACILITY) == 1
				&& count(OBSERVATORY) == 0)
			return true;
		if (detectionNeeded && count(DRAGOON) >= 5
				&& completed(OBSERVATORY) == 1
				&& count(OBSERVER) == 0)
			return true;
		if (detectionNeeded
				&& count(ROBOTICS_FACILITY) == 0)
			return true;
		if (attackStarted && count(NEXUS) < 3)
			return true;
		return false;
	}
	
	@Override
	public boolean overrideStructureOrder(Game game, Player self, Tyr bot, Agent agent)
	{
		if (agent.unit.getType() == GATEWAY && !agent.unit.isTraining())
		{
			boolean pause = pauseProduction();
			if (count(ZEALOT) < (counterZealotRush ? 6 : 4)
					&& minerals() >= 100
					&& !counterDragoonRush)
				agent.unit.build(ZEALOT);
			
			if (pause && minerals() >= 600)
				agent.unit.build(ZEALOT);
			else if (self.completedUnitCount(UnitType.Protoss_Templar_Archives) > 0
					&& count(DARK_TEMPLAR) < 3
					&& minerals() >= 125
					&& gas() >= 100)
				agent.unit.train(UnitType.Protoss_Dark_Templar);
			else if (completed(TEMPLAR_ARCHIVES) > 0 && count(DARK_TEMPLAR) == 0
					&& (!counterDragoonRush || count(DRAGOON) > 15))
				return true;
			else if (count(ZEALOT) < count(DRAGOON) - 5
					&& minerals() >= 100
					&& (minerals() >= 400 || !pause)
					&& (!counterDragoonRush || count(DRAGOON) > 15)
					)
				agent.unit.build(ZEALOT);
			else if (minerals() >= 125 && gas() >= 50
					&& ((minerals() >= 400 && gas() > 200) || !pause))
				agent.unit.build(DRAGOON);
			return true;
		}
		else if (agent.unit.getType() == FORGE && !agent.unit.isUpgrading())
		{
			if(pauseProduction())
				return true;

			if (buildingCannons && count(CANNON) < 2)
				return true;
			
			if(minerals() >= UpgradeType.Protoss_Ground_Weapons.mineralPrice()
					&& gas() >= UpgradeType.Protoss_Ground_Weapons.gasPrice())
				agent.unit.upgrade(UpgradeType.Protoss_Ground_Weapons);
			
			if(minerals() >= UpgradeType.Protoss_Ground_Armor.mineralPrice()
					&& gas() >= UpgradeType.Protoss_Ground_Armor.gasPrice())
				agent.unit.upgrade(UpgradeType.Protoss_Ground_Armor);
		}
		else if (agent.unit.getType() == CYBERNETICS_CORE && !agent.unit.isUpgrading())
		{
			if(minerals() >= UpgradeType.Singularity_Charge.mineralPrice()
					&& gas() >= UpgradeType.Singularity_Charge.gasPrice()
					&& count(DRAGOON) > 5)
				agent.unit.upgrade(UpgradeType.Singularity_Charge);
		}
		else if (agent.unit.getType() == ROBOTICS_FACILITY)
		{
			if ((count(OBSERVER) < 2)
					&& completed(OBSERVATORY) > 0
					&& minerals() >= 25
					&& gas() >= 75
					&& !agent.unit.isTraining())
				agent.unit.build(OBSERVER);
			return true;
		}
		else if (agent.unit.getType() == CITADEL && !agent.unit.isUpgrading())
		{
			if (gas() >= UpgradeType.Leg_Enhancements.gasPrice()
					&& minerals() >= UpgradeType.Leg_Enhancements.mineralPrice()
					&& count(TEMPLAR_ARCHIVES) > 0
					&& (completed(TEMPLAR_ARCHIVES) == 0 || count(DARK_TEMPLAR) > 0))
				agent.unit.upgrade(UpgradeType.Leg_Enhancements);
		}
		else if (agent.unit.getType() == TEMPLAR_ARCHIVES && !agent.unit.isResearching())
		{
			return true;
		}
		else if (agent.unit.getType() == STARGATE)
		{
			if (completed(ARBITER_TRIBUNAL) > 0
					&& minerals() >= 100
					&& gas() >= 350
					&& !agent.unit.isTraining())
				agent.unit.train(ARBITER);
			return true;
		}
		return false;
	}
}
