/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.builds;

import com.tyr.Settings;
import com.tyr.Tyr;
import com.tyr.UnitTracker;
import com.tyr.agents.Agent;
import com.tyr.agents.Attack;
import com.tyr.buildingplacement.SpaceManager;
import com.tyr.requirements.ConjRequirement;
import com.tyr.requirements.CostRequirement;
import com.tyr.requirements.UnitRequirement;
import com.tyr.tasks.BuildAtLocationSolution;
import com.tyr.tasks.BuildAtLocationTask;
import com.tyr.tasks.ConstantPushSolution;
import com.tyr.tasks.ConstantPushTask;
import com.tyr.tasks.DefendMainTask;
import com.tyr.unitgroups.DefendingWorkers;

import bwapi.Game;
import bwapi.Player;
import bwapi.Position;
import bwapi.UnitType;
import bwapi.UpgradeType;


public class CarrierRush extends CompositeBuildOrder
{
	private ConstantPushTask constantPushTask;
	
	private boolean attackStarted = false;
	private boolean makeDragoons = false;
	
	int requiredDragoons = 0;

	private int armySize = 9;
	private int retreatSize = 4;
	
	boolean expand = true;
	boolean cannonDefense = false;
	
	boolean defensesInitialized = false;
	private int cannonCount = 10;
	
	public CarrierRush() {}
	
	public CarrierRush(boolean makeDragoons, boolean cannonDefense)
	{
		this.makeDragoons = makeDragoons;
		if (makeDragoons)
			requiredDragoons = 20;
		this.cannonDefense = cannonDefense;
	}
	
	@Override
	public void initialize(Game game, Player self, Tyr bot)
	{
		Settings.setRequiredSize(100);
		Settings.setMaximumSize(100);
		Settings.setMaximumWorkers(40);
		Settings.setDefendExpandDist(2000);
		Settings.setSmallInvasionDist(768);
		Settings.setLargeInvasionDist(1024);
		Settings.setWorkersPerGas(3);
		Attack.dontWaitAtCannon = true;
		DefendMainTask.maxFlyerDefense = true;
		
		this.add(new ExpandPart(true));
		ExpandPart.maximumCcs = 1;
		this.add(new WorkerScoutPart(1600));
		
		DefendingWorkers.stop = true;
		
		super.initialize(game, self, bot);
	}
	
	private Position getDefensivePosition()
	{
		final Position location = SpaceManager.getNatural();
		return SpaceManager.getDefensePos(location);
	}
	
	@Override
	public void onFrame(Game game, Player self, Tyr bot) 
	{
		super.onFrame(game, self, bot);
		
		if (expand)
			ExpandPart.maximumCcs = 2;
		
		if (!attackStarted && completed(CARRIER) >= armySize)
		{
			attackStarted = true;
			Settings.setMaximumWorkers(60);
			
			constantPushTask = new ConstantPushTask(CARRIER, ConstantPushSolution.ATTACK);
			bot.taskManager.potentialTasks.add(constantPushTask);
		}
		
		if (minerals() >= 1000 && !cannonDefense)
			ExpandPart.maximumCcs = 3;
		
		if (constantPushTask != null && completed(CARRIER) <= retreatSize)
			constantPushTask.stop = true;
		if (constantPushTask != null && completed(CARRIER) >= armySize)
			constantPushTask.stop = false;

		if (!defensesInitialized && game.getFrameCount() >= 100)
		{
			defensesInitialized = true;
			

			
			Position defensivePosition = getDefensivePosition();
				
			BuildAtLocationTask buildTask = new BuildAtLocationTask(defensivePosition, true);
			buildTask.addBuilding(UnitType.Protoss_Pylon, new UnitRequirement(UnitType.Protoss_Probe, 7));
			buildTask.addBuilding(UnitType.Protoss_Forge, new ConjRequirement()
				.addRequirement(new UnitRequirement(UnitType.Protoss_Pylon, 1, true))
				.addRequirement(new CostRequirement(100, 0)));

			BuildAtLocationSolution.buildingDistance = 300;
			
			for (int i=0; i<cannonCount; i++)
				buildTask.addBuilding(UnitType.Protoss_Photon_Cannon, new ConjRequirement()
					.addRequirement(new UnitRequirement(UnitType.Protoss_Forge, 1, true))
					.addRequirement(new CostRequirement(150, 0)));

			buildTask.addBuilding(UnitType.Protoss_Pylon, new ConjRequirement()
				.addRequirement(new CostRequirement(800, 0)));
			
			for (int i=0; i<cannonCount; i++)
				buildTask.addBuilding(UnitType.Protoss_Photon_Cannon, new ConjRequirement()
				.addRequirement(new UnitRequirement(UnitType.Protoss_Forge, 1, true))
				.addRequirement(new UnitRequirement(NEXUS, 2, true))
				.addRequirement(new CostRequirement(800, 0)));
			bot.taskManager.potentialTasks.add(buildTask);
		}
		
		//if we're running out of supply and have enough minerals ...
		if ((self.supplyTotal() + UnitTracker.getSupplyConstructing() - self.supplyUsed() 
					<= UnitTracker.count(UnitType.Protoss_Gateway)*3 + UnitTracker.getCcCount() * 3
					+ count(STARGATE) * 16)
				&& (bot.getAvailableMinerals() >= 100)
				&& self.supplyTotal() + UnitTracker.getSupplyConstructing() < 400)
		{
			bot.spaceManager.build(UnitType.Protoss_Pylon);
		}
		
		if (SpaceManager.noPower
				&& minerals() >= 100
				&& count(PYLON) == completed(PYLON))
			bot.spaceManager.build(PYLON);
		
		//if we've the resources to build a Gateway ...
		if (bot.getAvailableMinerals() >= 150
				&& UnitTracker.count(UnitType.Protoss_Gateway) == 0
				&& self.completedUnitCount(UnitType.Protoss_Pylon) > 0
				&& (count(NEXUS) > 1 || minerals() >= 400 || !expand))
		{
			bot.spaceManager.build(UnitType.Protoss_Gateway);
		}
		
		if (makeDragoons
				&& count(GATEWAY) < 3
				&& completed(CYBERNETICS_CORE) > 0
				&& minerals() >= 150)
		{
			bot.spaceManager.build(GATEWAY);
		}
		
		if(bot.getAvailableMinerals() >= 100 
				&& UnitTracker.getGeyserCount() > 0
				&& UnitTracker.count(UnitType.Protoss_Gateway) > 0
				&& (count(NEXUS) > 1 || minerals() >= 400 || !expand))
		{
			bot.spaceManager.build(UnitType.Protoss_Assimilator);
		}
		
		if (bot.getAvailableMinerals() >= 200 
				&& UnitTracker.count(UnitType.Protoss_Cybernetics_Core) == 0
				&& self.completedUnitCount(UnitType.Protoss_Gateway) > 0
				&& (count(NEXUS) > 1 || minerals() >= 400 || !expand))
		{
			bot.spaceManager.build(UnitType.Protoss_Cybernetics_Core);
		}
		
		if (bot.getAvailableMinerals() >= 150
				&& bot.getAvailableGas() >= 150
				&& UnitTracker.count(UnitType.Protoss_Cybernetics_Core) > 0
				&& UnitTracker.count(UnitType.Protoss_Stargate) < 2
				&& (count(NEXUS) > 1 || minerals() >= 400 || !expand))
		{
			bot.spaceManager.build(UnitType.Protoss_Stargate);
		}
		
		if (bot.getAvailableMinerals() >= 150
				&& bot.getAvailableGas() >= 400
				&& UnitTracker.count(UnitType.Protoss_Cybernetics_Core) > 0
				&& UnitTracker.count(UnitType.Protoss_Stargate) == 2)
		{
			bot.spaceManager.build(UnitType.Protoss_Stargate);
		}
		
		/*
		if (bot.getAvailableGas() >= 200 && bot.getAvailableMinerals() >= 200
				&& (UnitTracker.count(UnitType.Protoss_Cybernetics_Core) > 0 && !dragoonOnly)
				&& UnitTracker.count(UnitType.Protoss_Gateway) >= (fastReaver?4:5)
				&& (UnitTracker.count(UnitType.Protoss_Scout) >= 1 || fastReaver)
				&& UnitTracker.count(UnitType.Protoss_Robotics_Facility) == 0
				&& UnitTracker.count(UnitType.Protoss_Dragoon) >= 5)
		{
			bot.spaceManager.build(UnitType.Protoss_Robotics_Facility);
		}
		
		if (bot.getAvailableGas() >= 50 && bot.getAvailableMinerals() >= 100
				&& self.completedUnitCount(UnitType.Protoss_Robotics_Facility) > 0
				&& UnitTracker.count(UnitType.Protoss_Observatory) == 0)
		{
			bot.spaceManager.build(UnitType.Protoss_Observatory);
		}
		*/
		
		if (count(FLEET_BEACON) == 0
				&& gas() >= 200
				&& minerals() >= 300
				&& completed(STARGATE) > 0)
		{
			bot.spaceManager.build(FLEET_BEACON);
		}
	}
	
	@Override
	public boolean overrideStructureOrder(Game game, Player self, Tyr bot, Agent agent)
	{
		if (agent.unit.getType() == UnitType.Protoss_Gateway && !agent.unit.isTraining())
		{
			if (count(DRAGOON) < requiredDragoons
					&& completed(CYBERNETICS_CORE) > 0
					&& minerals() >= 125
					&& gas() >= 50)
			{
				agent.unit.train(DRAGOON);
			}
			return true;
		}
		else if (agent.unit.getType() == UnitType.Protoss_Forge && !agent.unit.isUpgrading())
		{
			return true;
		}
		else if (agent.unit.getType() == UnitType.Protoss_Cybernetics_Core && !agent.unit.isUpgrading())
		{
			return true;
		}
		else if (agent.unit.getType() == UnitType.Protoss_Robotics_Facility)
		{
			if ((UnitTracker.count(UnitType.Protoss_Observer) < 2)
					&& bot.getAvailableMinerals() >= 25
					&& bot.getAvailableGas() >= 75
					&& !agent.unit.isTraining())
				agent.unit.build(UnitType.Protoss_Observer);
			return true;
		}
		else if (agent.unit.getType() == UnitType.Protoss_Stargate)
		{
			if (bot.getAvailableMinerals() >= 350 && bot.getAvailableGas() >= 250
					 && !agent.unit.isTraining()
					 && completed(FLEET_BEACON) > 0)
				agent.unit.build(CARRIER);
			return true;
		}
		else if (agent.unit.getType() == UnitType.Protoss_Citadel_of_Adun && !agent.unit.isUpgrading())
		{
			return true;
		}
		else if (agent.unit.getType() == FLEET_BEACON)
		{
			if (self.getUpgradeLevel(UpgradeType.Carrier_Capacity) == 0
					&& gas() >= 100
					&& minerals() >= 100)
			{
				agent.unit.upgrade(UpgradeType.Carrier_Capacity);
			}
		}
		return false;
	}
}
