/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.tasks;

import com.tyr.DebugMessages;
import com.tyr.EnemyManager;
import com.tyr.Tyr;
import com.tyr.agents.Agent;

import bwapi.Game;
import bwapi.Player;
import bwapi.UnitType;


/**
 * A task for harassing the opponents expands with vultures.
 */
public class VultureHarassTask extends Task
{
	/**
	 * The number of vultures with which we want to harass.
	 */
	public static int desiredVultures = 4;
	
	/**
	 * How many units must there be in the home group before we send out a harass?
	 */
	public static int defendingUnitsNeeded = 10;
	
	/**
	 * The direction in which we perform our first attack, either -1 or 1.
	 */
	private int startDir = 1;
	
	/**
	 * Do we alternate the direction in which we start the attack?
	 */
	private boolean alternateDir = false;
	
	/**
	 * A task for harassing the opponents expands with vultures.
	 */
	public VultureHarassTask() { }
	
	/**
	 * A task for harassing the opponents expands with vultures.
	 * @param alternateDir Do we alternate the direction in which we start the attack?
	 */
	public VultureHarassTask(boolean alternateDir) 
	{
		this.alternateDir = alternateDir;
	}
	
	/**
	 * A task for harassing the opponents expands with vultures.
	 * @param startDir The direction in which we perform our first attack, either -1 or 1.
	 */
	public VultureHarassTask(int startDir) 
	{
		this.startDir = startDir;
	}
	
	@Override
	public boolean isRequired(Game game, Player self, Tyr bot) 
	{
		if (solution != null)
		{
			if (((VultureHarassSolution)solution).vultureCount() == 0)
				solution = null;
			else return true;
		}

		// We want there to be enough units left to defend.
		if (bot.homeGroup.units.size() < defendingUnitsNeeded)
			return false;
			
		// We cannot start the harass if we cannot determine a target for the harass.
		if (EnemyManager.getManager().getOrderedExpands() == null)
			return false;
		
		// Count how many vultures we can use.
		int vultureCount = 0;
		
		for(Agent agent : bot.homeGroup.units)
		{
			if (agent.unit.getType() == UnitType.Terran_Vulture)
			{
				vultureCount++;
				if (vultureCount >= desiredVultures)
					break;
			}
		}
		
		return vultureCount >= desiredVultures;
	}
	
	@Override
	public void solve(Game game, Player self, Tyr bot)
	{
		// We call the super class, which will send the vultures out to harass.
		super.solve(game, self, bot);
	}

	@Override
	public void findSolution(Game game, Player self, Tyr bot)
	{
		solution = new VultureHarassSolution(this, startDir);
		if (alternateDir)
			startDir = -startDir;
		
		// The number of vultures we have added.
		int vultureCount = 0;
		
		// Get the vultures needed for the harass.
		for(int i = bot.homeGroup.units.size() -1; vultureCount < desiredVultures && i >= 0; i--)
		{
			Agent agent = bot.homeGroup.units.get(i);
			if (agent.unit.getType() == UnitType.Terran_Vulture)
			{
				((VultureHarassSolution)solution).add(agent);
				bot.homeGroup.units.remove(i);
				vultureCount++;
			}
		}
	}

	public boolean isHarassing()
	{
		if (solution != null)
			DebugMessages.addMessage("Harassing vultures: " + ((VultureHarassSolution)solution).vultureCount());
		return solution != null && ((VultureHarassSolution)solution).vultureCount() > 0;
	}
}
