/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.tasks;

import java.util.ArrayList;

import com.tyr.EnemyManager;
import com.tyr.EnemyPosition;
import com.tyr.Tyr;
import com.tyr.agents.Agent;

import bwapi.Color;
import bwapi.Game;
import bwapi.Player;
import bwapi.Position;
import bwta.BaseLocation;

/**
 * This class implements a solution for attacking the enemy player by going by every expand.
 */
public class SalesmenMarinesSolution extends SolutionItem 
{
	/**
	 * Wrapped solution which will manage the actual attack.
	 */
	private AttackRegroupSolution attackSolution;
	
	/**
	 * The position of the base we are currently attacking in the list of bases returned by EnemyManager.getOrderedexpands().
	 */
	private int currentPos;
	
	/**
	 * The direction in which we are going through the list of bases returned by EnemyManager.getOrderedexpands().
	 */
	private int currentDir = 1;
	
	/**
	 * The position of the base we currently intend to attack.
	 */
	private Position target;
	
	/**
	 * List of all expands, in the order that they are encountered in a sweep around the map.
	 * Includes opponents main.
	 */
	private ArrayList<Position> orderedExpands;
	
	/**
	 * This class implements a solution for attacking the enemy player by going by every expand.
	 * @param task The task that started this solution.
	 * @param direction The direction in which we will attack. Either 1 or -1.
	 */
	public SalesmenMarinesSolution(Task task, int direction) 
	{
		super(task);
		this.currentDir = direction;
		determineTarget();
		attackSolution = new AttackRegroupSolution(task, target);
	}
	
	@Override
	public void onFrame(Game game, Player self, Tyr bot)
	{
		// See if we need to change the target, and if so give the new target to the attack solution.
		if (determineTarget())
			attackSolution.setTarget(target);
		
		// Manage the attack solution.
		attackSolution.onFrame(game, self, bot);
	}

	/**
	 * Adds a unit to the group of attack units.
	 * @param agent The unit to be added.
	 */
	public void add(Agent agent)
	{
		attackSolution.add(agent);
	}
	
	/**
	 * Determines the base we want to attack.
	 * @return Whether the method call actually changes the target.
	 */
	private boolean determineTarget()
	{
		boolean result = false;

		initializeOrderedExpands();
		if (orderedExpands == null)
			return false;
		
		if (target != null)
			Tyr.drawCircle(target, Color.Red);
		
		// If the target has not previously been initialized, initalize it.
		if (target == null)
		{
			currentPos = EnemyManager.getManager().getSelfPos();
			if(currentPos != -1)
			{
				target = orderedExpands.get(currentPos);
				result = true;
			}
		}
		else
		{
		   	for(EnemyPosition p : EnemyManager.getManager().enemyBuildingMemory)
		   		if (p.pos.getX() == target.getX() && p.pos.getY() == target.getY())
		   			return false;
		}
		
		if (currentPos >= 0)
		{
			for(EnemyPosition p : EnemyManager.getManager().enemyBuildingMemory)
			{ 
				if (p.distanceSq(orderedExpands.get(currentPos)) <= 500*500)
				{
					target = p.pos;
					return true;
				}
			}
		}
		
		// Determine whether we are done attacking the current target.
		for(int i=0; i < 2*orderedExpands.size(); i++)
		{
			boolean isIsland = false;
			for (BaseLocation loc : Tyr.bot.expands)
			{
				if (!loc.isIsland())
					continue;
				if (loc.getDistance(target) <= 128)
					isIsland = true;
			}
			
			// If the target is invisible, we are not done.
			if (!isIsland && target != null && (!Tyr.game.isVisible(Tyr.positionToTile(target)) || i > 0))
				break;
			
			// We are done with the current target, get the next one.
			nextTarget();
			
			result = true;
		}
		
		return result;
	}
	
	/**
	 * Gets us the next target to harass.
	 */
	private void nextTarget()
	{
		initializeOrderedExpands();
		if (orderedExpands == null)
			return;
		
		currentPos = (currentPos + currentDir + orderedExpands.size())%orderedExpands.size();
		target = orderedExpands.get(currentPos);
	}
	
	/**
	 * Initializes the list of ordered expands.
	 */
	@SuppressWarnings("unchecked")
	private void initializeOrderedExpands()
	{
		if (orderedExpands == null)
		{
			orderedExpands = EnemyManager.getManager().getOrderedExpands();
			if (orderedExpands == null)
				return;
			orderedExpands = (ArrayList<Position>)orderedExpands.clone();
			if (Tyr.bot.suspectedEnemy.size() == 1)
				orderedExpands.add(Tyr.bot.suspectedEnemy.get(0).getPosition());
		}
	}

	@Override
	public boolean done() 
	{
		return attackSolution.done();
	}

	public int size() 
	{
		return attackSolution.size();
	}

	@Override
	public void clear() { }
}
