/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.tasks;

import java.util.ArrayList;
import java.util.List;

import com.tyr.BWTAProxy;
import com.tyr.EnemyManager;
import com.tyr.PositionUtil;
import com.tyr.Settings;
import com.tyr.Tyr;
import com.tyr.agents.Agent;
import bwapi.Color;
import bwapi.Game;
import bwapi.Player;
import bwapi.Position;
import bwapi.Unit;
import bwapi.UnitType;
import bwta.Region;

/**
 * This class implements a solution for defending a single base.
 */
public class LocalDefenseSolution extends Solution
{
	private List<Agent> units = new ArrayList<Agent>();
	
	private Unit target = null;
	
	private Region defendedRegion;
	
	private Position defendedPosition;
	
	/**
	 * This class implements a solution for defending a single base.
	 * @param task The task that started this solution.
	 * @param defendedRegion The region which we will defend.
	 */
	public LocalDefenseSolution(Task task, Region defendedRegion) 
	{
		super(task);
		this.defendedRegion = defendedRegion;
	}
	
	@Override
	public void onFrame(Game game, Player self, Tyr bot)
	{
		for (int i=units.size() - 1; i >= 0; i--)
			if (units.get(i) == null || units.get(i).isDead())
				units.remove(i);

		if (target != null &&
				(target.getHitPoints() <= 0
				|| target.getRemoveTimer() > 0 
				|| !target.exists()))
			target = null;
		
		updateTargetByRegion();
		updateTargetByPosition();
		
		if (target != null)
			Tyr.drawCircle(target.getPosition(), Color.Red, 16);
		
		for (Agent agent : units)
		{
			if (target != null)
				agent.attack(target);
			else if (defendedPosition != null)
				agent.unit.move(defendedPosition);
			else if (defendedRegion != null)
				agent.unit.move(defendedRegion.getCenter());
		}
	}
	
	private void updateTargetByRegion()
	{
		if (defendedRegion == null)
			return;
		
		if (target != null 
				&& (BWTAProxy.getRegion(target.getPosition()) != defendedRegion))
			target = null;
		
		if (target == null)
		{
			for (Unit unit : EnemyManager.getEnemyUnits())
			{
				if ((unit.getType() == UnitType.Protoss_Observer) || unit.getType() == UnitType.Protoss_Scarab)
						continue;
				
				if (PositionUtil.distanceSq(unit, defendedRegion.getCenter()) >= 1536 * 1536)
					continue;
				
				if (BWTAProxy.getRegion(unit.getPosition()) != defendedRegion)
					continue;
				
				target = unit;
				break;
			}
		}
	}
	
	private void updateTargetByPosition()
	{
		if (defendedPosition == null)
			return;
		
		int maxDistSq = Settings.getSmallInvasionDist() * Settings.getSmallInvasionDist();
		
		if (target != null &&
				(PositionUtil.distanceSq(target, defendedPosition) > maxDistSq)
			)
			target = null;
		
		if (target == null)
		{
			for (Unit unit : EnemyManager.getEnemyUnits())
			{
				if ((unit.getType() == UnitType.Protoss_Observer) || unit.getType() == UnitType.Protoss_Scarab)
						continue;
				
				if (PositionUtil.distanceSq(unit, defendedPosition) >= maxDistSq)
					continue;
				
				target = unit;
				break;
			}
		}
	}
	
	/**
	 * Add an agent to the list of defending units.
	 * @param unit The agent to be added.
	 */
	public void add(Agent agent)
	{
		units.add(agent);
	}

	/**
	 * Returns the number if agents sent out by this solution.
	 * @return Returns the number if agents sent out by this solution.
	 */
	public int size() 
	{
		return units.size();
	}
	
	public boolean done() 
	{
		return size() == 0;
	}

	public void setDefendedPosition(Position defendedPosition) 
	{
		this.defendedPosition = defendedPosition;
		this.defendedRegion = null;
	}
}
