/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.tasks;

import java.util.ArrayList;

import com.tyr.DebugMessages;
import com.tyr.EnemyManager;
import com.tyr.EnemyPosition;
import com.tyr.Tyr;
import com.tyr.agents.Agent;
import com.tyr.buildingplacement.SpaceManager;

import bwapi.Game;
import bwapi.Player;
import bwapi.Position;
import bwapi.UnitType;


/**
 * A task for dropping a large army in the opponents base.
 */
public class DoomDropTask extends Task
{
	/**
	 * List of the types of units we want for the drop.
	 */
	UnitType[][] requiredUnits = new UnitType[][]{
			new UnitType[]{
					UnitType.Terran_Goliath, 
					UnitType.Terran_Vulture, 
					UnitType.Terran_Vulture, 
					UnitType.Terran_Vulture},
					new UnitType[]{
					UnitType.Terran_Goliath, 
					UnitType.Terran_Vulture, 
					UnitType.Terran_Vulture, 
					UnitType.Terran_Vulture},
					new UnitType[]{
					UnitType.Terran_Siege_Tank_Siege_Mode,
					UnitType.Terran_Siege_Tank_Siege_Mode},
					new UnitType[]{
					UnitType.Terran_Siege_Tank_Siege_Mode,
					UnitType.Terran_Siege_Tank_Siege_Mode}
	};
	
	int startDir = 0;
	
	/**
	 * A task for harassing the opponents bases with a drop.
	 */
	public DoomDropTask() { }
	
	@Override
	public boolean isRequired(Game game, Player self, Tyr bot) 
	{
		if (solution != null)
			return true;
			
		// We cannot start the harass if we cannot determine a target for the harass.
		if (EnemyManager.getManager().getOrderedExpands() == null)
			return false;
		
		// Do we have enough units for the drop?
		return haveUnits(requiredUnits.length);
	}
	
	@Override
	public void solve(Game game, Player self, Tyr bot)
	{
		// We call the super class, which will send the units out to defend.
		super.solve(game, self, bot);
		
		DoomDropSolution doomDrop = (DoomDropSolution)solution;
		if (doomDrop.returned && haveUnits(requiredUnits.length - doomDrop.dropshipsLeft()))
		{
			if (doomDrop.attackersRemain())
			{
				boolean baseRemains = false;
				for (EnemyPosition pos : EnemyManager.getManager().enemyBuildingMemory)
				{
					if (pos.type.isResourceDepot())
					{
						baseRemains = true;
						break;
					}
				}
				if (!baseRemains)
					return;
			}
			
			doomDrop.clearDropships();
			assignUnits(doomDrop.getDropships(), doomDrop.getUnits());
			doomDrop.returned = false;
		}
	}

	/**
	 * Gets the dropships and attacking units required for the harass and adds them to dropships and units respectively. 
	 * @param dropships The list to which the dropships will be added.
	 * @param units The array of lists to which the attacking units will be added.
	 *        Assumed is one list per dropship.
	 *        Per list a number of units will be added that fits inside the dropship.
	 */
	private void assignUnits(ArrayList<Agent> dropships, ArrayList<Agent>[] units ) {
		boolean[][] found = new boolean[requiredUnits.length][];
		for (int listIt = 0; listIt < requiredUnits.length; listIt++)
			found[listIt] = new boolean[requiredUnits[listIt].length];
		
		// Get the units needed for the harass.
		for(int a = Tyr.bot.homeGroup.units.size() -1; a >= 0; a--)
		{
			Agent agent = Tyr.bot.homeGroup.units.get(a);
			if (agent.unit.getType() == UnitType.Terran_Dropship)
			{
				if (dropships.size() >= requiredUnits.length)
					continue;
				dropships.add(agent);
				Tyr.bot.homeGroup.units.remove(a);
				continue;
			}
			
			boolean success = false;
			for (int listIt=0; !success && listIt<requiredUnits.length; listIt++)
			{
				for (int typeIt = 0; !success && typeIt < requiredUnits[listIt].length; typeIt++)
				{
					if (!found[listIt][typeIt] && agent.unit.getType() == requiredUnits[listIt][typeIt])
					{
						found[listIt][typeIt] = true;
						units[listIt].add(agent);
						Tyr.bot.homeGroup.units.remove(a);
						success = true;
					}
				}
			}
		}
	}

	@Override
	public void findSolution(Game game, Player self, Tyr bot)
	{
		@SuppressWarnings("unchecked")
		ArrayList<Agent>[] units = new ArrayList[requiredUnits.length];
		for (int i=0; i<requiredUnits.length; i++)
			units[i] = new ArrayList<Agent>();
		
		ArrayList<Agent> dropships = new ArrayList<Agent>(); 
		
		assignUnits(dropships, units);

		determineDirection();
		solution = new DoomDropSolution(this, dropships, units, startDir);
	}

	private void determineDirection()
	{
		if (startDir != 0)
			return;

		ArrayList<Position> orderedExpands = EnemyManager.getManager().getOrderedExpands();
		
		if (orderedExpands == null)
			return;
		
		Position exit = SpaceManager.getMainExit();
		int down = (EnemyManager.getManager().getSelfPos() - 1 + orderedExpands.size())%orderedExpands.size();
		int up = (EnemyManager.getManager().getSelfPos() + 1)%orderedExpands.size();
		
		if (orderedExpands.get(down).getDistance(exit) < orderedExpands.get(up).getDistance(exit))
			startDir = 1;
		else
			startDir = -1;
	}
	
	private boolean haveUnits(int dropshipsNeeded)
	{
		// We want there to be enough units left to defend.
		if (Tyr.bot.homeGroup.units.size() < 10)
			return false;
		
		int tanks = 0;
		for (Agent agent : Tyr.bot.homeGroup.units)
			if (agent.isTank())
				tanks++;
		
		if (tanks < 6)
			return false;
		
		boolean[][] found = new boolean[requiredUnits.length][];
		for (int listIt = 0; listIt < requiredUnits.length; listIt++)
			found[listIt] = new boolean[requiredUnits[listIt].length];
		int dropships = 0;
		
		// For each type of unit we want, see if we have it.
		for(Agent agent : Tyr.bot.homeGroup.units)
		{
			if (agent.unit.getType() == UnitType.Terran_Dropship)
			{
				dropships++;
				continue;
			}
			for (int listIt = 0; listIt<requiredUnits.length; listIt++)
				for (int typeIt=0; typeIt<requiredUnits[listIt].length; typeIt++)
					if (!found[listIt][typeIt] && agent.unit.getType() == requiredUnits[listIt][typeIt])
						found[listIt][typeIt] = true;
		}
		
		if (dropships < dropshipsNeeded)
		{
			DebugMessages.addMessage("Not enough dropships for drop.");
			return false;
		}
		
		boolean success = true;
		
		for (boolean[] passengersFound : found)
			for (boolean passengerFound : passengersFound)
				if (!passengerFound)
				{
					DebugMessages.addMessage("Not enough units for drop.");
					success = false;
				}
		
		return success;
	}
}
