/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.tasks;

import java.util.ArrayList;

import com.tyr.EnemyManager;
import com.tyr.Tyr;
import com.tyr.agents.Agent;
import com.tyr.agents.None;
import com.tyr.unitgroups.IAttackGroup;
import com.tyr.unitgroups.PrioritizedAttackGroup;
import com.tyr.unitgroups.UnitGroup;

import bwapi.Color;
import bwapi.Game;
import bwapi.Player;
import bwapi.Position;

/**
 * This class implements a solution for harassing the opponent with corsairs.
 */
public class CorsairHarassSolution extends Solution
{
	/**
	 *  The group of attacking units.
	 */
	private IAttackGroup attackGroup;
	
	/**
	 * List of all expands, in the order that they are encountered in a sweep around the map.
	 * Includes opponents main.
	 */
	private ArrayList<Position> orderedExpands;
	
	/**
	 * The position of the base we are currently moving to observe in the list of bases returned by EnemyManager.getOrderedexpands().
	 */
	private int currentPos;
	
	/**
	 * This class implements a solution for harassing the opponent with corsairs.
	 * @param task The task that started this solution.
	 */
	public CorsairHarassSolution(Task task) 
	{
		super(task);
		this.attackGroup = new PrioritizedAttackGroup(getTarget());
	}

	@Override
	public void onFrame(Game game, Player self, Tyr bot)
	{
		Position attackTarget = null;
		if (size() > 0)
			attackTarget = getTarget();
		if (attackTarget != null)
		{
			attackGroup.setTarget(attackTarget);
			// Draw some debug info.
			Tyr.drawCircle(attackTarget, Color.Orange, 64);
		}

		// Manage the attackGroup.
		attackGroup.cleanup();
		attackGroup.onFrame(game, self, bot);
	}
	
	/**
	 * Add an agent to the attack group with the attacking agents.
	 * @param unit The agent to be added.
	 */
	public void add(Agent agent)
	{
		attackGroup.add(agent);
	}
	
	/**
	 * This method is called when this solution is no longer needed.
	 */
	public void done(UnitGroup unitGroup) 
	{
		for(Agent unit : attackGroup.units)
		{
			unitGroup.add(unit);
			unit.order(new None(unit));
		}
		attackGroup.units = new ArrayList<Agent>();
	}
	
	/**
	 * Clears the solution so that it can be removed.
	 */
	public void clear() 
	{
		attackGroup.clear();
	}

	/**
	 * Returns the number if agents sent out by this solution.
	 * @return Returns the number if agents sent out by this solution.
	 */
	public int size() 
	{
		return attackGroup.units.size();
	}
	
	public boolean done() 
	{
		return size() == 0;
	}
	
	/**
	 * Gets us the next target to harass.
	 */
	@SuppressWarnings("unchecked")
	private Position getTarget()
	{
		if (orderedExpands == null && Tyr.bot.suspectedEnemy.size() <= 1)
		{
			orderedExpands = (ArrayList<Position>)EnemyManager.getManager().getOrderedExpands().clone();
			if (Tyr.bot.suspectedEnemy.size() == 1)
				orderedExpands.add(Tyr.bot.suspectedEnemy.get(0).getPosition());
		}
		
		if (orderedExpands == null || orderedExpands.size() == 0)
			return null;
	
		if (Tyr.game.isVisible(Tyr.positionToTile(orderedExpands.get(currentPos))))
			currentPos = (currentPos - 1 + orderedExpands.size()) % orderedExpands.size();
		return orderedExpands.get(currentPos);
	}
}
