/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.tasks;

import java.util.ArrayList;

import com.tyr.Force;
import com.tyr.ForceTracker;
import com.tyr.Tyr;
import com.tyr.agents.Agent;
import com.tyr.agents.None;
import com.tyr.unitgroups.AttackGroup;
import com.tyr.unitgroups.UnitGroup;

import bwapi.Color;
import bwapi.Game;
import bwapi.Player;
import bwapi.Position;

/**
 * This class implements a solution for attacking the opponent.
 * During the attack it will periodically regroup to make sure attacking units stay together.
 */
public class AntiLurkerSolution extends SolutionItem implements Force
{
	/**
	 *  The group of attacking units.
	 */
	private AttackGroup attackGroup;
	
	/**
	 *  The target of the attack.
	 */
	private Position attackTarget;
		
	/**
	 * The average position of the attacking units.
	 */
	private Position averagePos;
	
	/**
	 * This class implements a solution for attacking the opponent.
	 * During the attack it will periodically regroup to make sure attacking units stay together.
	 * @param task The task that started this solution.
	 * @param attackTarget The position which we should attack.
	 */
	public AntiLurkerSolution(Task task, Position attackTarget) 
	{
		super(task);
		
		ForceTracker.register(this);
		
		this.attackTarget = attackTarget;
		
		this.attackGroup = new AttackGroup(attackTarget);
	}
	
	/**
	 * Sets the target which the units will attack.
	 * @param attackTarget
	 */
	public void setTarget(Position attackTarget)
	{
		this.attackTarget = attackTarget;
	}

	@Override
	public void onFrame(Game game, Player self, Tyr bot)
	{
		getAveragePos();
		
		// Draw some debug info.
		Tyr.drawCircle(attackTarget, Color.Orange, 64);

		// Manage the attackGroup.
		attackGroup.cleanup();
		
		attackGroup.setTarget(attackTarget);
		attackGroup.onFrame(game, self, bot);
	}
	
	/**
	 * Add an agent to the attack group with the attacking agents.
	 * @param unit The agent to be added.
	 */
	public void add(Agent agent)
	{
		attackGroup.add(agent);
	}
	
	/**
	 * This method is called when this solution is no longer needed.
	 */
	public void done(UnitGroup unitGroup) 
	{
		for(Agent unit : attackGroup.units)
		{
			unitGroup.add(unit);
			unit.order(new None(unit));
		}
		attackGroup.units = new ArrayList<Agent>();
	}

	/**
	 * Adds the attacking units to the scanner to start finding the remaining enemy buildings.
	 */
	public void addToScanner() 
	{
		for(Agent agent : attackGroup.units)
			Tyr.bot.scanner.add(agent);
		attackGroup.units = new ArrayList<Agent>();
	}
	
	/**
	 * Clears the solution so that it can be removed.
	 */
	public void clear() 
	{
		attackGroup.clear();
	}

	/**
	 * Returns the number if agents sent out by this solution.
	 * @return Returns the number if agents sent out by this solution.
	 */
	public int size() 
	{
		return attackGroup.units.size();
	}
	
	/**
	 * Computes the average position of the attacking units and stores it in averagePos.
	 */
	private void getAveragePos()
	{
		// We compute the average position of all attacking units.
		double averageX = 0, averageY = 0;
		
		for(Agent agent : attackGroup.units)
		{
			Position pos = agent.unit.getPosition();
			averageX += pos.getX();
			averageY += pos.getY();
		}
		
		averageX /= attackGroup.units.size();
		averageY /= attackGroup.units.size();
		
		averagePos = new Position((int)averageX, (int) averageY);
	}

	@Override
	public Position getCenter() 
	{
		if (averagePos == null)
			getAveragePos();
		return averagePos;
	}

	@Override
	public boolean done() 
	{
		return size() == 0;
	}

	@Override
	public boolean disabled()
	{
		return false;
	}
}
