/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.builds;

import com.tyr.DebugMessages;
import com.tyr.EnemyManager;
import com.tyr.PositionUtil;
import com.tyr.Settings;
import com.tyr.Tyr;
import com.tyr.UnitTracker;
import com.tyr.agents.Agent;
import com.tyr.agents.Attack;
import com.tyr.requirements.ConjRequirement;
import com.tyr.requirements.CostRequirement;
import com.tyr.requirements.UnitRequirement;
import com.tyr.tasks.BuildAtLocationTask;
import com.tyr.tasks.ConstantPushSolution;
import com.tyr.tasks.ConstantPushTask;
import com.tyr.tasks.DefendMainTask;
import com.tyr.unitgroups.PrioritizedAttackGroup;

import bwapi.Color;
import bwapi.Game;
import bwapi.Player;
import bwapi.Position;
import bwapi.TilePosition;
import bwapi.Unit;
import bwapi.UnitCommand;
import bwapi.UnitType;
import bwapi.UpgradeType;


public class ZealotPush extends CompositeBuildOrder
{
	private boolean doProxy = true;
	private boolean waitForAttack = false;
	private boolean workerRush = false;
	private AntiWorkerRushProtoss antiWorkerRush = new AntiWorkerRushProtoss();
	private ConstantPushTask constantPushTask;
	private boolean cancelGateways = true;
	private boolean zealotOnly = false;
	
	public ZealotPush()
	{
		this(false, false, false);
	}
	
	public ZealotPush(boolean doProxy, boolean waitForAttack)
	{
		this.doProxy = doProxy;
		this.waitForAttack = waitForAttack;
	}
	
	public ZealotPush(boolean doProxy, boolean waitForAttack, boolean zealotOnly)
	{
		this.doProxy = doProxy;
		this.waitForAttack = waitForAttack;
		this.zealotOnly = zealotOnly;
	}

	@Override
	public void initialize(Game game, Player self, Tyr bot)
	{
		Settings.setRequiredSize(100);
		Settings.setMaximumSize(100);
		Settings.setMaximumWorkers(7);
		Settings.setWorkersPerGas(2);

		this.add(new WorkerScoutPart(1600));
		this.add(new ExpandPart(true));
		
		//if (game.enemy().getRace() != Race.Terran)
		//	bot.taskManager.potentialTasks.add(RunbyTask.getTask());
		PrioritizedAttackGroup.allowRetreat = true;
		
		if (doProxy)
		{
			Position proxy = getProxyPosition();
			
			BuildAtLocationTask buildTask = new BuildAtLocationTask(proxy, false);
			buildTask.addBuilding(UnitType.Protoss_Pylon, new UnitRequirement(UnitType.Protoss_Probe, 7));
			buildTask.addBuilding(UnitType.Protoss_Gateway, new ConjRequirement()
				.addRequirement(new UnitRequirement(UnitType.Protoss_Pylon, 1, true))
				.addRequirement(new CostRequirement(250, 0)));
			buildTask.addBuilding(UnitType.Protoss_Gateway, new ConjRequirement()
			.addRequirement(new UnitRequirement(UnitType.Protoss_Gateway))
			.addRequirement(new CostRequirement(100, 0)));
			buildTask.addBuilding(UnitType.Protoss_Gateway, new CostRequirement(300, 0));
			buildTask.addBuilding(UnitType.Protoss_Gateway, new CostRequirement(250, 0));
			buildTask.addBuilding(UnitType.Protoss_Gateway, new CostRequirement(250, 0));
			bot.taskManager.potentialTasks.add(buildTask);
		}
		
		Attack.requiredAtCannon = 0;
		
		if (!waitForAttack)
		{
			constantPushTask = new ConstantPushTask(null, ConstantPushSolution.PRIORITIZE);
			bot.taskManager.potentialTasks.add(constantPushTask);
		}
		
		
		super.initialize(game, self, bot);
	}
	
	private Position getProxyPosition()
	{
		Game game = Tyr.game;
		int width = 20;
		int height = 20;
		int powerRange = 8;
		Position best = new Position(game.mapWidth()*16, game.mapHeight()*16);
		int power = 0;
		int dist = 0;
		for (int posX = game.mapWidth()/2 - width / 2; posX <= game.mapWidth()/2 + width / 2; posX ++)
		{
			for (int posY = game.mapHeight()/2 - height / 2; posY <= game.mapHeight()/2 + height / 2; posY ++)
			{
				int newDist = Math.abs(game.mapWidth()/2 - posX) + Math.abs(game.mapHeight()/2 - posY);
				int newPower = 0;
				for (int checkX = -powerRange; checkX <= powerRange; checkX++)
					for (int checkY = -powerRange; checkY <= powerRange; checkY++)
						if (game.isBuildable(new TilePosition(posX + checkX, posY + checkY)))
							newPower++;
				if (newPower > power || (newPower == power && newDist < dist))
				{
					power = newPower;
					dist = newDist;
					best = new Position(posX * 32, posY * 32);
				}
			}
		}
		
		System.out.println("Best power found: " + power);
		
		return best;
	}

	@Override
	public void onFrame(Game game, Player self, Tyr bot) 
	{
		if (!workerRush && EnemyManager.getManager().getAllCount(MARINE) < 5)
		{
			int count = 0;
			for (Unit unit : EnemyManager.getEnemyUnits())
				if (unit.getType().isWorker()
						&& PositionUtil.distanceSq(unit, Tyr.tileToPosition(self.getStartLocation())) <= Settings.getLargeInvasionDist() * Settings.getLargeInvasionDist())
					count++;
			
			if (count >= 3)
			{
				workerRush = true;
				antiWorkerRush.initialize(game, self, bot);
			}
		}
		
		if (EnemyManager.getManager().getAllCount(MARINE) >= 5)
		{
			workerRush = false;
			constantPushTask.stop = false;
			DefendMainTask.workerRush = false;
		}
		
		if (workerRush)
		{
			DebugMessages.addMessage("Countering worker rush.");
			
			if (UnitTracker.count(UnitType.Protoss_Gateway) < 4 && bot.getAvailableMinerals() >= 350)
			{
				cancelGateways = false;
				bot.spaceManager.build(UnitType.Protoss_Gateway);
			}
			
			Settings.setClearMinerals(false);
			if (constantPushTask != null)
				constantPushTask.stop = true;
			if (cancelGateways && UnitTracker.count(UnitType.Protoss_Gateway) > 1 && self.completedUnitCount(UnitType.Protoss_Gateway) < UnitTracker.count(UnitType.Protoss_Gateway))
			{
				if (game.getFrameCount() % 5 == 0 )
				{
					for (Unit unit : self.getUnits())
						if (unit.getType() == UnitType.Protoss_Gateway && ! unit.isCompleted())
						{
							Tyr.drawCircle(unit.getPosition(), Color.Purple, 6);
							unit.issueCommand(UnitCommand.cancelConstruction(unit));
							unit.cancelConstruction();
							break;
						}
				}
			}
			antiWorkerRush.onFrame(game, self, bot);
			return;
		}
		DebugMessages.addMessage("Performing zealot rush.");
		super.onFrame(game, self, bot);
		
		if (waitForAttack && self.completedUnitCount(UnitType.Protoss_Zealot) >= 10)
		{
			waitForAttack = false;
			constantPushTask = new ConstantPushTask(null, ConstantPushSolution.PRIORITIZE);
			bot.taskManager.potentialTasks.add(constantPushTask);
		}
		if (UnitTracker.count(UnitType.Protoss_Cybernetics_Core) > 0)
			Settings.setMaximumWorkers(30);
		else if (UnitTracker.count(UnitType.Protoss_Gateway) >= 4)
			Settings.setMaximumWorkers(25);
		else if (UnitTracker.count(UnitType.Protoss_Gateway) >= 2)
			Settings.setMaximumWorkers(9);
		if (UnitTracker.count(UnitType.Protoss_Zealot) >= 2 && UnitTracker.count(UnitType.Protoss_Gateway) < 4)
			Settings.setMaximumWorkers(20);
		
		if (!doProxy)
		{
			if (UnitTracker.count(UnitType.Protoss_Probe) >= 7 && UnitTracker.count(UnitType.Protoss_Pylon) == 0 && bot.getAvailableMinerals() >= 100)
			{
				bot.spaceManager.build(UnitType.Protoss_Pylon);
			}
			
			if (self.completedUnitCount(UnitType.Protoss_Pylon) > 0 && bot.getAvailableMinerals() >= 250 && UnitTracker.count(UnitType.Protoss_Gateway) == 0)
			{
				bot.spaceManager.build(UnitType.Protoss_Gateway);
			}
			
			if (UnitTracker.count(UnitType.Protoss_Gateway) == 1 && bot.getAvailableMinerals() >= 100)
			{
				bot.spaceManager.build(UnitType.Protoss_Gateway);
			}
			
			if (UnitTracker.count(UnitType.Protoss_Gateway) == 2 && bot.getAvailableMinerals() >= 300)
			{
				bot.spaceManager.build(UnitType.Protoss_Gateway);
			}
			
			if (UnitTracker.count(UnitType.Protoss_Gateway) >= 3 && UnitTracker.count(UnitType.Protoss_Gateway) < 4 && bot.getAvailableMinerals() >= 250)
			{
				bot.spaceManager.build(UnitType.Protoss_Gateway);
			}
		}
		
		//if we're running out of supply and have enough minerals ...
		if ((self.supplyTotal() + UnitTracker.getSupplyConstructing() - self.supplyUsed() 
					<= UnitTracker.count(UnitType.Protoss_Gateway)*3 + UnitTracker.getCcCount() * 3)
				&& (bot.getAvailableMinerals() >= 100)
				&& self.supplyTotal() + UnitTracker.getSupplyConstructing() < 400
				&& UnitTracker.count(UnitType.Protoss_Pylon) >= 1
				&& (UnitTracker.count(UnitType.Protoss_Zealot) >= 2 || self.supplyTotal() + UnitTracker.getSupplyConstructing() - self.supplyUsed() < 4))
		{
			bot.spaceManager.build(UnitType.Protoss_Pylon);
		}
		
		if(UnitTracker.count(UnitType.Protoss_Assimilator) < 2
				&& bot.getAvailableMinerals() >= 100 
				&& UnitTracker.getGeyserCount() > 0
				&& (UnitTracker.count(UnitType.Protoss_Probe) >= 25 || bot.getAvailableMinerals() >= 300)
				&& !zealotOnly
				//&& UnitTracker.count(UnitType.Protoss_Nexus) >= 2
				)
		{
			bot.spaceManager.build(UnitType.Protoss_Assimilator);
		}
		
		if (bot.getAvailableMinerals() >= 200 
				&& UnitTracker.count(UnitType.Protoss_Cybernetics_Core) < 1
				&& (UnitTracker.count(UnitType.Protoss_Probe) >= 25 || bot.getAvailableMinerals() >= 300)
				&& !zealotOnly
				//&& UnitTracker.count(UnitType.Protoss_Nexus) >= 2
				)
		{
			bot.spaceManager.build(UnitType.Protoss_Cybernetics_Core);
		}
		
		if (bot.getAvailableMinerals() >= 300
				&& UnitTracker.count(UnitType.Protoss_Nexus) >= 2
				&& UnitTracker.count(UnitType.Protoss_Cybernetics_Core) > 0
				&& UnitTracker.count(UnitType.Protoss_Forge) == 0)
		{
			bot.spaceManager.build(UnitType.Protoss_Forge);
		}
		
		//if we've the resources to build a Gateway ...
		/**
		if (UnitTracker.count(UnitType.Protoss_Gateway) < 6
				&& UnitTracker.count(UnitType.Protoss_Gateway) >= 4
				&& bot.getAvailableMinerals() >= 300) 
		{
			bot.spaceManager.build(UnitType.Protoss_Gateway);
		}
		**/
		if (UnitTracker.count(UnitType.Protoss_Gateway) < 7
				&& UnitTracker.count(UnitType.Protoss_Gateway) >= 4
				&& bot.getAvailableMinerals() >= 300
				&& UnitTracker.count(UnitType.Protoss_Nexus) >= 2
				&& UnitTracker.count(UnitType.Protoss_Forge) > 0) 
		{
			bot.spaceManager.build(UnitType.Protoss_Gateway);
		}
			
	}
	
	@Override
	public boolean overrideStructureOrder(Game game, Player self, Tyr bot, Agent agent)
	{
		if (workerRush)
			return antiWorkerRush.overrideStructureOrder(game, self, bot, agent);
		
		if (agent.unit.getType() == UnitType.Protoss_Gateway && !agent.unit.isTraining())
		{

			if ((UnitTracker.count(UnitType.Protoss_Zealot) <= UnitTracker.count(UnitType.Protoss_Dragoon) || self.completedUnitCount(UnitType.Protoss_Cybernetics_Core) == 0)
					&& bot.getAvailableMinerals() >= 100)
				agent.unit.build(UnitType.Protoss_Zealot);
			else if (UnitTracker.count(UnitType.Protoss_Zealot) > UnitTracker.count(UnitType.Protoss_Dragoon)
					&& bot.getAvailableMinerals() >= 125 && bot.getAvailableGas() >= 50)
				agent.unit.build(UnitType.Protoss_Dragoon);
			return true;
		}
		else if (agent.unit.getType() == UnitType.Protoss_Cybernetics_Core && !agent.unit.isUpgrading())
		{
			if(bot.getAvailableMinerals() >= UpgradeType.Singularity_Charge.mineralPrice()
					&& bot.getAvailableGas() >= UpgradeType.Singularity_Charge.gasPrice())
				agent.unit.upgrade(UpgradeType.Singularity_Charge);
		}
		else if (agent.unit.getType() == UnitType.Protoss_Forge && !agent.unit.isUpgrading())
		{
			if(bot.getAvailableMinerals() >= UpgradeType.Protoss_Ground_Weapons.mineralPrice()
					&& bot.getAvailableGas() >= UpgradeType.Protoss_Ground_Weapons.gasPrice())
				agent.unit.upgrade(UpgradeType.Protoss_Ground_Weapons);
			
			if(bot.getAvailableMinerals() >= UpgradeType.Protoss_Ground_Armor.mineralPrice()
					&& bot.getAvailableGas() >= UpgradeType.Protoss_Ground_Armor.gasPrice())
				agent.unit.upgrade(UpgradeType.Protoss_Ground_Armor);
		}
		return false;
	}
}