/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.builds;

import com.tyr.DebugMessages;
import com.tyr.Settings;
import com.tyr.StopWatch;
import com.tyr.Tyr;
import com.tyr.UnitTracker;
import com.tyr.agents.Agent;
import com.tyr.buildingplacement.DefensiveStructures;
import com.tyr.requirements.UnitRequirement;
import com.tyr.unitgroups.ScoutGroup;

import bwapi.Game;
import bwapi.Player;
import bwapi.UnitType;


/**
 * This build is supposed to work well against almost any Protoss opponent.
 * @author Simon
 *
 */
public class TvPGeneric extends CompositeBuildOrder
{
	private DefensiveMech mech = new DefensiveMech(true, true, true);
	private TAntiCannon antiCannon = new TAntiCannon();
	private TAntiZealot antiZealot = new TAntiZealot();
	private int mode = NONE;

	private static final int NONE = 0;
	private static final int MECH = 1;
	private static final int ANTI_CANNON = 2;
	private static final int ANTI_ZEALOT = 3;
	
	@Override
	public void initialize(Game game, Player self, Tyr bot)
	{
		Settings.setRequiredSize(20);
		Settings.setMaximumSize(40);
		
		Settings.setWorkersPerGas(1);
		
		this.add(new SupplyDepotPart());
		this.add(new WorkerScoutPart(1600));
		
		this.add(new GenericPartFactory(UnitType.Terran_Barracks)
			.setCost(100, 0)
			.create());
		
		this.add(new GenericPartFactory(UnitType.Terran_Engineering_Bay)
			.setMax(2)
			.add(new UnitRequirement(UnitType.Terran_Marine, 20))
			.add(new UnitRequirement(UnitType.Terran_Academy))
			.create());
		
		super.initialize(game, self, bot);
	}
	
	@Override
	public void onFrame(Game game, Player self, Tyr bot)
	{
		DebugMessages.addMessage("TvP build.");
		
		StopWatch watch = new StopWatch();
		long time;
		watch.start();
		
		super.onFrame(game, self, bot);
		
		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("TvP super time: " + time); 
		
		// Build bunkers.
		if (bot.getAvailableMinerals() >= 100 && UnitTracker.count(UnitType.Terran_Barracks) >= 1)
		{
			for(DefensiveStructures structures : bot.defensiveStructures)
			{
				int count = structures.getUnitCount(UnitType.Terran_Bunker);

				// We do not build a bunker at our main if there already is one at our natural.
				if (bot.defensiveStructures.size() > 1 
						&& structures.defendedPosition.getDistance(Tyr.tileToPosition(self.getStartLocation())) <= 200)
					continue;
				
				if(count < 1)
				{
					bot.spaceManager.buildDefensive(UnitType.Terran_Bunker, structures);
		  			break;
				}
			}
		}

		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Bio bunker time: " + time);

		// Decide which mode we should use against the opponent.
		if (mode == NONE)
		{
			mode = MECH;
			mech.initialize(game, self, bot);
		}
		
		if (mode == NONE || mode == MECH)
		{
			if (bot.scout.opponentStrategy == ScoutGroup.cannons)
			{
				mode = ANTI_CANNON;
				antiCannon.initialize(game, self, bot);
			}
			else if (bot.scout.opponentStrategy == ScoutGroup.zealotPush && UnitTracker.count(UnitType.Terran_Vulture) < 5)
			{
				mode = ANTI_ZEALOT;
				antiZealot.initialize(game, self, bot);
			}
		}
		if (mode == ANTI_ZEALOT && UnitTracker.count(UnitType.Terran_Vulture) >= 10)
			mode = MECH;
		
		if (mode == MECH)
			mech.onFrame(game, self, bot);
		
		if (mode == ANTI_CANNON)
			antiCannon.onFrame(game, self, bot);
		
		if (mode == ANTI_ZEALOT)
			antiZealot.onFrame(game, self, bot);
	}

	@Override
	public boolean overrideStructureOrder(Game game, Player self, Tyr bot, Agent agent)
	{
		if(super.overrideStructureOrder(game, self, bot, agent))
			return true;
		
		if (agent.unit.getType()== UnitType.Terran_Barracks
				&& UnitTracker.count(UnitType.Terran_Marine) < 4 * UnitTracker.count(UnitType.Terran_Bunker))
			return false;
		
		if (mode == MECH)
			return mech.overrideStructureOrder(game, self, bot, agent);
		if (mode == ANTI_CANNON)
			return antiCannon.overrideStructureOrder(game, self, bot, agent);
		if (mode == ANTI_ZEALOT)
			return antiZealot.overrideStructureOrder(game, self, bot, agent);
		return false;
	}
}
