/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.builds;

import bwapi.Game;
import bwapi.Player;
import bwapi.Position;
import bwapi.Race;
import bwapi.TechType;
import bwapi.Unit;
import bwapi.UnitType;
import bwapi.UpgradeType;
import bwta.BWTA;

import com.tyr.DebugMessages;
import com.tyr.EnemyManager;
import com.tyr.PositionUtil;
import com.tyr.Settings;
import com.tyr.Tyr;
import com.tyr.UnitTracker;
import com.tyr.agents.Agent;
import com.tyr.agents.Attack;
import com.tyr.buildingplacement.SpaceManager;
import com.tyr.requirements.ConjRequirement;
import com.tyr.requirements.CostRequirement;
import com.tyr.requirements.UnitRequirement;
import com.tyr.tasks.BuildAtLocationTask;
import com.tyr.tasks.ConstantPushSolution;
import com.tyr.tasks.ConstantPushTask;
import com.tyr.tasks.LocalDefenseTask;
import com.tyr.unitgroups.ScoutGroup;

public class PvT extends CompositeBuildOrder
{
	private ConstantPushTask constantPushTask;
	private LocalDefenseTask localDefenseTask;
	private LocalDefenseTask thirdDefenseTask;
	public boolean attackStarted;
	
	private int armySize = 25;
	private int retreatSize = 20;
	
	private boolean mechDetected = false;
	
	private boolean upgradeLegSpeed;
	private boolean detectionNeeded = false;

	private boolean useStorm = true;
	private boolean useArbiters = true;
	private boolean useDT = false;

	private boolean counterZerglingPush = false;
	private boolean counterMassZerglings = false;
	private boolean counterMutas = false;

	private boolean defensesInitialized = false;
	private boolean defendThird = true;
	
	public PvT(int armySize)
	{
		this.armySize = armySize;
	}
	
	public PvT(int armySize, boolean useArbiters, boolean useDT, boolean counterMutas, boolean defendThird)
	{
		this.armySize = armySize;
		this.useArbiters = useArbiters;
		this.useDT = useDT;
		this.counterMutas = counterMutas;
		this.defendThird = defendThird;
	}
	
	@Override
	public void initialize(Game game, Player self, Tyr bot)
	{
		Settings.setRequiredSize(100);
		Settings.setMaximumSize(100);
		Settings.setMaximumWorkers(60);
		Settings.setSmallInvasionDist(480);
		Settings.setLargeInvasionDist(768);
		Settings.setWorkersPerGas(2);
		
		Attack.dontWaitAtCannon = true;
		
		this.add(new ExpandPart(true));
		if (game.mapFileName().contains("Alchemist") || !ScoutGroup.enemyRaceKnown())
			this.add(new WorkerScoutPart(100));
		else
			this.add(new WorkerScoutPart(1600));
		ExpandPart.maximumCcs = 2;
		
		localDefenseTask = new LocalDefenseTask(BWTA.getRegion(self.getStartLocation()), 0);
		bot.taskManager.potentialTasks.add(localDefenseTask);
		
		thirdDefenseTask = new LocalDefenseTask(null, 0);
		bot.taskManager.potentialTasks.add(thirdDefenseTask);
		constantPushTask = new ConstantPushTask(null, ConstantPushSolution.ATTACK);
		constantPushTask.stop = true;
		bot.taskManager.potentialTasks.add(constantPushTask);
		
		ConstantPushTask.prioritizeExpands = true;
		
		super.initialize(game, self, bot);
	}
	
	@Override
	public void onFrame(Game game, Player self, Tyr bot) 
	{
		DebugMessages.addMessage("Performing PvT build.");
		super.onFrame(game, self, bot);
		
		if (!mechDetected && 
				(EnemyManager.getManager().getEnemyTypes().contains(UnitType.Terran_Siege_Tank_Siege_Mode)
				|| EnemyManager.getManager().getEnemyTypes().contains(UnitType.Terran_Siege_Tank_Tank_Mode)
				|| EnemyManager.getManager().getEnemyTypes().contains(UnitType.Terran_Machine_Shop)))
			mechDetected = true;
		
		if (!mechDetected)
		{
			for (Unit enemy : EnemyManager.getEnemyUnits())
				if (enemy.getType() == UnitType.Terran_Vulture && PositionUtil.distanceSq(enemy.getPosition(), Tyr.getStartLocation()) <= 1536 * 1536)
				{
					mechDetected = true;
					break;
				}
		}
		
		if (game.getFrameCount() >= 15000)
			counterZerglingPush = false;
		else if (EnemyManager.getManager().getAllCount(UnitType.Zerg_Zergling) >= 6)
			counterZerglingPush = true;
		
		if (EnemyManager.getManager().getAllCount(UnitType.Zerg_Mutalisk) > 0)
			counterMutas = true;
		
		counterMassZerglings = ScoutGroup.enemyRace == Race.Zerg 
			&& EnemyManager.getManager().getAllCount(UnitType.Zerg_Mutalisk)
				+ EnemyManager.getManager().getAllCount(UnitType.Zerg_Hydralisk) 
				+ EnemyManager.getManager().getAllCount(UnitType.Zerg_Lurker) == 0
			&& game.getFrameCount() >= 12000
			&& !counterMutas;
		
		if (mechDetected)
			detectionNeeded = true;
		
		useStorm = !mechDetected && ScoutGroup.enemyRace == Race.Terran;
		
		if (ScoutGroup.enemyRace == Race.Zerg)
			useArbiters = false;
		
		if (ScoutGroup.enemyRace == Race.Protoss || ScoutGroup.enemyRace == Race.Zerg)
			detectionNeeded = true;
		
		if (gas() >= 400)
			Settings.setWorkersPerGas(2);
		else if ((gas() < 250 && count(GATEWAY) >= 12) 
				|| gas() + 400 < minerals())
			Settings.setWorkersPerGas(3);
		
		if (constantPushTask != null && completed(ZEALOT) + completed(DRAGOON) <= retreatSize)
			constantPushTask.stop = true;
		if (constantPushTask != null && completed(ZEALOT) + completed(DRAGOON) >= armySize
				&& (completed(HIGH_TEMPLAR) > 0 || !useStorm) 
				&& (self.hasResearched(TechType.Psionic_Storm) || !useStorm))
		{
			ExpandPart.maximumCcs = 100;
			constantPushTask.stop = false;
			attackStarted = true;
			localDefenseTask.setDefendersNeeded(3);
			armySize = Math.max(armySize, 30);
			detectionNeeded = true;
		}
		
		if (defendThird && thirdDefenseTask.getDefendersNeeded() == 0 && Tyr.bot.workForce.mineralWorkers.size() >= 3)
		{
			thirdDefenseTask.setDefendersNeeded(3);
			thirdDefenseTask.setDefendedPosition(Tyr.bot.workForce.mineralWorkers.get(2).resourceDepot.getPosition());
		}
		
		if (completed(DRAGOON) >= 8)
			ExpandPart.maximumCcs = Math.max(3, ExpandPart.maximumCcs);
		
		if (completed(ZEALOT) + completed(DRAGOON) >= 9)
			upgradeLegSpeed = true;
		
		//if we're running out of supply and have enough minerals ...
		if ((self.supplyTotal() + UnitTracker.getSupplyConstructing() - self.supplyUsed() 
					<= count(GATEWAY)*3 + UnitTracker.getCcCount() * 3)
				&& (minerals() >= 100)
				&& self.supplyTotal() + UnitTracker.getSupplyConstructing() < 400)
		{
			bot.spaceManager.build(PYLON);
		}
		

		if (!defensesInitialized && counterMutas && game.getFrameCount() >= 8000)
		{
			defensesInitialized = true;
			
			final Position defensivePosition = SpaceManager.getNaturalDefensePos();
				
			final BuildAtLocationTask buildTask = new BuildAtLocationTask(defensivePosition, true);
			buildTask.addBuilding(UnitType.Protoss_Pylon, new UnitRequirement(UnitType.Protoss_Probe, 7));

			for (int i=0; i<2; i++)
			{
				final ConjRequirement requirements = new ConjRequirement()
						.addRequirement(new CostRequirement(150, 0));
				buildTask.addBuilding(UnitType.Protoss_Photon_Cannon, requirements);
			}
			
			bot.taskManager.potentialTasks.add(buildTask);
		}
		
		if (completed(FORGE) > 0
			&& minerals() >= 150
			&& count(CANNON) < 4
			&& counterMutas
			&& game.getFrameCount() >= 8000)
			build(CANNON);
		
		//if we've the resources to build a Gateway ...
		if ((count(NEXUS) >= 2)
				&& minerals() >= 150
				&& count(GATEWAY) == 0
				&& completed(PYLON) > 0
				&& count(PYLON) > 0)
		{
			bot.spaceManager.build(GATEWAY);
		}

		if ((count(NEXUS) >= 2)
				&& minerals() >= 150
				&& count(GATEWAY) < 2
				&& count(CYBERNETICS_CORE) > 0)
		{
			bot.spaceManager.build(GATEWAY);
		}
		
		if ((ScoutGroup.enemyRace == Race.Protoss || ScoutGroup.enemyRace == Race.Zerg)
				&& minerals() >= 150
				&& count(GATEWAY) < (ScoutGroup.enemyRace == Race.Zerg ? 2 : 1))
		{
			bot.spaceManager.build(GATEWAY);
		}
		
		if (!pauseProduction())
		{
			if ((count(NEXUS) >= 2)
					&& minerals() >= 250
					&& count(GATEWAY) < 4
					&& (count(CYBERNETICS_CORE) > 0))
			{
				bot.spaceManager.build(GATEWAY);
			}
			
			if (count(NEXUS) >= 2
					&& minerals() >= 250
					&& (count(CYBERNETICS_CORE) > 0)
					&& count(GATEWAY) < 8)
			{
				bot.spaceManager.build(GATEWAY);
			}
			
			if (count(NEXUS) >= 2
					&& minerals() >= 150
					&& completed(GATEWAY) >= 8
					&& count(GATEWAY) < 12)
			{
				bot.spaceManager.build(GATEWAY);
			}
		}
		
		if(minerals() >= 100 
				&& UnitTracker.getGeyserCount() > 0
				&& count(GATEWAY) > 0
				&& (count(NEXUS) >= 2))
		{
			bot.spaceManager.build(ASSIMILATOR);
		}
		
		if (minerals() >= 200 
				&& count(CYBERNETICS_CORE) < 1
				&& completed(GATEWAY) > 0
				&& (count(NEXUS) >= 2))
		{
			bot.spaceManager.build(CYBERNETICS_CORE);
		}
		
		if (minerals() >= 150
				&& count(FORGE) < 2
				&& count(DRAGOON) >= 2)
		{
			bot.spaceManager.build(FORGE);
		}
		
		if (upgradeLegSpeed 
				&& minerals() >= 150
				&& gas() >= 100
				&& count(CITADEL) == 0
				&& completed(CYBERNETICS_CORE) > 0
				&& count(NEXUS) >= 2)
		{
			bot.spaceManager.build(CITADEL);
		}
		
		if (minerals() >= 150
				&& gas() >= 200
				&& (useStorm || useArbiters 
						|| self.getUpgradeLevel(UpgradeType.Protoss_Ground_Armor) + self.getUpgradeLevel(UpgradeType.Protoss_Ground_Weapons) > 0) 
				&& count(TEMPLAR_ARCHIVES) < 1
				&& completed(CITADEL) > 0
				&& count(NEXUS) >= 2)
		{
			build(TEMPLAR_ARCHIVES);
		}
		
		if (gas() >= 200 && minerals() >= 200
				&& detectionNeeded
				&& count(ROBOTICS_FACILITY) == 0
				&& (count(DRAGOON) >= 5))
		{
			bot.spaceManager.build(ROBOTICS_FACILITY);
		}
		
		if (gas() >= 50 && minerals() >= 100
				&& completed(ROBOTICS_FACILITY) > 0
				&& count(OBSERVATORY) == 0
				&& detectionNeeded)
		{
			bot.spaceManager.build(OBSERVATORY);
		}
		
		if (minerals() >= 150
			&& gas() >= 100
			&& useArbiters
			&& count(STARGATE) == 0
			&& completed(CYBERNETICS_CORE) > 0
			&& count(DRAGOON) >= 8
					&& (self.hasResearched(TechType.Psionic_Storm) || self.isResearching(TechType.Psionic_Storm) || !useStorm))
		{
			build(STARGATE);
		}
		
		if (minerals() >= 200
				&& gas() >= 150
				&& completed(STARGATE) > 0
				&& completed(TEMPLAR_ARCHIVES) > 0
				&& count(ARBITER_TRIBUNAL) == 0
				&& (self.hasResearched(TechType.Psionic_Storm) || self.isResearching(TechType.Psionic_Storm) || !useStorm))
		{
			build(ARBITER_TRIBUNAL);
		}
	}
	
	private boolean pauseProduction()
	{
		if (gas() > 250 && minerals() > 400)
			return false;

		if (upgradeLegSpeed && count(CITADEL) == 0)
			return true;
		if (completed(CITADEL) > 0 
				&& !Tyr.self.isUpgrading(UpgradeType.Leg_Enhancements)
				&& Tyr.self.getUpgradeLevel(UpgradeType.Leg_Enhancements) == 0)
			return true;
		if (completed(CITADEL) > 0
				&& count(TEMPLAR_ARCHIVES) == 0)
			return true;
		if (completed(TEMPLAR_ARCHIVES) > 0
				&& count(HIGH_TEMPLAR) > 0
				&& !Tyr.self.isResearching(TechType.Psionic_Storm)
				&& !Tyr.self.hasResearched(TechType.Psionic_Storm)
				&& useStorm)
			return true;
		if (detectionNeeded && count(DRAGOON) >= 5
				&& completed(ROBOTICS_FACILITY) == 1
				&& count(OBSERVATORY) == 0)
			return true;
		if (detectionNeeded && count(DRAGOON) >= 5
				&& completed(OBSERVATORY) == 1
				&& count(OBSERVER) == 0)
			return true;
		if (completed(STARGATE) > 0
				&& useArbiters
				&& completed(TEMPLAR_ARCHIVES) > 0
				&& count(ARBITER_TRIBUNAL) == 0
				&& (Tyr.self.hasResearched(TechType.Psionic_Storm) || Tyr.self.isResearching(TechType.Psionic_Storm) || !useStorm)
				&& count(ARBITER_TRIBUNAL) == 0)
			return true;
		if (detectionNeeded && count(DRAGOON) >= 5
				&& count(ROBOTICS_FACILITY) == 0)
			return true;
		if (attackStarted && count(NEXUS) < 3)
			return true;
		return false;
	}
	
	@Override
	public boolean overrideStructureOrder(Game game, Player self, Tyr bot, Agent agent)
	{
		if (agent.unit.getType() == GATEWAY && !agent.unit.isTraining())
		{
			boolean pause = pauseProduction();
			if ((ScoutGroup.enemyRace == Race.Protoss || ScoutGroup.enemyRace == Race.Zerg)
					&& count(ZEALOT) < (counterZerglingPush ? 8 : 4)
					&& minerals() >= 100)
				agent.unit.build(ZEALOT);
			
			if (pause && minerals() >= 600)
				agent.unit.build(ZEALOT);
			else if (count(ZEALOT) + count(DRAGOON) >= 10 * count(HIGH_TEMPLAR)
					&& count(HIGH_TEMPLAR) < 6
					&& completed(TEMPLAR_ARCHIVES) > 0
					&& !pause
					&& useStorm)
			{
				if (minerals() >= 50
						&& gas() >= 150)
				agent.unit.build(HIGH_TEMPLAR);
			}
			else if (self.completedUnitCount(UnitType.Protoss_Templar_Archives) > 0
					&& count(DARK_TEMPLAR) == 0
					&& minerals() >= 125
					&& gas() >= 100
					&& useDT)
				agent.unit.train(UnitType.Protoss_Dark_Templar);
			else if (count(ZEALOT) < (counterMassZerglings ? 2 : 1) * count(DRAGOON) - 5
					&& minerals() >= 100
					&& (minerals() >= 400 || !pause)
					)
				agent.unit.build(ZEALOT);
			else if (minerals() >= 125 && gas() >= 50
					&& ((minerals() >= 400 && gas() > 200) || !pause))
				agent.unit.build(DRAGOON);
			return true;
		}
		else if (agent.unit.getType() == FORGE && !agent.unit.isUpgrading())
		{
			if(pauseProduction() && ScoutGroup.enemyRace != Race.Zerg)
				return true;
			if(minerals() >= UpgradeType.Protoss_Ground_Weapons.mineralPrice()
					&& gas() >= UpgradeType.Protoss_Ground_Weapons.gasPrice())
				agent.unit.upgrade(UpgradeType.Protoss_Ground_Weapons);
			
			if(minerals() >= UpgradeType.Protoss_Ground_Armor.mineralPrice()
					&& gas() >= UpgradeType.Protoss_Ground_Armor.gasPrice())
				agent.unit.upgrade(UpgradeType.Protoss_Ground_Armor);
		}
		else if (agent.unit.getType() == CYBERNETICS_CORE && !agent.unit.isUpgrading())
		{
			if(minerals() >= UpgradeType.Singularity_Charge.mineralPrice()
					&& gas() >= UpgradeType.Singularity_Charge.gasPrice()
					&& count(DRAGOON) > 5)
				agent.unit.upgrade(UpgradeType.Singularity_Charge);
		}
		else if (agent.unit.getType() == ROBOTICS_FACILITY)
		{
			if ((count(OBSERVER) < 2)
					&& completed(OBSERVATORY) > 0
					&& minerals() >= 25
					&& gas() >= 75
					&& !agent.unit.isTraining())
				agent.unit.build(OBSERVER);
			return true;
		}
		else if (agent.unit.getType() == CITADEL && !agent.unit.isUpgrading())
		{
			if (gas() >= UpgradeType.Leg_Enhancements.gasPrice()
					&& minerals() >= UpgradeType.Leg_Enhancements.mineralPrice())
				agent.unit.upgrade(UpgradeType.Leg_Enhancements);
		}
		else if (agent.unit.getType() == TEMPLAR_ARCHIVES && !agent.unit.isResearching())
		{
			if (useStorm
					&& gas() >= TechType.Psionic_Storm.gasPrice()
					&& minerals() >= TechType.Psionic_Storm.mineralPrice()
					&& !self.hasResearched(TechType.Psionic_Storm))
				agent.unit.research(TechType.Psionic_Storm);
		}
		else if (agent.unit.getType() == STARGATE)
		{
			if (completed(ARBITER_TRIBUNAL) > 0
					&& minerals() >= 100
					&& gas() >= 350
					&& !agent.unit.isTraining())
				agent.unit.train(ARBITER);
			return true;
		}
		return false;
	}
}
