#include "StrategyManager.h"
#include "ProductionManager.h"
#include "StrategyBossZerg.h"

using namespace UAlbertaBot;

StrategyManager::StrategyManager()
	: _self(BWAPI::Broodwar->self())
	, _enemy(BWAPI::Broodwar->enemy())
	, _selfRace(BWAPI::Broodwar->self()->getRace())
	, _enemyRace(BWAPI::Broodwar->enemy()->getRace())
	, _emptyBuildOrder(BWAPI::Broodwar->self()->getRace())
	, _denScore(0)
	, _spireScore(0)
	, _lurkerScore(0)
	, _ultraScore(0)
	, _techTarget("None")
	, _minUnit("None")
	, _gasUnit("None")
	, _buildOrderZerg(BWAPI::Races::Zerg)
	, _droneEco(0)
{
}

StrategyManager & StrategyManager::Instance() 
{
	static StrategyManager instance;
	return instance;
}

const int StrategyManager::getScore(BWAPI::Player player) const
{
	return player->getBuildingScore() + player->getKillScore() + player->getRazingScore() + player->getUnitScore();
}

const BuildOrder & StrategyManager::getOpeningBookBuildOrder() const
{
	if (BWAPI::Broodwar->mapFileName() == "(8)Big Game Hunters.scm")
	{
		Config::Strategy::StrategyName = "3HatchHydra BGH";
	}

	auto buildOrderIt = _strategies.find(Config::Strategy::StrategyName);

    // look for the build order in the build order map
	if (buildOrderIt != std::end(_strategies))
    {
        return (*buildOrderIt).second._buildOrder;
    }
    else
    {
        UAB_ASSERT_WARNING(false, "Strategy not found: %s, returning empty initial build order", Config::Strategy::StrategyName.c_str());
        return _emptyBuildOrder;
    }
}

const bool StrategyManager::shouldExpandNow() const
{
	// if there is no place to expand to, we can't expand
	if (MapTools::Instance().getNextExpansion() == BWAPI::TilePositions::None)
	{
        //BWAPI::Broodwar->printf("No valid expansion location");
		return false;
	}

	size_t numDepots    = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center)
                        + UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Protoss_Nexus)
                        + UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Hatchery)
                        + UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Lair)
                        + UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Hive);
	int frame           = BWAPI::Broodwar->getFrameCount();
    int minute          = frame / (24*60);

	// if we have a ton of idle workers then we need a new expansion
	if (WorkerManager::Instance().getNumIdleWorkers() > 10)
	{
		return true;
	}

    // if we have a ridiculous stockpile of minerals, expand
	if (BWAPI::Broodwar->self()->minerals() > 600)
    {
        return true;
    }

    // we will make expansion N after array[N] minutes have passed
    std::vector<int> expansionTimes = {5, 10, 14, 17, 20, 22};

    for (size_t i(0); i < expansionTimes.size(); ++i)
    {
        if (numDepots < (i+2) && minute > expansionTimes[i])
        {
            return true;
        }
    }

	return false;
}

void StrategyManager::addStrategy(const std::string & name, Strategy & strategy)
{
    _strategies[name] = strategy;
}

const MetaPairVector StrategyManager::getBuildOrderGoal()
{
    if (_selfRace == BWAPI::Races::Protoss)
    {
		return getProtossBuildOrderGoal();
    }
	else if (_selfRace == BWAPI::Races::Terran)
	{
		return getTerranBuildOrderGoal();
	}

    return MetaPairVector();
}

const MetaPairVector StrategyManager::getProtossBuildOrderGoal() const
{
	// the goal to return
	MetaPairVector goal;

	int numZealots          = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Protoss_Zealot);
    int numPylons           = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Protoss_Pylon);
	int numDragoons         = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Protoss_Dragoon);
	int numProbes           = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Protoss_Probe);
	int numNexusCompleted   = BWAPI::Broodwar->self()->completedUnitCount(BWAPI::UnitTypes::Protoss_Nexus);
	int numNexusAll         = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Protoss_Nexus);
	int numCyber            = BWAPI::Broodwar->self()->completedUnitCount(BWAPI::UnitTypes::Protoss_Cybernetics_Core);
	int numCannon           = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Protoss_Photon_Cannon);
    int numScout            = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Protoss_Corsair);
    int numReaver           = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Protoss_Reaver);
    int numDarkTeplar       = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Protoss_Dark_Templar);

    if (Config::Strategy::StrategyName == "Protoss_ZealotRush")
    {
        goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Zealot, numZealots + 8));

        // once we have a 2nd nexus start making dragoons
        if (numNexusAll >= 2)
        {
            goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Dragoon, numDragoons + 4));
        }
    }
    else if (Config::Strategy::StrategyName == "Protoss_DragoonRush")
    {
        goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Dragoon, numDragoons + 6));
    }
    else if (Config::Strategy::StrategyName == "Protoss_Drop")
    {
        if (numZealots == 0)
        {
            goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Zealot, numZealots + 4));
            goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Shuttle, 1));
        }
        else
        {
            goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Zealot, numZealots + 8));
        }
    }
    else if (Config::Strategy::StrategyName == "Protoss_DTRush")
    {
        goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Dark_Templar, numDarkTeplar + 2));

        // if we have a 2nd nexus then get some goons out
        if (numNexusAll >= 2)
        {
            goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Dragoon, numDragoons + 4));
        }
    }
    else
    {
        UAB_ASSERT_WARNING(false, "Unknown Protoss Strategy Name: %s", Config::Strategy::StrategyName.c_str());
    }

    // if we have 3 nexus, make an observer
    if (numNexusCompleted >= 3)
    {
        goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Observer, 1));
    }
    
    // add observer to the goal if the enemy has cloaked units
	if (InformationManager::Instance().enemyHasCloakTech())
	{
		goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Robotics_Facility, 1));
		
		if (BWAPI::Broodwar->self()->completedUnitCount(BWAPI::UnitTypes::Protoss_Robotics_Facility) > 0)
		{
			goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Observatory, 1));
		}
		if (BWAPI::Broodwar->self()->completedUnitCount(BWAPI::UnitTypes::Protoss_Observatory) > 0)
		{
			goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Observer, 1));
		}
	}

    // if we want to expand, insert a nexus into the build order
	if (shouldExpandNow())
	{
		goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Nexus, numNexusAll + 1));
	}

	return goal;
}

const MetaPairVector StrategyManager::getTerranBuildOrderGoal() const
{
	// the goal to return
	std::vector<MetaPair> goal;

    int numWorkers      = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_SCV);
    int numCC           = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center);            
    int numMarines      = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine);
	int numMedics       = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Medic);
	int numWraith       = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Wraith);
    int numVultures     = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Vulture);
    int numGoliath      = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Goliath);
    int numTanks        = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
                        + UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode);
    int numBay          = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay);

    if (Config::Strategy::StrategyName == "Terran_MarineRush")
    {
	    goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Marine, numMarines + 8));

        if (numMarines > 5)
        {
            goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Engineering_Bay, 1));
        }
    }
    else if (Config::Strategy::StrategyName == "Terran_4RaxMarines")
    {
	    goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Marine, numMarines + 8));
    }
    else if (Config::Strategy::StrategyName == "Terran_VultureRush")
    {
        goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Vulture, numVultures + 8));

        if (numVultures > 8)
        {
            goal.push_back(std::pair<MacroAct, int>(BWAPI::TechTypes::Tank_Siege_Mode, 1));
            goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode, 4));
        }
    }
    else if (Config::Strategy::StrategyName == "Terran_TankPush")
    {
        goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode, 6));
        goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Goliath, numGoliath + 6));
        goal.push_back(std::pair<MacroAct, int>(BWAPI::TechTypes::Tank_Siege_Mode, 1));
    }
    else
    {
        BWAPI::Broodwar->printf("Warning: No build order goal for Terran Strategy: %s", Config::Strategy::StrategyName.c_str());
    }

    if (shouldExpandNow())
    {
        goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Command_Center, numCC + 1));
        goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_SCV, numWorkers + 10));
    }

	return goal;
}

void StrategyManager::getZergBuildOrder(BuildOrderQueue & queue)
{
	queue.clearAll();

	// Get the race when it is known
	_enemyRace = BWAPI::Broodwar->enemy()->getRace();

	int minerals		= std::max(0, _self->minerals() - BuildingManager::Instance().getReservedMinerals());
	int gas				= std::max(0, _self->gas() - BuildingManager::Instance().getReservedGas());
	int nLarvas			= UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Larva);
	int nMinPatches		= InformationManager::Instance().getMyNumMineralPatches();
	int nSupplyUsed		= BWAPI::Broodwar->self()->supplyUsed();
	int nSupplyTotal	= BWAPI::Broodwar->self()->supplyTotal();

	// Tech stuff. It has to be completed for the tech to be available.
	int nLairs			= UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Lair);
	int nLairsAll		= UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Lair);
	int nHives			= UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Hive);
	int nHivesAll		= UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Hive);
	int nHatches		= UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Hatchery)
						+ nLairsAll + nHivesAll;
	int nHatchesAll		= UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Hatchery)
						+ nLairsAll + nHivesAll;

	int nGas			= UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Extractor);
	int nGasAll			= UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Extractor);
	int nEvo			= UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Evolution_Chamber);
	int nEvoAll			= UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Evolution_Chamber);
	bool hasPool		= UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Spawning_Pool) > 0;
	bool hasPoolAll		= UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Spawning_Pool) > 0;
	bool hasDen			= UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Hydralisk_Den) > 0;
	bool hasDenAll		= UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Hydralisk_Den) > 0;
	bool hasLurker		= _self->hasResearched(BWAPI::TechTypes::Lurker_Aspect);
	bool hasSpire		= UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Spire) > 0 ||
						UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Greater_Spire) > 0;
	bool hasSpireAll	= UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Spire) > 0 ||
						UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Greater_Spire) > 0;
	bool hasUltra		= UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Ultralisk_Cavern) > 0;
	bool hasUltraAll	= UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Ultralisk_Cavern) > 0;
	bool hasGreaterSpire = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Greater_Spire) > 0;
	bool hasGreaterSpireAll = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Greater_Spire) > 0;
	bool hasQueensNest	= UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Queens_Nest) > 0;
	bool hasQueensNestAll = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Queens_Nest) > 0;

	// Unit stuff. This includes uncompleted units.
	int nDrones			= UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Drone);
	int nGasDrones		= WorkerManager::Instance().getNumGasWorkers();
	int nMinDrones		= WorkerManager::Instance().getNumMineralWorkers();
	int nLings			= UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Zergling);
	int nHydras			= UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Hydralisk);
	int nHydrasComp		= UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Hydralisk);
	int nLurkers		= UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Lurker);
	int nMutas			= UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Mutalisk);
	int nScourge		= UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Scourge);
    int nGuardians		= UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Guardian);
	int nDevourers		= UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Devourer);
	int nUltras			= UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Ultralisk);

	// hasLairTech means "can research stuff in the lair" (not "can research stuff that needs lair").
	bool hasHive		= UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Hive) > 0;
	bool hasHiveTech	= UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Hive) > 0;
	bool hasLair		= UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Lair) > 0;
	bool hasLairTech	= hasLair || nHives > 0;

	BWAPI::UpgradeType melee = BWAPI::UpgradeTypes::Zerg_Melee_Attacks;
	BWAPI::UpgradeType missile = BWAPI::UpgradeTypes::Zerg_Missile_Attacks;
	BWAPI::UpgradeType armor = BWAPI::UpgradeTypes::Zerg_Carapace;
	BWAPI::UpgradeType airAttack = BWAPI::UpgradeTypes::Zerg_Flyer_Attacks;
	BWAPI::UpgradeType airArmor = BWAPI::UpgradeTypes::Zerg_Flyer_Carapace;
	BWAPI::TechType lurkerAspect = BWAPI::TechTypes::Lurker_Aspect;
	int meleeUps		= _self->getUpgradeLevel(melee);
	int missileUps		= _self->getUpgradeLevel(missile);
	int armorUps		= _self->getUpgradeLevel(armor);
	int airarmorUps		= _self->getUpgradeLevel(airArmor);

	// minimum drones to mine minerals at all
	int droneMin = 1 + (3 * nGas);
	// maximum drones must never fall to 0!
	int droneMax = maxDrones();
	// global drone eco
	double droneEco = (double)(nDrones * 2) / (double)(nSupplyTotal + 1);
	_droneEco = droneEco;

	// compensate for ingame min / gas counter, counting 16 each frame
	//nLarvas = std::max(0, nLarvas - 1);
	if (nMinDrones > 9)
	{
		minerals = std::max(0, minerals - 16);
	}
	if (nGasDrones > 3)
	{
		gas = std::max(0, gas - 16);
	}

	// emegency
	int ourForce = InformationManager::Instance().getPower(_self);
	int enemyForce = InformationManager::Instance().getPower(_enemy);
	int ourForceGround = InformationManager::Instance().getMyPowerGroundWeapon();
	int enemyForceGround = InformationManager::Instance().getEnemyPowerGroundUnits();
	int enemyForceGroundNearby = InformationManager::Instance().getEnemyPowerGroundUnitsIncoming();
	bool emergencyGroundDefense = StrategyBossZerg::Instance().emergencyGroundDefense();

	// enemy has air
	//bool enemyHasAirTech = InformationManager::Instance().enemyHasAirTech();

	// Decide gas unit
	chooseGasUnit();
	// Decide techTarget
	chooseTechTarget();

	// Decide unit mix
	bool makeDrones = false;
	bool makeLings = false;
	bool makeHydras = false;
	bool makeMutas = false;
	bool makeUltras = false;
	bool makeLurkers = false;
	
	// Lurkers
	if ((_gasUnit == "Lurker")) { makeLurkers = true; }

	// Mutas
	if ((_gasUnit == "Mutalisk")) { makeMutas = true; }

	// Hydras
	if ((_gasUnit == "Hydralisk")) { makeHydras = true; }

	// Ultras
	// Do not produce ultras if we are low on resources
	if ((minerals > 125 && gas > 125) && nLarvas > 1 && hasUltra && makeLurkers) { makeUltras = true; }
	if ((minerals > 100 && gas > 100) && nLarvas > 1 && hasUltra && makeMutas) { makeUltras = true; }
	if ((minerals > 75 && gas > 25) && nLarvas > 1 && hasUltra && makeHydras) { makeUltras = true; }
	if ((minerals > 200 && gas > 200) && nLarvas > 1 && hasUltra && !makeMutas && !makeHydras) { makeUltras = true; }
	if (hasUltra && !makeHydras && !makeMutas && !makeLurkers) { makeUltras = true; }

	// Lings
	// Do not produce lings if we are low on resources
	//if (minerals > 125 && hasPool && nLarvas > 1 && makeLurkers) { makeLings = true; }
	//if (minerals > 100 && hasPool && nLarvas > 1 && makeMutas) { makeLings = true; }
	//if (minerals > 75 && hasPool && nLarvas > 1 && makeHydras) { makeLings = true; }
	//if (minerals > 200 && hasPool && nLarvas > 1 && makeUltras) { makeLings = true; }
	//if (hasPool && !hasSpire && !hasDen && !hasUltra) { makeLings = true; }
	//if (hasPool && !makeHydras && !makeMutas && !makeLurkers) { makeLings = true; }

	// Make drones?
	if (ourForce >= enemyForce) { makeDrones = true; }
	if (ourForce < enemyForce && nDrones < nMinPatches) { makeDrones = true; }
	if (ourForce < 6) { makeDrones = false; }
	if (emergencyGroundDefense) { makeDrones = false; }
	if (minerals > 800 && nHatches > 0) { makeDrones = false; }

	// Decide mineral unit
	if (makeDrones && nDrones < droneMax)
	{
		_minUnit = "Drone";
		makeLings = false;
		makeDrones = true;
	}
	else
	{
		_minUnit = "Zergling";
		makeLings = true;
		makeDrones = false;
	}		

	int larvasLeft = nLarvas;
	int mineralsLeft = minerals;
	int gasLeft = gas;
	int supplyLeft = nSupplyTotal - nSupplyUsed;

	
	// STRATEGY CODE

	if (Config::Strategy::StrategyName != "3HatchHydra BGH" && 
		Config::Strategy::StrategyName != "5Pool Zergling Hell")
	{
		// UNITS

		// Drones
		if (makeDrones && nDrones < droneMax) {
			//for (int i = 0; i < std::max(9, droneMax - nDrones); ++i)
			//{
				//if (larvasLeft > 0 && mineralsLeft > 0 && supplyLeft > 0) {
					queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Drone));
					--larvasLeft;
					mineralsLeft -= 50;
					//supplyLeft -= 2;
				//}
			//}
		}

		// Lurkers
		if (hasDen && hasLurker && makeLurkers && nHydrasComp > 0 && nLurkers < 2 * nHydras &&
			mineralsLeft > 0 && gasLeft > 0 && supplyLeft > 3) {
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Lurker));
			mineralsLeft -= 125;
			gasLeft -= 125;
			//supplyLeft -= 4;
		}

		// Hydralisks
		else if (hasDen && (makeHydras || makeLurkers) && larvasLeft > 0 && 
			mineralsLeft > 0 && gasLeft > 0 && supplyLeft > 1) {
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Hydralisk));
			--larvasLeft;
			mineralsLeft -= 75;
			gasLeft -= 25;
			//supplyLeft -= 2;
		}

		// Mutalisks
		if (hasSpire && makeMutas && larvasLeft > 0 && mineralsLeft > 0 && gasLeft > 0 && supplyLeft > 3) {
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Mutalisk));
			--larvasLeft;
			mineralsLeft -= 100;
			gasLeft -= 100;
			//supplyLeft -= 4;
		}

		// Ultralisks
		if (hasUltra && makeUltras && larvasLeft > 0 && mineralsLeft > 0 && gasLeft > 0 && supplyLeft > 3) {
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Ultralisk));
			--larvasLeft;
			mineralsLeft -= 200;
			gasLeft -= 200;
			//supplyLeft -= 4;
		}

		// Zerglings
		if (hasPool && makeLings && larvasLeft > 0 && mineralsLeft > 0 && supplyLeft > 0) {
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Zergling));
			--larvasLeft;
			mineralsLeft -= 50;
			//supplyLeft -= 2;
			//BWAPI::Broodwar->printf("Lings!");
		}

		// Drones
		if (makeDrones && nDrones < droneMax && larvasLeft > 0 && mineralsLeft > 0 && supplyLeft > 0) {
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Drone));
		}

		// BUILDINGS

		// Hydralisks
		if (_techTarget == "Hydralisks" && hasPool && !hasDenAll && nDrones >= 12 && nGas > 0 &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Hydralisk_Den) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Hydralisk_Den)) 
		{
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Hydralisk_Den));
		}

		// Lurkers
		if (_techTarget == "Lurkers" && hasPool && !hasDenAll && nDrones >= 9 && nGas > 0 && 
			(nLairsAll > 0 || BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Lair)) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Hydralisk_Den) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Hydralisk_Den)) 
		{
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Hydralisk_Den));
		}

		// Mutalisks
		if (_techTarget == "Mutalisks" && !hasSpireAll && hasLairTech && nDrones >= 9 && nGas > 0 &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Spire) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Spire))
		{
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Spire));
		}

		// Ultralisks
		if (_techTarget == "Ultralisks" && !hasUltraAll && hasHiveTech && nDrones >= 0.7 * droneMax && nGas >= 3 &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Ultralisk_Cavern) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Ultralisk_Cavern))
		{
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Ultralisk_Cavern));
		}

		// Zerglings
		if (!hasPoolAll &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Spawning_Pool) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Spawning_Pool))
		{
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Spawning_Pool));
			// If we're low on drones, replace the drone.
			if (nDrones <= 9) {
				queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Drone));
			}
		}

		// Make lair early for mutalisks. 
		if ((_techTarget == "Mutalisks") && hasPool && 
			(nLairsAll + nHivesAll == 0) && nGas > 0 && nDrones >= 9 &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Lair) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Lair))
		{
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Lair));
		}
		// Make lair early for lurkers. 
		else if (_techTarget == "Lurkers" && hasPool &&
			(nLairsAll + nHivesAll == 0) && nGas > 0 && nDrones >= 9 &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Lair) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Lair))
		{
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Lair));
		}
		// Make lair later if we want it for ultras.
		else if ((_techTarget == "Ultralisks") && hasPool && 
			(nLairsAll + nHivesAll == 0) && nGas >= 3 && nDrones >= 0.5 * droneMax &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Lair) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Lair))
		{
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Lair));
		}
		// Make lair for upgrades.
		else if ((armorUps == 1 || meleeUps == 1 || airarmorUps == 1) && 
			hasPool && (nLairsAll + nHivesAll == 0) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Lair) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Lair))
		{
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Lair));
		}

		// Make a queen's nest for ultras.
		if (_techTarget == "Ultralisks" && !hasQueensNestAll && 
			hasLair && nGas >= 3 && nDrones >= 0.6 * droneMax &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Queens_Nest) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Queens_Nest))
		{
			// Get some intermediate units before moving toward hive.
			if (nHydras >= 8){
				queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Queens_Nest));
			}
			else if (nMutas >= 6){
				queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Queens_Nest));
			}
		}
		// Make a queen's nest for upgrades.
		else if ((armorUps == 2 || meleeUps == 2 || airarmorUps == 2) && 
			!hasQueensNestAll && hasLair && 
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Queens_Nest) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Queens_Nest))
		{
			// Get some intermediate units before moving toward hive.
			if (nHydras >= 8){
				queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Queens_Nest));
			}
			else if (nMutas >= 6){
				queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Queens_Nest));
			}
		}

		// Make a hive for Ultralisks.
		if ((_techTarget == "Ultralisks") && 
			hasQueensNest && hasLair && nHivesAll == 0 && nDrones >= 0.7 * droneMax && nGas >= 3 &&
			!_self->isUpgrading(BWAPI::UpgradeTypes::Pneumatized_Carapace) &&
			!_self->isUpgrading(BWAPI::UpgradeTypes::Ventral_Sacs) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Pneumatized_Carapace) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Ventral_Sacs) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Hive) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Hive))
		{
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Hive));
		}
		// Make a hive for upgrades.
		else if ((armorUps == 2 || meleeUps == 2 || airarmorUps == 2) &&
			hasQueensNest && hasLair && nHivesAll == 0 &&
			!_self->isUpgrading(BWAPI::UpgradeTypes::Pneumatized_Carapace) &&
			!_self->isUpgrading(BWAPI::UpgradeTypes::Ventral_Sacs) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Pneumatized_Carapace) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Ventral_Sacs) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Hive) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Hive))
		{
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Hive));
		}

		// Prepare an evo chamber or two.
		if (hasPool && nGas > 0 && minerals > 100 && gas > 100 &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Evolution_Chamber) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Evolution_Chamber))
		{
			if ((nEvo == 0 && nDrones >= 18 && (_enemyRace != BWAPI::Races::Terran || hasDen ||  hasSpire || hasUltra)) ||
				(nEvo == 1 && nDrones >= 24 && (hasDen || hasUltra) && minerals > 200 && gas > 200 && nGas > 0 &&
				_self->isUpgrading(armor) &&
				(hasDen || hasSpire || hasUltra)))
			{
				queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Evolution_Chamber));
			}
		}

		// UPGRADES AND TECH

		// Decide about lurker aspect tech.
		if (hasDen && hasLair && _techTarget == "Lurkers" && minerals > 50 && gas > 50)
		{
			if (!_self->hasResearched(BWAPI::TechTypes::Lurker_Aspect) &&
				!_self->isResearching(BWAPI::TechTypes::Lurker_Aspect) &&
				!_self->isUpgrading(BWAPI::UpgradeTypes::Muscular_Augments) &&
				!_self->isUpgrading(BWAPI::UpgradeTypes::Grooved_Spines) &&
				!queue.anyInQueue(BWAPI::TechTypes::Lurker_Aspect) &&
				!queue.anyInQueue(BWAPI::UpgradeTypes::Muscular_Augments) &&
				!queue.anyInQueue(BWAPI::UpgradeTypes::Grooved_Spines))
			{
				queue.queueAsLowestPriority(MacroAct(BWAPI::TechTypes::Lurker_Aspect));
			}
		}

		// Decide about hydralisk upgrades.
		else if (hasDen && makeHydras && nHydras >= 3 && nDrones >= 12 && minerals > 100 && gas > 100)
		{
			if (_self->getUpgradeLevel(BWAPI::UpgradeTypes::Muscular_Augments) == 0 &&
				!_self->isUpgrading(BWAPI::UpgradeTypes::Muscular_Augments) &&
				!_self->isUpgrading(BWAPI::UpgradeTypes::Grooved_Spines) &&
				!_self->isResearching(BWAPI::TechTypes::Lurker_Aspect) &&
				!queue.anyInQueue(BWAPI::UpgradeTypes::Muscular_Augments)&&
				!queue.anyInQueue(BWAPI::TechTypes::Lurker_Aspect))
			{
				queue.queueAsLowestPriority(MacroAct(BWAPI::UpgradeTypes::Muscular_Augments));
			}
			else if (_self->getUpgradeLevel(BWAPI::UpgradeTypes::Muscular_Augments) == 1 &&
				_self->getUpgradeLevel(BWAPI::UpgradeTypes::Grooved_Spines) == 0 &&
				!_self->isUpgrading(BWAPI::UpgradeTypes::Grooved_Spines) &&
				!_self->isUpgrading(BWAPI::UpgradeTypes::Muscular_Augments) &&
				!_self->isResearching(BWAPI::TechTypes::Lurker_Aspect) &&
				!queue.anyInQueue(BWAPI::UpgradeTypes::Grooved_Spines) &&
				!queue.anyInQueue(BWAPI::TechTypes::Lurker_Aspect))
			{
				queue.queueAsLowestPriority(MacroAct(BWAPI::UpgradeTypes::Grooved_Spines));
			}
		}

		// Decide about zergling upgrades
		if (hasPool && makeLings && minerals > 100 && gas > 100){
			if (nDrones >= 9 && nLings > 6 &&
				_self->getUpgradeLevel(BWAPI::UpgradeTypes::Metabolic_Boost) == 0 &&
				!_self->isUpgrading(BWAPI::UpgradeTypes::Metabolic_Boost) &&
				!queue.anyInQueue(BWAPI::UpgradeTypes::Metabolic_Boost))
			{
				queue.queueAsLowestPriority(MacroAct(BWAPI::UpgradeTypes::Metabolic_Boost));
			}
			else if (hasHiveTech && nDrones >= 12 && nLings > 8 &&
				_self->getUpgradeLevel(BWAPI::UpgradeTypes::Metabolic_Boost) == 1 &&
				_self->getUpgradeLevel(BWAPI::UpgradeTypes::Adrenal_Glands) == 0 &&
				!_self->isUpgrading(BWAPI::UpgradeTypes::Adrenal_Glands) &&
				!queue.anyInQueue(BWAPI::UpgradeTypes::Adrenal_Glands))
			{
				queue.queueAsLowestPriority(MacroAct(BWAPI::UpgradeTypes::Adrenal_Glands));
			}
		}

		// Decide about flyer armor upgrades.
		if (hasSpire && makeMutas && nMutas > 3 && nDrones >= 13 && minerals > 150 && gas > 150 &&
			!_self->isUpgrading(airArmor) && !queue.anyInQueue(airArmor))
		{
			if (airarmorUps == 0 ||
				airarmorUps == 1 && hasLairTech ||
				airarmorUps == 2 && hasHiveTech) {
				queue.queueAsLowestPriority(MacroAct(airArmor));
			}
		}
		
		// Decide about ultralisk upgrades.
		if (hasUltra && makeUltras && nUltras > 3 && nDrones >= 0.8 * droneMax && 
			nGas >= 3 && minerals > 150 && gas > 150)
		{
			if (_self->getUpgradeLevel(BWAPI::UpgradeTypes::Anabolic_Synthesis) == 0 &&
				!_self->isUpgrading(BWAPI::UpgradeTypes::Anabolic_Synthesis) &&
				!queue.anyInQueue(BWAPI::UpgradeTypes::Anabolic_Synthesis))
			{
				queue.queueAsLowestPriority(MacroAct(BWAPI::UpgradeTypes::Anabolic_Synthesis));
			}
			else if (_self->getUpgradeLevel(BWAPI::UpgradeTypes::Anabolic_Synthesis) != 0 &&
				_self->getUpgradeLevel(BWAPI::UpgradeTypes::Chitinous_Plating) == 0 &&
				!_self->isUpgrading(BWAPI::UpgradeTypes::Chitinous_Plating) &&
				!queue.anyInQueue(BWAPI::UpgradeTypes::Chitinous_Plating))
			{
				queue.queueAsLowestPriority(MacroAct(BWAPI::UpgradeTypes::Chitinous_Plating));
			}
		}

		// Decide about ground armor upgrades.
		if (hasPool && nEvo >= 1 && nDrones >= 12 && nGas > 0 && minerals > 150 && gas > 150 &&
			!_self->isUpgrading(armor) && !queue.anyInQueue(armor))
		{
			// But delay if we're going mutas and don't have many yet. They want the gas.
			if (!(hasSpire && makeMutas && gas < 600 && nMutas < 6))
			{
				if (armorUps == 0 ||
					armorUps == 1 && hasLairTech ||
					armorUps == 2 && hasHiveTech) 
				{
					queue.queueAsLowestPriority(MacroAct(armor));
				}
			}
		}

		// Decide about ground missile upgrades.
		if ((nEvo >= 2 || nEvo >= 1 && armorUps == 3) && nDrones >= 14 && nGas > 0 && minerals > 150 && gas > 150 &&
			hasPool && hasDen && !_self->isUpgrading(missile) && !queue.anyInQueue(missile))
		{
			// But delay if we're going mutas and don't have many yet. They want the gas.
			if (!(hasSpire && makeMutas && gas < 600 && nMutas < 6))
			{
				if (missileUps == 0 ||
					missileUps == 1 && hasLairTech ||
					missileUps == 2 && hasHiveTech)
				{
					queue.queueAsLowestPriority(MacroAct(missile));
				}
			}
		}

		// Decide about ground melee upgrades.
		if (nEvo >= 2 && armorUps == 3 && nDrones >= 16 && nGas >= 2 && minerals > 150 && gas > 150 &&
			hasPool && hasUltra && !_self->isUpgrading(melee) && !queue.anyInQueue(melee) &&
			!_self->isUpgrading(missile) && !queue.anyInQueue(missile))
		{
			// But delay if we're going mutas and don't have many yet. They want the gas.
			if (!(hasSpire && makeMutas && gas < 600 && nMutas < 6))
			{
				if (meleeUps == 0 ||
					meleeUps == 1 && hasLairTech ||
					meleeUps == 2 && hasHiveTech)
				{
					queue.queueAsLowestPriority(MacroAct(melee));
				}
			}
		}
	}
	else if (Config::Strategy::StrategyName == "3HatchHydra BGH")
	{
		BWAPI::Broodwar->printf("BHG hydralisk macro");
		Config::Macro::WorkersPerPatch = 1.6;
		droneMax = maxDrones();

		// UNITS

		// Drones
		if (makeDrones && nDrones < droneMax) {
			if (larvasLeft > 0 && mineralsLeft > 0 && supplyLeft > 0) {
				queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Drone));
			}
		}

		// Hydralisks
		if (hasDen && larvasLeft > 0 && mineralsLeft > 0 && gasLeft > 0 && supplyLeft > 0) {
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Hydralisk));
		}

		// Drones
		if (makeDrones && nDrones < droneMax && larvasLeft > 0 && mineralsLeft > 0 && supplyLeft > 0) {
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Drone));
		}

		// BUILDINGS

		// Zerglings
		if (!hasPoolAll &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Spawning_Pool) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Spawning_Pool))
		{
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Spawning_Pool));
			// If we're low on drones, replace the drone.
			if (nDrones <= 9) {
				queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Drone));
			}
		}

		// Hydralisks
		if (hasPool && !hasDenAll && nDrones >= 13 && nGas > 0 &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Hydralisk_Den) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Hydralisk_Den))
		{
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Hydralisk_Den));
		}

		// Make a lair.
		if ((missileUps > 0 && armorUps > 0) &&
			hasPool && (nLairsAll + nHivesAll == 0) && nGas > 0 && nDrones >= 26 &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Lair) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Lair))
		{
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Lair));
		}

		// Make a queen's nest.
		if (!hasQueensNestAll && hasLair && nGas >= 3 && nDrones >= 26 &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Queens_Nest) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Queens_Nest))
		{
			// Get some intermediate units before moving toward hive.
			if (nHydras >= 12){
				queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Queens_Nest));
			}
			else if (nMutas >= 6){
				queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Queens_Nest));
			}
		}

		// Make a hive.
		if ((missileUps == 2 && armorUps == 2) && hasQueensNest &&
			hasLair && nHivesAll == 0 && nDrones >= 26 && nGas >= 3 &&
			!_self->isUpgrading(BWAPI::UpgradeTypes::Pneumatized_Carapace) &&
			!_self->isUpgrading(BWAPI::UpgradeTypes::Ventral_Sacs) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Pneumatized_Carapace) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Ventral_Sacs) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Hive) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Hive))
		{
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Hive));
		}

		// Prepare an evo chamber or two.
		if (hasPool && nGas > 0 && minerals > 100 && gas > 100 &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Evolution_Chamber) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Evolution_Chamber))
		{
			if ((nEvo == 0 && nDrones >= 18 && hasDen) ||
				(nEvo == 1 && nDrones >= 24 && hasDen && minerals > 200 && gas > 200 && nGas > 0 &&
				_self->isUpgrading(missile)))
			{
				queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Evolution_Chamber));
			}
		}

		// UPGRADES AND TECH

		// Decide about hydralisk upgrades.
		if (hasDen && nDrones >= 13 && minerals > 100 && gas > 100)
		{
			if (_self->getUpgradeLevel(BWAPI::UpgradeTypes::Muscular_Augments) == 0 &&
				!_self->isUpgrading(BWAPI::UpgradeTypes::Muscular_Augments) &&
				!queue.anyInQueue(BWAPI::UpgradeTypes::Muscular_Augments))
			{
				queue.queueAsLowestPriority(MacroAct(BWAPI::UpgradeTypes::Muscular_Augments));
			}
			else if (_self->getUpgradeLevel(BWAPI::UpgradeTypes::Muscular_Augments) == 1 &&
				_self->getUpgradeLevel(BWAPI::UpgradeTypes::Grooved_Spines) == 0 &&
				!_self->isUpgrading(BWAPI::UpgradeTypes::Grooved_Spines) &&
				!queue.anyInQueue(BWAPI::UpgradeTypes::Grooved_Spines))
			{
				queue.queueAsLowestPriority(MacroAct(BWAPI::UpgradeTypes::Grooved_Spines));
			}
		}

		// Decide about ground missile upgrades.
		if (nEvo >= 1 && nDrones >= 12 && nGas > 0 && minerals > 150 && gas > 150 &&
			hasPool && hasDen && !_self->isUpgrading(missile) && !queue.anyInQueue(missile))
		{
			// But delay if we're going mutas and don't have many yet. They want the gas.
			if (!(hasSpire && makeMutas && gas < 600 && nMutas < 6))
			{
				if (missileUps == 0 ||
					missileUps == 1 && hasLairTech ||
					missileUps == 2 && hasHiveTech)
				{
					queue.queueAsLowestPriority(MacroAct(missile));
				}
			}
		}

		// Decide about ground armor upgrades.
		if ((nEvo >= 2 || nEvo >= 1 && missileUps == 3) && nDrones >= 16 && nGas > 0 && minerals > 150 && gas > 150 &&
			!_self->isUpgrading(armor) && !queue.anyInQueue(armor))
		{
			// But delay if we're going mutas and don't have many yet. They want the gas.
			if (!(hasSpire && makeMutas && gas < 600 && nMutas < 6))
			{
				if (armorUps == 0 ||
					armorUps == 1 && hasLairTech ||
					armorUps == 2 && hasHiveTech)
				{
					queue.queueAsLowestPriority(MacroAct(armor));
				}
			}
		}

	}
	else
	{
		// Zergling strategy
		BWAPI::Broodwar->printf("Zergling macro");

		// UNITS

		// Drones
		if (makeDrones && nDrones < droneMax) {
			if (larvasLeft > 0 && mineralsLeft > 0 && supplyLeft > 0) {
				queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Drone));
			}
		}

		// Zerglings
		if (hasPool && larvasLeft > 0 && mineralsLeft > 0 && supplyLeft > 0) {
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Zergling));
		}

		// Drones
		if (makeDrones && nDrones < droneMax && larvasLeft > 0 && mineralsLeft > 0 && supplyLeft > 0) {
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Drone));
		}

		// BUILDINGS

		// Zerglings
		if (!hasPoolAll &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Spawning_Pool) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Spawning_Pool))
		{
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Spawning_Pool));
			// If we're low on drones, replace the drone.
			if (nDrones <= 9) {
				queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Drone));
			}
		}

		// Make a lair.
		if ((meleeUps > 0 && armorUps > 0) &&
			hasPool && (nLairsAll + nHivesAll == 0) && nGas > 0 && nDrones >= 26 &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Lair) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Lair))
		{
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Lair));
		}

		// Make a queen's nest.
		if (!hasQueensNestAll && hasLair && nGas >= 3 && nDrones >= 26 &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Queens_Nest) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Queens_Nest))
		{
			// Get some intermediate units before moving toward hive.
			if (nHydras >= 12){
				queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Queens_Nest));
			}
			else if (nMutas >= 6){
				queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Queens_Nest));
			}
		}

		// Make a hive.
		if ((meleeUps == 2 && armorUps == 2) && hasQueensNest &&
			hasLair && nHivesAll == 0 && nDrones >= 26 && nGas >= 3 &&
			!_self->isUpgrading(BWAPI::UpgradeTypes::Pneumatized_Carapace) &&
			!_self->isUpgrading(BWAPI::UpgradeTypes::Ventral_Sacs) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Pneumatized_Carapace) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Ventral_Sacs) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Hive) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Hive))
		{
			queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Hive));
		}

		// Prepare an evo chamber or two.
		if (hasPool && nGas > 0 && minerals > 100 && gas > 100 &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Zerg_Evolution_Chamber) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Zerg_Evolution_Chamber))
		{
			if ((nEvo == 0 && nDrones >= 18) ||
				(nEvo == 1 && nDrones >= 24 && minerals > 200 && gas > 200 && nGas > 0 && _self->isUpgrading(melee)))
			{
				queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Zerg_Evolution_Chamber));
			}
		}

		// UPGRADES AND TECH

		// Decide about zergling upgrades
		if (hasPool && minerals > 100 && gas > 100){
			if (nDrones >= 9 && nLings > 6 &&
				_self->getUpgradeLevel(BWAPI::UpgradeTypes::Metabolic_Boost) == 0 &&
				!_self->isUpgrading(BWAPI::UpgradeTypes::Metabolic_Boost) &&
				!queue.anyInQueue(BWAPI::UpgradeTypes::Metabolic_Boost))
			{
				queue.queueAsLowestPriority(MacroAct(BWAPI::UpgradeTypes::Metabolic_Boost));
			}
			else if (hasHiveTech && nDrones >= 12 && nLings > 8 &&
				_self->getUpgradeLevel(BWAPI::UpgradeTypes::Metabolic_Boost) == 1 &&
				_self->getUpgradeLevel(BWAPI::UpgradeTypes::Adrenal_Glands) == 0 &&
				!_self->isUpgrading(BWAPI::UpgradeTypes::Adrenal_Glands) &&
				!queue.anyInQueue(BWAPI::UpgradeTypes::Adrenal_Glands))
			{
				queue.queueAsLowestPriority(MacroAct(BWAPI::UpgradeTypes::Adrenal_Glands));
			}
		}

		// Decide about ground melee upgrades.
		if (nEvo >= 1 && nDrones >= 12 && nGas > 0 && minerals > 150 && gas > 150 &&
			hasPool && !_self->isUpgrading(melee) && !queue.anyInQueue(melee))
		{
			// But delay if we're going mutas and don't have many yet. They want the gas.
			if (!(hasSpire && makeMutas && gas < 600 && nMutas < 6))
			{
				if (meleeUps == 0 ||
					meleeUps == 1 && hasLairTech ||
					meleeUps == 2 && hasHiveTech)
				{
					queue.queueAsLowestPriority(MacroAct(melee));
				}
			}
		}

		// Decide about ground armor upgrades.
		if ((nEvo >= 2 || nEvo >= 1 && melee == 3) && nDrones >= 16 && nGas > 0 && minerals > 150 && gas > 150 &&
			!_self->isUpgrading(armor) && !queue.anyInQueue(armor))
		{
			// But delay if we're going mutas and don't have many yet. They want the gas.
			if (!(hasSpire && makeMutas && gas < 600 && nMutas < 6))
			{
				if (armorUps == 0 ||
					armorUps == 1 && hasLairTech ||
					armorUps == 2 && hasHiveTech)
				{
					queue.queueAsLowestPriority(MacroAct(armor));
				}
			}
		}

	}

}

void StrategyManager::readResults()
{
    if (!Config::Modules::UsingStrategyIO)
    {
        return;
    }

    std::string enemyName = BWAPI::Broodwar->enemy()->getName();
    std::replace(enemyName.begin(), enemyName.end(), ' ', '_');

    std::string enemyResultsFile = Config::Strategy::ReadDir + enemyName + ".txt";
    
    std::string strategyName;
    int wins = 0;
    int losses = 0;

    FILE *file = fopen ( enemyResultsFile.c_str(), "r" );
    if ( file != nullptr )
    {
        char line [ 4096 ]; /* or other suitable maximum line size */
        while ( fgets ( line, sizeof line, file ) != nullptr ) /* read a line */
        {
            std::stringstream ss(line);

            ss >> strategyName;
            ss >> wins;
            ss >> losses;

            //BWAPI::Broodwar->printf("Results Found: %s %d %d", strategyName.c_str(), wins, losses);

            if (_strategies.find(strategyName) == _strategies.end())
            {
                //BWAPI::Broodwar->printf("Warning: Results file has unknown Strategy: %s", strategyName.c_str());
            }
            else
            {
                _strategies[strategyName]._wins = wins;
                _strategies[strategyName]._losses = losses;
            }
        }

        fclose ( file );
    }
    else
    {
        //BWAPI::Broodwar->printf("No results file found: %s", enemyResultsFile.c_str());
    }
}

void StrategyManager::writeResults()
{
    if (!Config::Modules::UsingStrategyIO)
    {
        return;
    }

    std::string enemyName = BWAPI::Broodwar->enemy()->getName();
    std::replace(enemyName.begin(), enemyName.end(), ' ', '_');

    std::string enemyResultsFile = Config::Strategy::WriteDir + enemyName + ".txt";

    std::stringstream ss;

    for (auto & kv : _strategies)
    {
        const Strategy & strategy = kv.second;

        ss << strategy._name << " " << strategy._wins << " " << strategy._losses << "\n";
    }

    Logger::LogOverwriteToFile(enemyResultsFile, ss.str());
}

void StrategyManager::onEnd(const bool isWinner)
{
    if (!Config::Modules::UsingStrategyIO)
    {
        return;
    }

    if (isWinner)
    {
        _strategies[Config::Strategy::StrategyName]._wins++;
    }
    else
    {
        _strategies[Config::Strategy::StrategyName]._losses++;
    }

    writeResults();
}

void StrategyManager::setLearnedStrategy()
{
    // we are currently not using this functionality for the competition so turn it off 
    //return;

    if (!Config::Modules::UsingStrategyIO)
    {
        return;
    }

    const std::string & strategyName = Config::Strategy::StrategyName;
    Strategy & currentStrategy = _strategies[strategyName];

    int totalGamesPlayed = 0;
    int strategyGamesPlayed = currentStrategy._wins + currentStrategy._losses;
    double winRate = strategyGamesPlayed > 0 ? currentStrategy._wins / static_cast<double>(strategyGamesPlayed) : 0;

    // if we are using an enemy specific strategy
    if (Config::Strategy::FoundEnemySpecificStrategy)
    {        
        return;
    }

    // if our win rate with the current strategy is super high don't explore at all
    // also we're pretty confident in our base strategies so don't change if insufficient games have been played
    if (strategyGamesPlayed < 5 || (strategyGamesPlayed > 0 && winRate > 0.49))
    {
		BWAPI::Broodwar->sendText("Using default strategy selection: games = %d  winrate = %.3lf", strategyGamesPlayed, winRate);
        return;
    }

    // get the total number of games played so far with this race
    for (auto & kv : _strategies)
    {
        Strategy & strategy = kv.second;
		if (strategy._race == _selfRace)
        {
            totalGamesPlayed += strategy._wins + strategy._losses;
        }
    }

    // calculate the UCB value and store the highest
    double C = 0.5;
    std::string bestUCBStrategy;
    double bestUCBStrategyVal = std::numeric_limits<double>::lowest();
	double bestUCBWinRate = std::numeric_limits<double>::lowest();
	int bestUCBStrategyGames = 0;
    for (auto & kv : _strategies)
    {
		// Do not consider strategies for another race
		Strategy & strategy = kv.second;
		if (strategy._race != _selfRace)
        {
            continue;
        }

		// Do not consider strategies that we have not played
		int sGamesPlayed = strategy._wins + strategy._losses;
		if (sGamesPlayed == 0)
		{
			continue;
		}

		double sWinRate = sGamesPlayed > 0 ? strategy._wins / static_cast<double>(sGamesPlayed) : 0;
        double ucbVal = C * sqrt( log( (double)totalGamesPlayed / sGamesPlayed ) );
        double val = sWinRate + ucbVal;

        if (val > bestUCBStrategyVal)
        {
            bestUCBStrategy = strategy._name;
            bestUCBStrategyVal = val;
			bestUCBStrategyGames = sGamesPlayed;
			bestUCBWinRate = sWinRate;
        }
    }

	BWAPI::Broodwar->sendText("Adapting strategy selection: usb = %.3lf  games = %d  winrate = %.3lf", bestUCBStrategyVal, bestUCBStrategyGames, bestUCBWinRate);
    Config::Strategy::StrategyName = bestUCBStrategy;
}

void StrategyManager::handleUrgentProductionIssues(BuildOrderQueue & queue)
{
	if (_selfRace == BWAPI::Races::Zerg)
	{
		StrategyBossZerg::Instance().handleUrgentProductionIssues(queue);
	}
	else
	{
		// detect if there's a supply block once per second
		if ((BWAPI::Broodwar->getFrameCount() % 24 == 1) && detectSupplyBlock(queue))
		{
			if (Config::Debug::DrawBuildOrderSearchInfo)
			{
				BWAPI::Broodwar->printf("Supply block, building supply!");
			}

			queue.queueAsHighestPriority(MacroAct(BWAPI::Broodwar->self()->getRace().getSupplyProvider()));
		}

		// If they have mobile cloaked units, get some static detection.
		if (InformationManager::Instance().enemyHasMobileCloakTech())
		{
			if (BWAPI::Broodwar->self()->getRace() == BWAPI::Races::Protoss)
			{
				if (BWAPI::Broodwar->self()->allUnitCount(BWAPI::UnitTypes::Protoss_Photon_Cannon) < 2)
				{
					queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Protoss_Photon_Cannon));
					queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Protoss_Photon_Cannon));
				}
				if (BWAPI::Broodwar->self()->allUnitCount(BWAPI::UnitTypes::Protoss_Forge) == 0)
				{
					queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Protoss_Forge));
				}
			}
			else if (BWAPI::Broodwar->self()->getRace() == BWAPI::Races::Terran)
			{
				if (BWAPI::Broodwar->self()->allUnitCount(BWAPI::UnitTypes::Terran_Missile_Turret) < 2)
				{
					queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));
					queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));
				}
				if (BWAPI::Broodwar->self()->allUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) == 0)
				{
					queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Engineering_Bay));
				}
			}

			if (Config::Debug::DrawBuildOrderSearchInfo)
			{
				BWAPI::Broodwar->printf("Enemy has cloaking tech!");
			}
		}
	}
}

// Return true if we're supply blocked and should build supply.
// Note: This understands zerg supply but is not used when we are zerg.
bool StrategyManager::detectSupplyBlock(BuildOrderQueue & queue)
{
	// If the _queue is empty or supply is maxed, there is no block.
	if (queue.isEmpty() || BWAPI::Broodwar->self()->supplyTotal() >= 400)
	{
		return false;
	}

	// If supply is being built now, there's no block. Return right away.
	// Terran and protoss calculation:
	if (BuildingManager::Instance().isBeingBuilt(BWAPI::Broodwar->self()->getRace().getSupplyProvider()))
	{
		return false;
	}

	// Terran and protoss calculation:
	int supplyAvailable = BWAPI::Broodwar->self()->supplyTotal() - BWAPI::Broodwar->self()->supplyUsed();

	// Zerg calculation:
	// Zerg can create an overlord that doesn't count toward supply until the next check.
	// To work around it, add up the supply by hand, including hatcheries.
	if (BWAPI::Broodwar->self()->getRace() == BWAPI::Races::Zerg) {
		supplyAvailable = -BWAPI::Broodwar->self()->supplyUsed();
		for (auto & unit : BWAPI::Broodwar->self()->getUnits())
		{
			if (unit->getType() == BWAPI::UnitTypes::Zerg_Overlord)
			{
				supplyAvailable += 16;
			}
			else if (unit->getType() == BWAPI::UnitTypes::Zerg_Egg &&
				unit->getBuildType() == BWAPI::UnitTypes::Zerg_Overlord)
			{
				return false;    // supply is building, return immediately
				// supplyAvailable += 16;
			}
			else if ((unit->getType() == BWAPI::UnitTypes::Zerg_Hatchery && unit->isCompleted()) ||
				unit->getType() == BWAPI::UnitTypes::Zerg_Lair ||
				unit->getType() == BWAPI::UnitTypes::Zerg_Hive)
			{
				supplyAvailable += 2;
			}
		}
	}

	int supplyCost = queue.getHighestPriorityItem().macroAct.supplyRequired();
	// Available supply can be negative, which breaks the test below. Fix it.
	supplyAvailable = std::max(0, supplyAvailable);

	// if we don't have enough supply, we're supply blocked
	if (supplyAvailable < supplyCost)
	{
		// If we're zerg, check to see if a building is planned to be built.
		// Only count it as releasing supply very early in the game.
		if (BWAPI::Broodwar->self()->getRace() == BWAPI::Races::Zerg
			&& BuildingManager::Instance().buildingsQueued().size() > 0
			&& BWAPI::Broodwar->self()->supplyTotal() <= 18)
		{
			return false;
		}
		return true;
	}

	return false;
}


int StrategyManager::getLingScore()
{
	int score = 0;

	return score;
}

int StrategyManager::getHydraScore()
{
	int score = 0;

	// vs Protoss
	if (_enemyRace == BWAPI::Races::Protoss)
	{
		for (const auto & kv : InformationManager::Instance().getUnitData(_enemy).getUnits())
		{
			const UnitInfo & ui(kv.second);

			if (!ui.type.isWorker() && !ui.type.isBuilding())
			{

				// Can not target ground and spellcasters
				if (ui.type.groundWeapon() == BWAPI::WeaponTypes::None ||
					ui.type == BWAPI::UnitTypes::Protoss_Archon ||
					ui.type == BWAPI::UnitTypes::Protoss_Scout)
				{
					score += ui.type.mineralPrice() + ui.type.gasPrice();
				}
			}

			// Cannons
			else if (ui.type == BWAPI::UnitTypes::Protoss_Photon_Cannon)
			{
				score += ui.type.mineralPrice() + ui.type.gasPrice();
			}
		}
	}

	// vs Terran
	if (_enemyRace == BWAPI::Races::Terran)
	{
		for (const auto & kv : InformationManager::Instance().getUnitData(_enemy).getUnits())
		{
			const UnitInfo & ui(kv.second);

			if (!ui.type.isWorker() && !ui.type.isBuilding())
			{

				// Vulture + Goliath
				if (ui.type == BWAPI::UnitTypes::Terran_Vulture || 
					ui.type == BWAPI::UnitTypes::Terran_Goliath)
				{
					score += ui.type.mineralPrice() + ui.type.gasPrice();
				}
			
				// Air
				if (ui.type == BWAPI::UnitTypes::Terran_Valkyrie ||
					ui.type == BWAPI::UnitTypes::Terran_Battlecruiser || 
					ui.type == BWAPI::UnitTypes::Terran_Wraith)
				{
					score += ui.type.mineralPrice() + ui.type.gasPrice();
				}

				// Can not target ground
				if (ui.type.groundWeapon() == BWAPI::WeaponTypes::None)
				{
					score += ui.type.mineralPrice() + ui.type.gasPrice();
				}
			}

			// Static defense
			else if (ui.type == BWAPI::UnitTypes::Terran_Missile_Turret ||
				ui.type == BWAPI::UnitTypes::Terran_Bunker)
			{
				score += ui.type.mineralPrice() + ui.type.gasPrice();
			}

		}
	}
	
	int costArmy = InformationManager::Instance().getCostArmy(_self);
	//score += costArmy;

	int costHydra = InformationManager::Instance().getCostUnit(_self, BWAPI::UnitTypes::Zerg_Hydralisk);
	//score -= costHydra;

	// Activate hydras when score is above techcost
	int techcost = 0;
	bool hasDenAll = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Hydralisk_Den) > 0;
	if (!hasDenAll) {
		techcost += 150;
	}

	if (score <= techcost) {
		score = 0;
	}

	// Bias
	if (_enemyRace == BWAPI::Races::Terran)
	{
		score += 100;
		// Hysteresis. Make it bigger than the bias.
		if (_techTarget == "Hydralisks") { score += 300; };
	}
	else if (_enemyRace == BWAPI::Races::Protoss)
	{
		// Hysteresis. Make it bigger than the bias.
		if (_techTarget == "Hydralisks") { score += 200; };
	}

	_denScore = score;

	return score;
}

int StrategyManager::getMutaScore()
{
	int score = 0;

	// vs Protoss
	if (_enemyRace == BWAPI::Races::Protoss)
	{
		for (const auto & kv : InformationManager::Instance().getUnitData(_enemy).getUnits())
		{
			const UnitInfo & ui(kv.second);

			if (!ui.type.isWorker() && !ui.type.isBuilding())
			{
				// Enemy mobile combat units.
				if (ui.type.airWeapon() == BWAPI::WeaponTypes::None ||
					ui.type == BWAPI::UnitTypes::Protoss_Dragoon)
				{
					// can not target air and dragoons
					score += ui.type.mineralPrice() + ui.type.gasPrice();

					// Some units have no weapons and are counted for both sides.
					// Double-count them here if they favor spire.
					if (ui.type == BWAPI::UnitTypes::Protoss_High_Templar)
					{
						// double-count high templar for spire
						score += ui.type.mineralPrice() + ui.type.gasPrice();
					}
					// Reavers especially favor spire.
					else if (ui.type == BWAPI::UnitTypes::Protoss_Reaver)
					{
						score += ui.type.mineralPrice() + ui.type.gasPrice() + 150;
					}
				}
			}

			else if (ui.type == BWAPI::UnitTypes::Protoss_Robotics_Facility)
			{
				// Mutalisks are good against anything from the robo fac.
				score += ui.type.mineralPrice() + ui.type.gasPrice();;
			}
			else if (ui.type == BWAPI::UnitTypes::Protoss_Robotics_Support_Bay)
			{
				// Mutalisks are especially good against reavers.
				score += ui.type.mineralPrice() + ui.type.gasPrice() + 150;
			}
		}
	}

	// vs Terran
	if (_enemyRace == BWAPI::Races::Terran)
	{
		for (const auto & kv : InformationManager::Instance().getUnitData(_enemy).getUnits())
		{
			const UnitInfo & ui(kv.second);

			if (ui.type == BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
			{
				score += ui.type.mineralPrice() + ui.type.gasPrice();
			}
			else if (ui.type == BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode)
			{
				score += ui.type.mineralPrice() + ui.type.gasPrice() + 300;
			}
		}
	}

	int costArmy = InformationManager::Instance().getCostArmy(_self);
	//score += costArmy;

	int costMuta = InformationManager::Instance().getCostUnit(_self, BWAPI::UnitTypes::Zerg_Mutalisk);
	//score -= costMuta;

	// Activate mutas when score is above techcost
	int techcost = 0;
	int nLairsAll = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Lair);
	if (nLairsAll == 0) {
		techcost += 250;
	}

	bool hasSpireAll = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Spire) > 0 ||
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Greater_Spire) > 0;
	if (!hasSpireAll) {
		techcost += 350;
	}

	if (score <= techcost) {
		score = 0;
	}

	// Bias
	if (_enemyRace == BWAPI::Races::Protoss)
	{
		score += 200;
		// Hysteresis. Make it bigger than the bias.
		if (_techTarget == "Mutalisks") { score += 400; };
	}
	else if (_enemyRace == BWAPI::Races::Terran)
	{
		// Hysteresis. Make it bigger than the bias.
		if (_techTarget == "Mutalisks") { score += 200; };
	}
	else
	{
		score += 100;
	}

	_spireScore = score;

	return score;
}

int StrategyManager::getLurkerScore()
{
	int score = 0;

	// vs Protoss
	if (_enemyRace == BWAPI::Races::Protoss)
	{
		for (const auto & kv : InformationManager::Instance().getUnitData(_enemy).getUnits())
		{
			const UnitInfo & ui(kv.second);

			if (!ui.type.isWorker() && !ui.type.isBuilding())
			{
				// Gate units
				if (ui.type == BWAPI::UnitTypes::Protoss_Zealot ||
					ui.type == BWAPI::UnitTypes::Protoss_Dragoon)
				{
					score += ui.type.mineralPrice() + ui.type.gasPrice();
				}

				// Can not target ground
				if (ui.type.groundWeapon() == BWAPI::WeaponTypes::None)
				{
					score += ui.type.mineralPrice() + ui.type.gasPrice();
				}
			}
		}
	}

	// vs Terran
	if (_enemyRace == BWAPI::Races::Terran)
	{
		for (const auto & kv : InformationManager::Instance().getUnitData(_enemy).getUnits())
		{
			const UnitInfo & ui(kv.second);

			if (!ui.type.isWorker() && !ui.type.isBuilding())
			{
				// Bio
				if (ui.type == BWAPI::UnitTypes::Terran_Marine || 
					ui.type == BWAPI::UnitTypes::Terran_Medic ||
					ui.type == BWAPI::UnitTypes::Terran_Firebat)
				{
					score += ui.type.mineralPrice() + ui.type.gasPrice();
				}

				// Can not target ground
				if (ui.type.groundWeapon() == BWAPI::WeaponTypes::None)
				{
					score += ui.type.mineralPrice() + ui.type.gasPrice();
				}
			}
		}
	}

	int costArmy = InformationManager::Instance().getCostArmy(_self);
	//score += costArmy;

	int costLurker = InformationManager::Instance().getCostUnit(_self, BWAPI::UnitTypes::Zerg_Lurker);
	//score -= costLurker;

	// Activate lurkers when score is above techcost
	int techcost = 0;
	bool hasDenAll = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Hydralisk_Den) > 0;
	if (!hasDenAll) {
		//techcost += 150;
	}

	int nLairsAll = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Lair);
	if (nLairsAll == 0) {
		techcost += 250;
	}

	bool lurkerAspRes = _self->hasResearched(BWAPI::TechTypes::Lurker_Aspect);
	if (!lurkerAspRes) {
		techcost += 400;
	}

	if (score <= techcost) {
		score = 0;
	}

	// Bias
	if (_enemyRace == BWAPI::Races::Terran)
	{
		score += 200;
		// Hysteresis. Make it bigger than the bias.
		if (_techTarget == "Lurkers") { score += 400; };
	}
	else if (_enemyRace == BWAPI::Races::Protoss)
	{
		score += 100;
		// Hysteresis. Make it bigger than the bias.
		if (_techTarget == "Lurkers") { score += 300; };
	}

	// Disable lurkers at hive tech for vs Terran
	//bool hasHive = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Hive) > 0;
	//if (hasHive && _enemyRace == BWAPI::Races::Terran)
	//{
	//	score = 0;
	//}

	_lurkerScore = score;

	return score;
}

int StrategyManager::getUltraScore()
{
	int score = 0;

	// vs Protoss
	if (_enemyRace == BWAPI::Races::Protoss)
	{
		for (const auto & kv : InformationManager::Instance().getUnitData(_enemy).getUnits())
		{
			const UnitInfo & ui(kv.second);

			if (!ui.type.isWorker() && !ui.type.isBuilding())
			{
				// Spellcasters
				if (ui.type == BWAPI::UnitTypes::Protoss_High_Templar||
					ui.type == BWAPI::UnitTypes::Protoss_Dark_Templar)
				{
					score += ui.type.mineralPrice() + ui.type.gasPrice();
				}

				// Can not target ground
				if (ui.type.groundWeapon() == BWAPI::WeaponTypes::None)
				{
					score += ui.type.mineralPrice() + ui.type.gasPrice();
				}
			}

			// Static defense
			else if (ui.type == BWAPI::UnitTypes::Protoss_Photon_Cannon)
			{
				score += ui.type.mineralPrice() + ui.type.gasPrice();
			}

		}
	}

	// vs Terran
	if (_enemyRace == BWAPI::Races::Terran)
	{
		for (const auto & kv : InformationManager::Instance().getUnitData(_enemy).getUnits())
		{
			const UnitInfo & ui(kv.second);

			if (!ui.type.isWorker() && !ui.type.isBuilding())
			{
				// Bio
				if (ui.type == BWAPI::UnitTypes::Terran_Marine ||
					ui.type == BWAPI::UnitTypes::Terran_Medic ||
					ui.type == BWAPI::UnitTypes::Terran_Firebat)
				{
					score += ui.type.mineralPrice() + ui.type.gasPrice();
				}

				// Can not target ground
				if (ui.type.groundWeapon() == BWAPI::WeaponTypes::None)
				{
					score += ui.type.mineralPrice() + ui.type.gasPrice();
				}
			}

			// Static defense
			else if (ui.type == BWAPI::UnitTypes::Terran_Missile_Turret ||
				ui.type == BWAPI::UnitTypes::Terran_Bunker)
			{
				score += ui.type.mineralPrice() + ui.type.gasPrice();
			}

		}
	}

	// Activate ultras when score is above techcost
	int techcost = 0;
	bool hasUltraAll = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Ultralisk_Cavern) > 0;
	if (!hasUltraAll) {
		techcost += 350;
	}

	if (score <= techcost) {
		score = 0;
	}

	// Enable ultras at hive tech vs Terran
	bool hasHive = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Hive) > 0;
	if (!hasHive && _enemyRace == BWAPI::Races::Terran)
	{
		score = 0;
	}

	_ultraScore = score;

	return score;
}
/*
// Versus protoss, decide whether hydras or mutas are more valuable.
// Decide by looking at the protoss unit mix.
bool StrategyManager::vProtossDenOverSpire()
{
	// Bias.
	int denScore = 0;
	int spireScore = 3;

	// Hysteresis. Make it bigger than the bias.
	if (_techTarget == "Hydralisks") { denScore += 8; };
	if (_techTarget == "Mutalisks")  { spireScore += 12; };

	for (const auto & kv : InformationManager::Instance().getUnitData(_enemy).getUnits())
	{
		const UnitInfo & ui(kv.second);

		if (!ui.type.isWorker() && !ui.type.isBuilding())
		{
			// Enemy mobile combat units.
			if (ui.type.airWeapon() == BWAPI::WeaponTypes::None ||
				ui.type == BWAPI::UnitTypes::Protoss_Dragoon)
			{
				// can not target air and dragoons
				spireScore += ui.type.supplyRequired();

				// Some units have no weapons and are counted for both sides.
				// Double-count them here if they favor spire.
				if (ui.type == BWAPI::UnitTypes::Protoss_High_Templar)
				{
					// double-count high templar for spire
					spireScore += ui.type.supplyRequired();
				}
				// Reavers especially favor spire.
				else if (ui.type == BWAPI::UnitTypes::Protoss_Reaver)
				{
					spireScore += ui.type.supplyRequired() + 6;
				}
			}

			if (ui.type.groundWeapon() == BWAPI::WeaponTypes::None || 
				ui.type == BWAPI::UnitTypes::Protoss_Archon ||
				ui.type == BWAPI::UnitTypes::Protoss_Scout)
			{
				// can not target ground, corsairs and other spellcasters
				denScore += ui.type.supplyRequired();
			}
		}
		
		else if (ui.type == BWAPI::UnitTypes::Protoss_Photon_Cannon)
		{
			// Hydralisks are efficient against cannons.
			denScore += 2;
		}
		else if (ui.type == BWAPI::UnitTypes::Protoss_Robotics_Facility)
		{
			// Mutalisks are good against anything from the robo fac.
			spireScore += 2;
		}
		else if (ui.type == BWAPI::UnitTypes::Protoss_Robotics_Support_Bay)
		{
			// Mutalisks are especially good against reavers.
			spireScore += 6;
		}
	}

	_denScore = denScore;
	_spireScore = spireScore;

	return denScore > spireScore;
}

// Versus terran, decide whether hydras or mutas are more valuable.
// Decide by looking at the terran unit mix.
bool StrategyManager::vTerranDenOverSpire()
{
	// Bias.
	int denScore = 3;
	int spireScore = 0;

	// Hysteresis. Make it bigger than the bias.
	if (_techTarget == "Hydralisks") { denScore += 12; };
	if (_techTarget == "Mutalisks")  { spireScore += 8; };

	for (const auto & kv : InformationManager::Instance().getUnitData(_enemy).getUnits())
	{
		const UnitInfo & ui(kv.second);
		
		//if (ui.type == BWAPI::UnitTypes::Terran_Marine ||
		//	ui.type == BWAPI::UnitTypes::Terran_Medic ||
		//	ui.type == BWAPI::UnitTypes::Terran_Firebat)
		//{
		//	denScore += 1;
		//}

		if (ui.type == BWAPI::UnitTypes::Terran_Marine)
		{
			denScore += 1;
		}
		else if (ui.type == BWAPI::UnitTypes::Terran_Missile_Turret)
		{
			denScore += 2;
		}
		else if (ui.type == BWAPI::UnitTypes::Terran_Vulture)
		{
			//denScore += 2;
			denScore += 1;
		}
		else if (ui.type == BWAPI::UnitTypes::Terran_Goliath)
		{
			denScore += 2;
		}
		else if (ui.type == BWAPI::UnitTypes::Terran_Valkyrie ||
			ui.type == BWAPI::UnitTypes::Terran_Battlecruiser)
		{
			denScore += 4;
		}
		else if (ui.type == BWAPI::UnitTypes::Terran_Wraith)
		{
			denScore += 1;
		}
		else if (ui.type == BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode || 
			ui.type == BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
		{
			spireScore += 10;
		}
	}

	_denScore = denScore;
	_spireScore = spireScore;

	return denScore > spireScore;
}
*/

void StrategyManager::chooseGasUnit()
{
	int mutascore = getMutaScore();
	int hydrascore = getHydraScore();
	int lurkerscore = getLurkerScore();
	int ultrascore = getUltraScore();

	bool hasDen = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Hydralisk_Den) > 0;
	bool hasLurker = _self->hasResearched(BWAPI::TechTypes::Lurker_Aspect);
	bool hasSpire = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Spire) > 0 ||
		UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Greater_Spire) > 0;

	if (_enemyRace == BWAPI::Races::Protoss ||
		_enemyRace == BWAPI::Races::Terran)
	{
		_gasUnit = "None";
		if (hasDen && hasLurker && hasSpire) {
			if (lurkerscore > mutascore && lurkerscore > hydrascore)
			{
				_gasUnit = "Lurker";
			}
			else if (mutascore > lurkerscore && mutascore > hydrascore)
			{
				_gasUnit = "Mutalisk";
			}
			else if (hydrascore > lurkerscore && hydrascore > mutascore)
			{
				_gasUnit = "Hydralisk";
			}
		}
		else if (hasDen && hasLurker && !hasSpire) {
			_gasUnit = lurkerscore > hydrascore ? "Lurker" : "Hydralisk";
		}
		else if (hasDen && !hasLurker && hasSpire) {
			_gasUnit = hydrascore > mutascore ? "Hydralisk" : "Mutalisk";
		}
		else if (!hasDen && !hasLurker && hasSpire) {
			_gasUnit = "Mutalisk";
		}
		else if (hasDen && !hasLurker && !hasSpire) {
			_gasUnit = "Hydralisk";
		}
	}

	// Otherwise enemy is zerg or random. Always go spire.
	else
	{
		if (hasSpire) {
			_gasUnit = "Mutalisk";
		}
	}
}
/*
// Are hydras or mutas more valuable?
// true = hydras, false = mutas
bool StrategyManager::chooseDenOverSpire()
{
	if (_enemyRace == BWAPI::Races::Protoss || 
		_enemyRace == BWAPI::Races::Terran)
	{
		return getHydraScore() >= getMutaScore();
	}

	// Otherwise enemy is zerg or random. Always go spire.
	return false;
}
*/
// Choose the next tech to aim for
void StrategyManager::chooseTechTarget()
{
	const bool ZvZ = _enemyRace == BWAPI::Races::Zerg;
	bool hasDen = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Hydralisk_Den) > 0;
	bool hasSpire = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Spire) > 0 ||
		UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Greater_Spire) > 0;
	bool hasUltra = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Ultralisk_Cavern) > 0;
	bool hasLurker = _self->hasResearched(BWAPI::TechTypes::Lurker_Aspect);

	// True if we DON'T want it: We have it or choose to skip it.
	const bool den = ZvZ || hasDen;
	const bool spire = hasSpire;
	const bool ultra = hasUltra;
	const bool lurker = ZvZ || hasLurker;

	// Default. Value at the start of the game and after all tech is available.
	_techTarget = "None";

	// From low tech to high.
	if (!den && !spire && !ultra && !lurker)
	{
		_techTarget = _lurkerScore > _spireScore && _lurkerScore > _denScore ? "Lurkers" : "None";
		if (_techTarget == "None")
		{
			_techTarget = _spireScore > _denScore && _spireScore > _lurkerScore ? "Mutalisks" : "None";
		}
		if (_techTarget == "None")
		{
			_techTarget = _denScore > _spireScore && _denScore > _lurkerScore ? "Hydralisks" : "Ultralisks";
		}
	}
	//else if (!den && !spire && !ultra)
	//{
	//		_techTarget = _denScore > _spireScore ? "Hydralisks" : "Mutalisks";
	//}
	else if (!den && spire && !ultra)
	{
		_techTarget = _denScore > _spireScore ? "Hydralisks" : "Ultralisks";
	}
	else if (den && !spire && !ultra)
	{
		if (!lurker)
			_techTarget = _lurkerScore > _spireScore ? "Lurkers" : "Mutalisks";
		else
			_techTarget = _denScore > _spireScore ? "Ultralisks" : "Mutalisks";
	}
	else if (den && spire && !ultra)
	{
		if (!lurker)
			_techTarget = _lurkerScore > _denScore && _lurkerScore > _spireScore ? "Lurkers" : "Ultralisks";
		else
			_techTarget = "Ultralisks";
	}
	else if (!den && !spire && ultra)
	{
		_techTarget = _denScore > _spireScore ? "Hydralisks" : "Mutalisks";
	}
	else if (!den && spire && ultra)
	{
		_techTarget = _denScore > _spireScore ? "Hydralisks" : "None";
	}
	else if (den && !spire && ultra)
	{
		if (!lurker)
			_techTarget = _lurkerScore > _spireScore ? "Lurkers" : "Mutalisks";
		else
			_techTarget = _denScore > _spireScore ? "None" : "Mutalisks";
	}
	else if (den && spire && ultra)
	{
		if (!lurker)
			_techTarget = _lurkerScore > _denScore && _lurkerScore > _spireScore ? "Lurkers" : "None";
	}

}

// Return true if the building is in the building queue with any status.
bool StrategyManager::isBeingBuilt(BWAPI::UnitType unitType)
{
	UAB_ASSERT(unitType.isBuilding(), "not a building");
	return BuildingManager::Instance().isBeingBuilt(unitType);
}

int StrategyManager::maxDrones()
{
	int droneMax;
	droneMax = std::max(9, std::min(WorkerManager::Instance().getMaxWorkers(), absoluteMaxDrones));

	return droneMax;
}

void StrategyManager::drawStrategyInformation(int x, int y)
{
	if (!Config::Debug::DrawStrategyInfo)
	{
		return;
	}

	BWAPI::Broodwar->drawTextScreen(x, y, "\x04Strategy:");
	BWAPI::Broodwar->drawTextScreen(x+50, y, "\x03%s%s  %s%s  %s%s  %s%s  %s%d  %s%d  %s%d",
		Config::Strategy::StrategyName.c_str(), Config::Strategy::FoundEnemySpecificStrategy ? " (enemy specific)" : "",
		"TechTarget: ", _techTarget.c_str(),
		"MinUnit: ", _minUnit.c_str(),
		"GasUnit: ", _gasUnit.c_str(),
		"Hydras: ", _denScore,
		"Mutas: ", _spireScore,
		"Lurkers: ", _lurkerScore);
}
