#include "CombatSimulation.h"

using namespace UAlbertaBot;
using namespace Neolib;
using namespace BWAPI;

CombatSimulation::CombatSimulation()
{
}

// sets the starting states based on the combat units within a radius of a given position
// this center will most likely be the position of the forwardmost combat unit we control
bool CombatSimulation::simulateCombatFAP(const BWAPI::Position & center, const int radius)
{
	FastAPproximation s;

	if (Config::Debug::DrawCombatSimulationInfo)
	{
		BWAPI::Broodwar->drawCircleMap(center.x, center.y, 6, BWAPI::Colors::Red, true);
		BWAPI::Broodwar->drawCircleMap(center.x, center.y, radius, BWAPI::Colors::Red);
	}

	BWAPI::Unitset ourCombatUnits;
	std::vector<UnitInfo> enemyCombatUnits;
	int mySize = 0;
	int enSize = 0;

	MapGrid::Instance().GetUnits(ourCombatUnits, center, radius, true, false);
	InformationManager::Instance().getNearbyForce(enemyCombatUnits, center, BWAPI::Broodwar->enemy(), radius);

	for (auto & unit : ourCombatUnits)
	{
		// Exclude workers and buildings (and static defense)
		if (unit->getType().isWorker() || unit->getType().isBuilding())
		{
			continue;
		}

		// Only include completed units
		if (unit->isCompleted())
		{
			s.addIfCombatUnitPlayer1(unit);
			mySize++;
		}
	}

	for (UnitInfo & ui : enemyCombatUnits)
	{
		// Exclude workers
		if (ui.type.isWorker())
		{
			continue;
		}

		// Exclude unpowered buildings for protoss
		if (ui.player->getRace() == BWAPI::Races::Protoss && 
			ui.type.isBuilding() && !ui.unit->isPowered())
		{
			continue;
		}

		// Only include completed units
		if (ui.completed)
		{
			s.addIfCombatUnitPlayer2(ui);
			enSize++;
		}
	}


	auto initState = s.getState();
	auto initStatus = s.playerScores();
	auto initStatusB = s.playerScoresBuildings().second;
	s.simulate(135);
	auto simState = s.getState();
	auto simStatus = s.playerScores();
	auto simStatusB = s.playerScoresBuildings().second;
	//bool loss = (initStatus.first - simStatus.first) > (initStatus.second - simStatus.second);
	int result = simStatus.first - simStatus.second;

	if (Config::Debug::DrawCombatSimulationInfo)
	{
		std::stringstream ss1;
		ss1 << "Initial score: " << initStatus.first << " : " << initStatus.second << "\n";
		ss1 << "P1: " << mySize << "\n";
		ss1 << "P2: " << enSize;

		std::stringstream ss2;
		ss2 << "Sim score: " << simStatus.first << " : " << simStatus.second << "\n";
		ss2 << "P1: " << simState.first->size() << "\n";
		ss2 << "P2: " << simState.second->size();

		BWAPI::Broodwar->drawTextScreen(150, 200, "%s", ss1.str().c_str());
		BWAPI::Broodwar->drawTextScreen(300, 200, "%s", ss2.str().c_str());

		std::string prefix = result < 0 ? "\x06" : "\x07";
		BWAPI::Broodwar->drawTextScreen(240, 240, "Combat Sim: %s%d", prefix.c_str(), result);
	}

	//return result < 100 && simState.second->size() != 0;
	return result < 0;
}

// sets the starting states based on the combat units within a radius of a given position
// this center will most likely be the position of the forwardmost combat unit we control
void CombatSimulation::setCombatUnits(const BWAPI::Position & center, const int radius)
{
	SparCraft::GameState s;

	if (Config::Debug::DrawCombatSimulationInfo)
	{
		BWAPI::Broodwar->drawCircleMap(center.x, center.y, 6, BWAPI::Colors::Red, true);
		BWAPI::Broodwar->drawCircleMap(center.x, center.y, radius, BWAPI::Colors::Red);
	}

	BWAPI::Unitset ourCombatUnits;
	std::vector<UnitInfo> enemyCombatUnits;
	int count1 = 0;
	int count2 = 0;

	MapGrid::Instance().GetUnits(ourCombatUnits, center, radius, true, false);
	InformationManager::Instance().getNearbyForce(enemyCombatUnits, center, BWAPI::Broodwar->enemy(), radius);

	for (auto & unit : ourCombatUnits)
	{
		// BUG WORKAROUND: Sparcraft only handles 100 units (sum of both sides)
		if (count1 == 50) {
			continue;
		}

		// Exclude workers and buildings (and static defense)
		if (unit->getType().isWorker() || unit->getType().isBuilding())
        {
            continue;
        }

        if (InformationManager::Instance().isCombatUnit(unit->getType()) && 
			SparCraft::System::isSupportedUnitType(unit->getType()) && unit->isCompleted())
		{
            try
            {
			    s.addUnit(getSparCraftUnit(unit));
				count1++;
            }
            catch (int e)
            {
				// Ignore the exception and the unit.
				e = 0;    // use the variable to avoid a pointless warning
				//BWAPI::Broodwar->printf("Problem Adding Self Unit with ID: %d", unit->getID());
            }
		}
	}

	for (UnitInfo & ui : enemyCombatUnits)
	{
		// BUG WORKAROUND: Sparcraft only handles 100 units (sum of both sides)
		if (count2 == 50) {
			continue;
		}

		// Exclude workers
        if (ui.type.isWorker())
        {
            continue;
        }

		// Pretend that a bunker are marines with prorated hit points.
		// TODO account for repair--we can pretend that the pretend marines have more hit points
		if (ui.type == BWAPI::UnitTypes::Terran_Bunker && ui.completed)
        {
            double hpRatio = static_cast<double>(ui.lastHealth) / ui.type.maxHitPoints();

            SparCraft::Unit marine( BWAPI::UnitTypes::Terran_Marine,
                            SparCraft::Position(ui.lastPosition), 
                            ui.unitID, 
                            getSparCraftPlayerID(ui.player), 
							static_cast<int>(BWAPI::UnitTypes::Terran_Marine.maxHitPoints() * hpRatio * 1),
                            0,
		                    BWAPI::Broodwar->getFrameCount(), 
                            BWAPI::Broodwar->getFrameCount());

            for (size_t i(0); i < 4; ++i)
            {
                s.addUnit(marine);
				count2++;
            }
            
            continue;
        }

		// I think it excludes enemy air units so they don't scare our zerglings away--
		// only a good idea in certain circumstances.
		// SparCraft claims to support mutas, wraiths, BCs, scouts.
        if (!ui.type.isFlyer() && SparCraft::System::isSupportedUnitType(ui.type) && ui.completed)
		{
            try
            {
			    s.addUnit(getSparCraftUnit(ui));
				count2++;
            }
            catch (int e)
            {
				// Ignore the exception and the unit.
				e = 0;    // use the variable to avoid a pointless warning
                //BWAPI::Broodwar->printf("Problem Adding Enemy Unit with ID: %d %d", ui.unitID, e);
            }
		}
	}

	s.finishedMoving();

	state = s;
}

// Gets a SparCraft unit from a BWAPI::Unit, used for our own units since we have all their info
const SparCraft::Unit CombatSimulation::getSparCraftUnit(BWAPI::Unit unit) const
{
    return SparCraft::Unit( unit->getType(),
                            SparCraft::Position(unit->getPosition()), 
                            unit->getID(), 
                            getSparCraftPlayerID(unit->getPlayer()), 
                            unit->getHitPoints() + unit->getShields(), 
                            0,
		                    BWAPI::Broodwar->getFrameCount(), 
                            BWAPI::Broodwar->getFrameCount());	
}

// Gets a SparCraft unit from a UnitInfo struct, needed to get units of enemy behind FoW
const SparCraft::Unit CombatSimulation::getSparCraftUnit(const UnitInfo & ui) const
{
	BWAPI::UnitType type = ui.type;

    // this is a hack, treat medics as a marine for now
	// TODO this is weird: SparCraft appears to support healing
	if (type == BWAPI::UnitTypes::Terran_Medic)
	{
		type = BWAPI::UnitTypes::Terran_Marine;
	}

    return SparCraft::Unit( ui.type, 
                            SparCraft::Position(ui.lastPosition), 
                            ui.unitID, 
                            getSparCraftPlayerID(ui.player), 
                            ui.lastHealth, 
                            0,
		                    BWAPI::Broodwar->getFrameCount(), 
                            BWAPI::Broodwar->getFrameCount());	
}

SparCraft::ScoreType CombatSimulation::simulateCombat()
{
    try
    {
	    SparCraft::GameState s1(state);
        SparCraft::PlayerPtr selfNOK(new SparCraft::Player_NOKDPS(getSparCraftPlayerID(BWAPI::Broodwar->self())));
	    SparCraft::PlayerPtr enemyNOK(new SparCraft::Player_NOKDPS(getSparCraftPlayerID(BWAPI::Broodwar->enemy())));
	    SparCraft::Game g (s1, selfNOK, enemyNOK, 2000);

	    g.play();
	
	    SparCraft::ScoreType eval =  g.getState().eval(SparCraft::Players::Player_One, SparCraft::EvaluationMethods::LTD2).val();

        if (Config::Debug::DrawCombatSimulationInfo)
        {
            std::stringstream ss1;
            ss1 << "Initial State:\n";
            ss1 << s1.toStringCompact() << "\n\n";

            std::stringstream ss2;
            ss2 << "Predicted Outcome: " << eval << "\n";
            ss2 << g.getState().toStringCompact() << "\n";

            BWAPI::Broodwar->drawTextScreen(150,200,"%s", ss1.str().c_str());
            BWAPI::Broodwar->drawTextScreen(300,200,"%s", ss2.str().c_str());

			std::string prefix = (eval < 0) ? "\x06" : "\x07";
	        BWAPI::Broodwar->drawTextScreen(240, 280, "Combat Sim : %s%d", prefix.c_str(), eval);
        }
        
	    return eval;
    }
    catch (int e)
    {
        //BWAPI::Broodwar->printf("SparCraft FatalError, simulateCombat() threw");

        return e;
    }
}

const SparCraft::GameState & CombatSimulation::getSparCraftState() const
{
	return state;
}

const SparCraft::IDType CombatSimulation::getSparCraftPlayerID(BWAPI::Player player) const
{
	if (player == BWAPI::Broodwar->self())
	{
		return SparCraft::Players::Player_One;
	}
	if (player == BWAPI::Broodwar->enemy())
	{
		return SparCraft::Players::Player_Two;
	}

	return SparCraft::Players::Player_None;
}
