#include "InformationManager.h"

#define SELF_INDEX 0
#define ENEMY_INDEX 1

// constructor
InformationManager::InformationManager()
{
	// initalized Player
	_self = BWAPI::Broodwar->self();
	_enemy = BWAPI::Broodwar->enemy();

	// initializedRegionInformation
	_mainBaseLocations[_self] = BWTA::getStartLocation(_self);
	_mainBaseLocations[_enemy] = BWTA::getStartLocation(_enemy);
	updateOccupiedRegions(BWTA::getRegion(_mainBaseLocations[_self]->getPosition()), _self);

	// initializeNaturalBase
	initializeNaturalBase();

	enemyInfluenceMap.resize(BWAPI::Broodwar->mapWidth(), std::vector<gridInfo>(BWAPI::Broodwar->mapHeight(), gridInfo()));

	selfNaturalBaseLocation = BWAPI::TilePositions::None;
	selfStartBaseLocation = BWAPI::Broodwar->self()->getStartLocation();
	// resource depot is the base of all races.
	BOOST_FOREACH(BWAPI::Unit u, BWAPI::Broodwar->self()->getUnits())
	{
		if (u->getType().isResourceDepot())
			selfAllBase.insert(u);
	}


	selfAllBattleUnit[BWAPI::UnitTypes::Zerg_Zergling] = std::set<BWAPI::Unit>();
	selfAllBattleUnit[BWAPI::UnitTypes::Zerg_Hydralisk] = std::set<BWAPI::Unit>();
	selfAllBattleUnit[BWAPI::UnitTypes::Zerg_Mutalisk] = std::set<BWAPI::Unit>();

	selfAllBuilding[BWAPI::UnitTypes::Zerg_Sunken_Colony] = std::set<BWAPI::Unit>();
	selfAllBuilding[BWAPI::UnitTypes::Zerg_Spore_Colony] = std::set<BWAPI::Unit>();
	selfAllBuilding[BWAPI::UnitTypes::Zerg_Creep_Colony] = std::set<BWAPI::Unit>();

	defendTrig = false;

	BOOST_FOREACH(BWAPI::Unit u, selfAllBase)
	{
		if (u->getTilePosition() == BWAPI::Broodwar->self()->getStartLocation())
		{
			selfBaseUnit = u;
			break;
		}
	}

	_weHaveCombatUnits = false;

	depotBalanceFlag = true;
	waitforDepotTime = 0;
	depotTrigMap[selfBaseUnit] = true;
	earlyRush = true;

	
	waitToBuildSunker = 0;

	baseSunkenBuildingPosition = BWAPI::TilePositions::None;
	natrualSunkenBuildingPosition = BWAPI::TilePositions::None;

	enemyEarlyRushSuccess = false;
	airDropTrigger = true;

	airDefendTrigger = true;
	chamberTrigger = true;

	needDefendCheck = true;
	
	defendBuildZerglingsCount = 0;

	debugMode = true;
}

// FIgure out what base is our "natural expansion". In rare cases, there might be none.
// Prerequisite: Call initializeRegionInformation() first.
void InformationManager::initializeNaturalBase()
{
	// We'll go through the bases and pick the best one as the natural.
	BWTA::BaseLocation * bestBase = nullptr;
	double bestScore = 0.0;

	BWAPI::TilePosition homeTile = _self->getStartLocation();
	BWAPI::Position myBasePosition(homeTile);

	for (BWTA::BaseLocation * base : BWTA::getBaseLocations())
	{
		double score = 0.0;

		BWAPI::TilePosition tile = base->getTilePosition();

		// The main is not the natural.
		if (tile == homeTile)
		{
			continue;
		}

		// Want to be close to our own base (unless this is to be a hidden base).
		double distanceFromUs = UAlbertaBot::MapTools::Instance().getGroundDistance(BWAPI::Position(tile), myBasePosition);

		// if it is not connected, continue
		if (!BWTA::isConnected(homeTile, tile) || distanceFromUs < 0)
		{
			continue;
		}

		// Add up the score.
		score = -distanceFromUs;

		// More resources -> better.
		score += 0.01 * base->minerals() + 0.02 * base->gas();

		if (!bestBase || score > bestScore)
		{
			bestBase = base;
			bestScore = score;
		}
	}

	// bestBase may be null on unusual maps.
	_myNaturalBaseLocation = bestBase;
}


int	InformationManager::getEnemyGroundBattleUnitSupply()
{
	int totalSupply = 0;
	for (auto unitCategory : enemyAllBattleUnit)
	{
		if (unitCategory.first.groundWeapon() != BWAPI::WeaponTypes::None && !unitCategory.first.isWorker())
		{
			totalSupply += unitCategory.first.supplyRequired() * unitCategory.second.size();
		}
	}

	return totalSupply;
}


void InformationManager::checkSelfNewDepotFinish()
{
	BOOST_FOREACH(BWAPI::Unit depot, selfAllBase)
	{
		//new depot finish
		if (BWAPI::Broodwar->getFrameCount() > 300 && depotTrigMap.find(depot) == depotTrigMap.end() && depot->isCompleted())
		{
			//waiting for the creep and vision expand
			if (depotBalanceFlag)
			{
				depotBalanceFlag = false;
				waitforDepotTime = BWAPI::Broodwar->getFrameCount() + 30 * 5;  // wait for more than 5 mins.
			}
			if (BWAPI::Broodwar->getFrameCount() > waitforDepotTime)
			{
				WorkerManager::Instance().balanceWorkerOnDepotComplete(depot);
				depotBalanceFlag = true;
				depotTrigMap[depot] = true;
			}
		}
	}
}


void InformationManager::checkEarlyRushDefend()
{

	if (ProductionManager::Instance().getProductionStage() == 1 && needDefendCheck == false)
	{
		return;
	}


	if (BWAPI::Broodwar->getFrameCount() % 25 == 0)
		return;


	int enemySupply = 0;
	for (std::map<BWAPI::UnitType, std::set<BWAPI::Unit>>::iterator it = enemyAllBattleUnit.begin(); it != enemyAllBattleUnit.end(); it++)
	{
		if (it->first.canAttack() && !it->first.isWorker())
		{
			if (it->first == BWAPI::UnitTypes::Zerg_Zergling)
			{
				enemySupply += int(it->first.supplyRequired() * int(it->second.size()) * 2);  // Why enemy is 0.9 and my is 2.0. Here should be consider as a problem.
			}
			else
			{
				enemySupply += it->first.supplyRequired() * int(it->second.size());
			}
		}
	}

	int enemyWorkerSupply = 0;
	BOOST_FOREACH(BWAPI::Unit enemyWorker, BWAPI::Broodwar->enemy()->getUnits())
	{
		if (enemyWorker->getType().isWorker() && (enemyWorker->isAttacking() || enemyWorker->isMoving()) && occupiedRegions[1].size() > 0
			&& occupiedRegions[1].find(BWTA::getRegion(enemyWorker->getPosition())) == occupiedRegions[1].end()) // judge as a scout worker.
		{
			enemyWorkerSupply += enemyWorker->getType().supplyRequired();
		}
	}
	if (enemyWorkerSupply > 2)
	{
		enemySupply += enemyWorkerSupply;
	}

	int ourSupply = 0;
	int zerglingsCount = 0;
	for (std::map<BWAPI::UnitType, std::set<BWAPI::Unit>>::iterator it = selfAllBattleUnit.begin(); it != selfAllBattleUnit.end(); it++)
	{
		if (it->first.canAttack() && !it->first.isWorker())
		{
			ourSupply += it->first.supplyRequired() * int(it->second.size());
			if (it->first == BWAPI::UnitTypes::Zerg_Zergling)
			{
				zerglingsCount += int(it->second.size());
			}
		}
	}

	
	//army is under morph
	int morphSupply = 0;
	BOOST_FOREACH(BWAPI::Unit unit, BWAPI::Broodwar->getAllUnits())
	{
		if (unit->getType() == BWAPI::UnitTypes::Zerg_Egg)
		{
			if (unit->getBuildType().canAttack() && !unit->getBuildType().isWorker())
			{
				if (unit->getBuildType() == BWAPI::UnitTypes::Zerg_Zergling)
				{
					zerglingsCount += 2;
					morphSupply += unit->getBuildType().supplyRequired() * 2;
				}
				else
					morphSupply += unit->getBuildType().supplyRequired();
			}
		}
	}
	ourSupply += morphSupply;

	//waiting to produce unit
	int waitingProductSupply = 0;
	int waitToBuildZergling = 0;
	bool hasDefendItem = false;
	std::vector<BWAPI::UnitType> waitingProduct = ProductionManager::Instance().getWaitingProudctUnit();
	BOOST_FOREACH(BWAPI::UnitType u, waitingProduct)
	{
		if (u.canAttack() && !u.isWorker())
		{
			if (u == BWAPI::UnitTypes::Zerg_Zergling)
			{
				waitToBuildZergling = 0;
				zerglingsCount += 2;
				waitingProductSupply += u.supplyRequired() * 2;
			}
			else
			{
				waitingProductSupply += u.supplyRequired();
			}
			
			hasDefendItem = true;
		}
		else if (u == BWAPI::UnitTypes::Zerg_Sunken_Colony || u == BWAPI::UnitTypes::Zerg_Creep_Colony)
		{
			hasDefendItem = true;
		}
		else
			continue;
			//waitingProductSupply += u.supplyRequired();
	}
	ourSupply += waitingProductSupply;

	//if (hasDefendItem == true)
	//{
		//return;
	//}
	
	//need zergling to scout for enemy info
	if (zerglingsCount <= 2 && selfAllBuilding[BWAPI::UnitTypes::Zerg_Spawning_Pool].size() > 0 && (*selfAllBuilding[BWAPI::UnitTypes::Zerg_Spawning_Pool].begin())->isCompleted())
	{
		ProductionManager::Instance().triggerUnit(BWAPI::UnitTypes::Zerg_Zergling, 1);
	}
	
	int sunkenSupply = 0;
	int sunkenSize = selfAllBuilding[BWAPI::UnitTypes::Zerg_Sunken_Colony].size() + waitToBuildSunker;

	int sunkenFactor = BWAPI::Broodwar->enemy()->getRace() == BWAPI::Races::Zerg ? 4 : 6;  // original is zerg, but I think should be protoss.
	
	if (sunkenSize <= 4)
		sunkenSupply = int(sunkenSize * sunkenFactor * 0.75);
	else if (sunkenSize > 4 && sunkenSize <= 8)
		sunkenSupply = sunkenSize * sunkenFactor;
	else
		sunkenSupply = int(sunkenSize * sunkenFactor * 1.3);
	ourSupply += sunkenSupply;
	
	/*
	//add sunken's force
	if (BWAPI::Broodwar->enemy()->getRace() == BWAPI::Races::Zerg)
		ourSupply += (selfAllBuilding[BWAPI::UnitTypes::Zerg_Sunken_Colony].size() + waitToBuildSunker) * 6;
	else
		ourSupply += (selfAllBuilding[BWAPI::UnitTypes::Zerg_Sunken_Colony].size() + waitToBuildSunker) * 6;
	*/

	
	int idleLarvaCount = BWAPI::Broodwar->self()->allUnitCount(BWAPI::UnitTypes::Zerg_Larva);
	int eggCount = BWAPI::Broodwar->self()->allUnitCount(BWAPI::UnitTypes::Zerg_Egg);
	if ((eggCount + idleLarvaCount + waitToBuildZergling) > 0 && (idleLarvaCount * 10 / (eggCount + idleLarvaCount + waitToBuildZergling) > 5) 
		&& idleLarvaCount >= 6
		&& BWAPI::Broodwar->getFrameCount() % (24 * 2) == 0  // 2 seconds.
		&& defendBuildZerglingsCount <= 3)  // origin:10
	{
		ProductionManager::Instance().triggerUnit(BWAPI::UnitTypes::Zerg_Zergling, 1);
		defendBuildZerglingsCount += 1;
	}


	// fix the defendBuildZerglings less.
	if (BWAPI::Broodwar->self()->completedUnitCount(BWAPI::UnitTypes::Zerg_Zergling) <= 3 && defendBuildZerglingsCount >3 && BWAPI::Broodwar->getFrameCount() %(BWAPI::UnitTypes::Zerg_Zergling.buildTime()) == 0)
		defendBuildZerglingsCount = (int)(3 - BWAPI::Broodwar->self()->completedUnitCount(BWAPI::UnitTypes::Zerg_Zergling)/2);
	


	if (ourSupply < enemySupply)
	{


		int productionSupply = enemySupply - ourSupply;

		if (selfAllBuilding[BWAPI::UnitTypes::Zerg_Spawning_Pool].size() > 0 && (*selfAllBuilding[BWAPI::UnitTypes::Zerg_Spawning_Pool].begin())->isCompleted())
		{
			openingStrategy opening = StrategyManager::Instance().getCurrentopeningStrategy();

			//is natural complete
			bool isNatrualComplete = false;
			BOOST_FOREACH(BWAPI::Unit base, selfAllBuilding[BWAPI::UnitTypes::Zerg_Hatchery])
			{
				if (BWTA::getRegion(base->getPosition()) == BWTA::getRegion(getOurNatrualLocation()))
				{
					if (base->isCompleted())
						isNatrualComplete = true;
				}
			}

			// For Zerg, 5 sunkens.
			if (sunkenSize > 4 && BWAPI::Broodwar->enemy()->getRace() == BWAPI::Races::Zerg)
				return;

			// For Protoss, 9 sunkens.
			if (sunkenSize > 8 && BWAPI::Broodwar->enemy()->getRace() == BWAPI::Races::Protoss)
				return;
			
			// For Terran, 6 sunkens.
			if (sunkenSize > 5 && BWAPI::Broodwar->enemy()->getRace() == BWAPI::Races::Terran)
				return;


			int sunkenBuildSupply = productionSupply;
			int count;
			if (BWAPI::Broodwar->enemy()->getRace() == BWAPI::Races::Protoss)
				count = sunkenBuildSupply / 6 + 1 >= 2 ? 2 : sunkenBuildSupply / 6 + 1;
			else
				count = sunkenBuildSupply / 8 + 1 >= 2 ? 2 : sunkenBuildSupply / 6 + 1;
			ProductionManager::Instance().triggerBuilding(BWAPI::UnitTypes::Zerg_Sunken_Colony, getSunkenBuildingPosition(), count);
			waitToBuildSunker += count;

		}
		else
			return;
	}
}

void InformationManager::checkAirDrop()
{
	if (BWAPI::Broodwar->getFrameCount() > 12000)
		return;

	if (airDropTrigger && enemyAllBattleUnit[BWAPI::UnitTypes::Protoss_Shuttle].size() > 0 || enemyAllBattleUnit[BWAPI::UnitTypes::Terran_Dropship].size() > 0)
	{
		ProductionManager::Instance().triggerBuilding(BWAPI::UnitTypes::Zerg_Sunken_Colony, BWAPI::Broodwar->self()->getStartLocation(), 2);
		airDropTrigger = false;
	}
}

void InformationManager::checkAirDefend()
{
	if (chamberTrigger && (enemyAllBuilding[BWAPI::UnitTypes::Zerg_Spire].size() > 0 || enemyAllBuilding[BWAPI::UnitTypes::Protoss_Stargate].size() > 0 || isEnemyHasFlyerAttacker()) 
		&& BWAPI::Broodwar->self()->allUnitCount(BWAPI::UnitTypes::Zerg_Evolution_Chamber) == 0)
	{
		chamberTrigger = false;
		ProductionManager::Instance().triggerBuilding(BWAPI::UnitTypes::Zerg_Evolution_Chamber, BWAPI::Broodwar->self()->getStartLocation(), 1);
	}

	if (airDefendTrigger && 
		((enemyAllBuilding[BWAPI::UnitTypes::Zerg_Spire].size() > 0)
		|| isEnemyHasFlyerAttacker() || enemyAllBuilding[BWAPI::UnitTypes::Protoss_Stargate].size() > 0))
	{
		airDefendTrigger = false;
		ProductionManager::Instance().triggerBuilding(BWAPI::UnitTypes::Zerg_Spore_Colony, BWAPI::Broodwar->self()->getStartLocation(), 1);
		ProductionManager::Instance().triggerBuilding(BWAPI::UnitTypes::Zerg_Spore_Colony, selfNaturalBaseLocation, 1);
	}
}

void InformationManager::update()
{

	// spar craft
	updateUnitInfo();
	updateBaseLocationInfo();
	
	// original
	checkAirDefend();
	checkAirDrop();

	if (BWAPI::Broodwar->self()->allUnitCount(BWAPI::UnitTypes::Zerg_Drone) >= 8 && StrategyManager::Instance().getCurrentopeningStrategy()!=FivePoolling)
		checkEarlyRushDefend();

	////updateEnemyUnitInfluenceMap();

	//for enemy buildings not destroy by us
	checkOccupiedDetail();


	//if (debugMode)
		//printInfo();
}

void InformationManager::updateUnitInfo()
{
	for (auto & unit : _enemy->getUnits())
	{
			_unitData[unit->getPlayer()].updateUnit(unit);
	}

	for (auto & unit : _self->getUnits())
	{
			_unitData[unit->getPlayer()].updateUnit(unit);
	}

	_unitData[_enemy].removeBadUnits();
	_unitData[_self].removeBadUnits();
}




void InformationManager::checkOccupiedDetail()
{
	for (std::map<BWTA::Region*, std::map<BWAPI::Unit, buildingInfo>>::iterator it = enemyOccupiedDetail.begin(); it != enemyOccupiedDetail.end();)
	{
		for (std::map<BWAPI::Unit, buildingInfo>::iterator detailIt = it->second.begin(); detailIt != it->second.end();)
		{
			if (BWAPI::Broodwar->isVisible(BWAPI::TilePosition(detailIt->second.initPosition)))
			{
				bool isExist = false;
				for (auto u : BWAPI::Broodwar->getUnitsOnTile(detailIt->second.initPosition))
				{
					if (u->getPlayer() == BWAPI::Broodwar->enemy() && u->getType() == detailIt->second.unitType)
					{
						isExist = true;
						break;
					}
				}
				if (isExist == false)
				{
					it->second.erase(detailIt++);
				}
				else
				{
					detailIt++;
				}
			}
			else
				detailIt++;
		}
		if (it->second.size() == 0)
		{
			enemyOccupiedDetail.erase(it++);
			occupiedRegions[1].erase(it->first);
		}
		else
			it++;
	}
}

BWAPI::TilePosition	InformationManager::GetNextExpandLocation()
{
	double closest = 999999999;
	BWAPI::TilePosition nextBase = BWAPI::TilePositions::None;
	BOOST_FOREACH(BWTA::BaseLocation* base, BWTA::getBaseLocations())
	{
		/*
		//check if it is not our base
		if (base->getGroundDistance(BWTA::getNearestBaseLocation(selfStartBaseLocation)) < 90)
		{
			continue;
		}*/
		//already expand
		// reginon which has been occupied. And geysers have the priority.
		if (occupiedRegions[0].find(base->getRegion()) != occupiedRegions[0].end() || occupiedRegions[1].find(base->getRegion()) != occupiedRegions[1].end())
		{
			continue;
		}
		if (base->getGroundDistance(BWTA::getNearestBaseLocation(selfStartBaseLocation)) < closest && base->getGeysers().size() > 0 && !base->isIsland())
		{
			closest = base->getGroundDistance(BWTA::getNearestBaseLocation(selfStartBaseLocation));
			nextBase = base->getTilePosition();
		}
	}


	// where without greyers.
	if (nextBase == BWAPI::TilePositions::None)
	{
		BOOST_FOREACH(BWTA::BaseLocation* base, BWTA::getBaseLocations())
		{
			/*
			//check if it is not our base
			if (base->getGroundDistance(BWTA::getNearestBaseLocation(selfStartBaseLocation)) < 90)
			{
				continue;
			}*/
			//already expand
			if (occupiedRegions[0].find(base->getRegion()) != occupiedRegions[0].end() || occupiedRegions[1].find(base->getRegion()) != occupiedRegions[1].end())
			{
				continue;
			}
			if (base->getGroundDistance(BWTA::getNearestBaseLocation(selfStartBaseLocation)) < closest && !base->isIsland())
			{
				closest = base->getGroundDistance(BWTA::getNearestBaseLocation(selfStartBaseLocation));
				nextBase = base->getTilePosition();
			}
		}
	}

	return nextBase;
}


BWAPI::TilePosition	InformationManager::getOurNatrualLocation()
{
	if (selfNaturalBaseLocation != BWAPI::TilePositions::None)
		return selfNaturalBaseLocation;

	// first call
	double closest = 999999999;
	BOOST_FOREACH(BWTA::BaseLocation* base, BWTA::getBaseLocations())
	{
		//check if it is not our base
		if (base->getGroundDistance(BWTA::getNearestBaseLocation(selfStartBaseLocation)) < 90)
		{
			continue;
		}
		if (base->getGroundDistance(BWTA::getNearestBaseLocation(selfStartBaseLocation)) < closest && base->getGeysers().size() > 0)
		{
			closest = base->getGroundDistance(BWTA::getNearestBaseLocation(selfStartBaseLocation));
			selfNaturalBaseLocation = base->getTilePosition();
		}
	}

	double maxDist = 0;
	BWAPI::Position maxChokeCenter;
	BOOST_FOREACH(BWTA::Chokepoint* p, BWTA::getRegion(selfNaturalBaseLocation)->getChokepoints())
	{
		if (selfBaseUnit->getDistance(p->getCenter()) > maxDist)
		{
			maxDist = selfBaseUnit->getDistance(p->getCenter());
			maxChokeCenter = p->getCenter();
		}
	}
	selfNaturalChokePoint = maxChokeCenter;
	return selfNaturalBaseLocation;
}


BWAPI::TilePosition	InformationManager::getSunkenBuildingPosition()
{
	//generate selfNaturalChokePoint
	getOurNatrualLocation();

	// if No second basement with known position.
	if (selfOccupiedDetail.find(BWTA::getRegion(selfNaturalBaseLocation)) == selfOccupiedDetail.end() && baseSunkenBuildingPosition != BWAPI::TilePositions::None)
	{
		return baseSunkenBuildingPosition;
	}
	// build in second basement with known position.
	if (selfOccupiedDetail.find(BWTA::getRegion(selfNaturalBaseLocation)) != selfOccupiedDetail.end() && natrualSunkenBuildingPosition != BWAPI::TilePositions::None)
	{
		return natrualSunkenBuildingPosition;
	}

	// if not second basement with unknown position
	if (selfOccupiedDetail.find(BWTA::getRegion(selfNaturalBaseLocation)) == selfOccupiedDetail.end())
	{
		BWAPI::Position baseChoke = BWTA::getNearestChokepoint(BWAPI::Broodwar->self()->getStartLocation())->getCenter();
		double2 direc = baseChoke - BWAPI::Position(selfStartBaseLocation);
		double2 direcNormal = direc / direc.len();


		int targetx = BWAPI::Position(selfStartBaseLocation).x + int(direcNormal.x * 32 * 8);
		int targety = BWAPI::Position(selfStartBaseLocation).y + int(direcNormal.y * 32 * 8);

		baseSunkenBuildingPosition = BWAPI::TilePosition(targetx / 32, targety / 32);
		return baseSunkenBuildingPosition;
	}
	else
	{
		double2 direc = selfNaturalChokePoint - BWAPI::Position(selfNaturalBaseLocation);
		double2 direcNormal = direc / direc.len();

		BWAPI::Position desireLocation = BWAPI::Position(selfNaturalBaseLocation);
		while (desireLocation.getDistance(selfNaturalChokePoint) > 32 * 2)
		{
			if (BWAPI::Broodwar->hasCreep(desireLocation.x / 32, desireLocation.y / 32))
			{
				int targetx = desireLocation.x + int(direcNormal.x * 32 * 1);
				int targety = desireLocation.y + int(direcNormal.y * 32 * 1);
				desireLocation = BWAPI::Position(targetx, targety);
			}
			else
				break;
		}

		//int targetx = BWAPI::Position(selfNaturalBaseLocation).x + int(direcNormal.x * 32 * 8);
		//int targety = BWAPI::Position(selfNaturalBaseLocation).y + int(direcNormal.y * 32 * 8);

		natrualSunkenBuildingPosition = BWAPI::TilePosition(desireLocation.x / 32, desireLocation.y / 32);
		return natrualSunkenBuildingPosition;
	}

	return BWAPI::TilePositions::None;
}





void InformationManager::onUnitShow(BWAPI::Unit unit)
{
	updateUnit(unit);
}

void InformationManager::onUnitMorph(BWAPI::Unit unit)
{
 	updateUnit(unit);
}


void InformationManager::updateEnemyUnitInfluenceMap()
{
	//reset enemy unit's influence
	for (int i = 0; i < BWAPI::Broodwar->mapWidth(); i++)
	{
		for (int j = 0; j < BWAPI::Broodwar->mapHeight(); j++)
		{
			enemyInfluenceMap[i][j].enemyUnitAirForce = 0;
			enemyInfluenceMap[i][j].enemyUnitGroundForce = 0;
			enemyInfluenceMap[i][j].enemyUnitDecayAirForce = 0;
			enemyInfluenceMap[i][j].enemyUnitDecayGroundForce = 0;
			
			if (int(enemyInfluenceMap[i][j].airForce) != 0 || int(enemyInfluenceMap[i][j].decayAirForce) != 0)
			{
				if (int(enemyInfluenceMap[i][j].airForce) != 0)
					BWAPI::Broodwar->drawTextMap(i * 32, j * 32, "%d", int(enemyInfluenceMap[i][j].airForce));
				//else
					//BWAPI::Broodwar->drawTextMap(i * 32, j * 32, "%d", int(enemyInfluenceMap[i][j].decayAirForce));
			}
		}
	}

	BOOST_FOREACH(BWAPI::Unit enemy, BWAPI::Broodwar->enemy()->getUnits())
	{
		if (!enemy->getType().isBuilding() && enemy->getType().canAttack() && !enemy->getType().isWorker())
		{

			int attackRange = 0;
			if (enemy->getType().groundWeapon() != BWAPI::WeaponTypes::None)
				attackRange = (enemy->getType().groundWeapon().maxRange() / 32) + 2;
			else
				attackRange = (enemy->getType().airWeapon().maxRange() / 32) + 2;
			
			int y_start = enemy->getTilePosition().y - attackRange > 0 ? enemy->getTilePosition().y - attackRange : 0;
			int y_end = enemy->getTilePosition().y + attackRange > BWAPI::Broodwar->mapHeight() - 1 ? BWAPI::Broodwar->mapHeight() - 1 : enemy->getTilePosition().y + attackRange;
			
			int x_start = enemy->getTilePosition().x - attackRange > 0 ? enemy->getTilePosition().x - attackRange : 0;
			int x_end = enemy->getTilePosition().x + attackRange > BWAPI::Broodwar->mapWidth() - 1 ? BWAPI::Broodwar->mapWidth() - 1 : enemy->getTilePosition().x + attackRange;
			
			for (int i = x_start; i <= x_end; i++)
			{
				for (int j = y_start; j <= y_end; j++)
				{
					if (enemy->getType().groundWeapon() != BWAPI::WeaponTypes::None)
						enemyInfluenceMap[i][j].enemyUnitGroundForce += enemy->getType().groundWeapon().damageAmount();
					if (enemy->getType().airWeapon() != BWAPI::WeaponTypes::None)
						enemyInfluenceMap[i][j].enemyUnitAirForce += enemy->getType().airWeapon().damageAmount();
				}
			}
		}
	}
	
	for (int i = 0; i < BWAPI::Broodwar->mapWidth(); i++)
	{
		for (int j = 0; j < BWAPI::Broodwar->mapHeight(); j++)
		{
			if (int(enemyInfluenceMap[i][j].enemyUnitGroundForce) != 0 || int(enemyInfluenceMap[i][j].enemyUnitDecayGroundForce) != 0)
			{
				if (int(enemyInfluenceMap[i][j].enemyUnitGroundForce) != 0)
					BWAPI::Broodwar->drawTextMap(i * 32, j * 32, "%d", int(enemyInfluenceMap[i][j].enemyUnitGroundForce));
				//else
					//BWAPI::Broodwar->drawTextMap(i * 32, j * 32, "%d", int(enemyInfluenceMap[i][j].enemyUnitDecayGroundForce));
			}
		}
	}
}


void InformationManager::setInfluenceMap(BWAPI::Position initPosition, int attackRange, int groundDamage, int airDamage, bool addOrdestroy)
{
	// get the approximate center of the building
	
	double2 normalLength = double2(1, 0);
	std::set<BWAPI::TilePosition> alreadySetPosition;
	int startDegree = 0;
	while (startDegree < 360)
	{
		double2 rotateNormal(normalLength.rotateReturn(startDegree));
		for (int length = 1; length <= attackRange; length++)
		{
			double2 rotateVector(rotateNormal * length + initPosition);
			BWAPI::TilePosition tmp(int(rotateVector.x), int(rotateVector.y));
			if (int(tmp.x) >= 0 && int(tmp.x) < BWAPI::Broodwar->mapWidth() && int(tmp.y) >= 0 && int(tmp.y) < BWAPI::Broodwar->mapHeight()
				&& alreadySetPosition.find(tmp) == alreadySetPosition.end())
			{
				alreadySetPosition.insert(tmp);
				if (addOrdestroy)
				{
					enemyInfluenceMap[tmp.x][tmp.y].groundForce += groundDamage;
					enemyInfluenceMap[tmp.x][tmp.y].airForce += airDamage;
				}
				else
				{
					enemyInfluenceMap[tmp.x][tmp.y].groundForce -= groundDamage;
					enemyInfluenceMap[tmp.x][tmp.y].airForce -= airDamage;
				}
			}
		}
		startDegree += 5;
	}

	int decayRange = 4;
	startDegree = 0;
	double decayGroundValue = 0;
	double decayAirValue = 0;
	while (startDegree < 360)
	{
		double2 rotateNormal(normalLength.rotateReturn(startDegree));
		for (int length = attackRange + 1; length <= attackRange + decayRange; length++)
		{
			if (groundDamage > 0)
				decayGroundValue = groundDamage * (decayRange - (length - attackRange)) / double(decayRange);
			if (airDamage > 0)
				decayAirValue = airDamage * (decayRange - (length - attackRange)) / double(decayRange);
			double2 rotateVector(rotateNormal * length + initPosition);
			BWAPI::TilePosition tmp(int(rotateVector.x), int(rotateVector.y));
			if (int(tmp.x) >= 0 && int(tmp.x) < BWAPI::Broodwar->mapWidth() && int(tmp.y) >= 0 && int(tmp.y) < BWAPI::Broodwar->mapHeight()
				&& alreadySetPosition.find(tmp) == alreadySetPosition.end())
			{
				alreadySetPosition.insert(tmp);
				if (addOrdestroy)
				{
					enemyInfluenceMap[tmp.x][tmp.y].decayGroundForce += decayGroundValue;
					enemyInfluenceMap[tmp.x][tmp.y].decayAirForce += decayAirValue;
				}
				else
				{
					enemyInfluenceMap[tmp.x][tmp.y].decayGroundForce -= decayGroundValue;
					enemyInfluenceMap[tmp.x][tmp.y].decayAirForce -= decayAirValue;
				}
			}
		}
		startDegree += 5;
	}
}

void InformationManager::addUnitInfluenceMap(BWAPI::Unit unit, bool addOrdestroy)
{
	if (addOrdestroy && enemyAllBuilding[unit->getType()].find(unit) != enemyAllBuilding[unit->getType()].end())
		return;

	if (!addOrdestroy && enemyAllBuilding[unit->getType()].find(unit) == enemyAllBuilding[unit->getType()].end())
		return;

	if (unit->getType() == BWAPI::UnitTypes::Protoss_Photon_Cannon
		|| unit->getType() == BWAPI::UnitTypes::Zerg_Spore_Colony
		|| unit->getType() == BWAPI::UnitTypes::Terran_Missile_Turret
		|| unit->getType() == BWAPI::UnitTypes::Zerg_Sunken_Colony
		|| unit->getType() == BWAPI::UnitTypes::Terran_Bunker)
	{		
		int attackRange = 0;
		int airDamage = 0;
		int groundDamage = 0;
		int buildingWidth = unit->getType().tileWidth();
		int buildingHeight = unit->getType().tileHeight();
		int maxSize = buildingWidth > buildingHeight ? buildingWidth / 2 : buildingHeight / 2;

		if (unit->getType() == BWAPI::UnitTypes::Terran_Bunker)
		{
			attackRange = (BWAPI::UnitTypes::Terran_Marine.groundWeapon().maxRange() / 32) + maxSize + 2;

			groundDamage = BWAPI::UnitTypes::Terran_Marine.groundWeapon().damageAmount() * 3;
			airDamage = BWAPI::UnitTypes::Terran_Marine.airWeapon().damageAmount() * 3;
		}

		if (unit->getType().groundWeapon() != BWAPI::WeaponTypes::None)
		{
			attackRange = (unit->getType().groundWeapon().maxRange() / 32) + maxSize + 1;
			groundDamage = unit->getType().groundWeapon().damageAmount();
		}
		if (unit->getType().airWeapon() != BWAPI::WeaponTypes::None)
		{
			attackRange = (unit->getType().airWeapon().maxRange() / 32) + maxSize + 1;
			airDamage = unit->getType().airWeapon().damageAmount();
		}

		double2 initPosition(unit->getTilePosition().x + buildingWidth / 2, unit->getTilePosition().y + buildingHeight / 2);
		setInfluenceMap(initPosition, attackRange, groundDamage, airDamage, addOrdestroy);
	}
} 



void InformationManager::updateUnit(BWAPI::Unit unit)
{
	if (unit->getPlayer() == BWAPI::Broodwar->enemy())
	{
		if (unit->getType().isBuilding())
		{
			if ((unit->getType().canAttack() || unit->getType() == BWAPI::UnitTypes::Terran_Bunker) && !unit->isCompleted())
				return;

			addUnitInfluenceMap(unit, true);
			//updateOccupiedRegions(BWTA::getRegion(unit->getPosition()), BWAPI::Broodwar->enemy());
			addOccupiedRegionsDetail(BWTA::getRegion(unit->getPosition()), BWAPI::Broodwar->enemy(), unit);

			if (unit->getType().isResourceDepot())
			{
				// update base unit for zerg
				BOOST_FOREACH(BWTA::BaseLocation* base, BWTA::getBaseLocations())
				{
					//check if it is not our base
					if (base->getTilePosition().getDistance(unit->getTilePosition()) < 3)
					{
						enemyAllBase.insert(unit);
						break;
					}
				}
			}

			if (enemyAllBuilding.find(unit->getType()) == enemyAllBuilding.end())
			{
				enemyAllBuilding[unit->getType()] = std::set<BWAPI::Unit>();
				enemyAllBuilding[unit->getType()].insert(unit);
			}
			else
				enemyAllBuilding[unit->getType()].insert(unit);
		}
		else
		{
			if (enemyAllBattleUnit.find(unit->getType()) == enemyAllBattleUnit.end())
			{
				enemyAllBattleUnit[unit->getType()] = std::set<BWAPI::Unit>();
				enemyAllBattleUnit[unit->getType()].insert(unit);
			}
			else
				enemyAllBattleUnit[unit->getType()].insert(unit);
		}
	}
	else if (unit->getPlayer() == BWAPI::Broodwar->self())
	{
		if (unit->getType().isBuilding())
		{
			selfAllBattleUnit[BWAPI::UnitTypes::Zerg_Drone].erase(unit);

			addOccupiedRegionsDetail(BWTA::getRegion(unit->getPosition()), BWAPI::Broodwar->self(), unit);
			if (unit->getType().isResourceDepot())
			{
				BOOST_FOREACH(BWTA::BaseLocation* base, BWTA::getBaseLocations())
				{
					if (base->getTilePosition().getDistance(unit->getTilePosition()) < 3)
					{
						selfAllBase.insert(unit);
						break;
					}
				}
			}

			if (unit->getType() == BWAPI::UnitTypes::Zerg_Sunken_Colony && waitToBuildSunker > 0)
			{
				waitToBuildSunker--;
			}

			if (selfAllBuilding.find(unit->getType()) == selfAllBuilding.end())
			{
				selfAllBuilding[unit->getType()] = std::set<BWAPI::Unit>();
				selfAllBuilding[unit->getType()].insert(unit);
			}
			else
				selfAllBuilding[unit->getType()].insert(unit);
		}
		else
		{
			if (selfAllBattleUnit.find(unit->getType()) == selfAllBattleUnit.end())
			{
				selfAllBattleUnit[unit->getType()] = std::set<BWAPI::Unit>();
				selfAllBattleUnit[unit->getType()].insert(unit);
			}
			else
				selfAllBattleUnit[unit->getType()].insert(unit);
		}
	}
}


bool InformationManager::isEnemyHasInvisibleUnit()
{
	for (auto eU : enemyAllBattleUnit)
	{
		if (eU.first.hasPermanentCloak() 
			|| eU.first == BWAPI::UnitTypes::Zerg_Lurker
			|| eU.first == BWAPI::UnitTypes::Terran_Wraith)
		{
			return true;
		}
	}
	return false;
}


bool InformationManager::isEnemyHasFlyerAttacker()
{
	for (auto eU : enemyAllBattleUnit)
	{
		if (eU.first.isFlyer() && 
			(eU.first.airWeapon() != BWAPI::WeaponTypes::None))
		{
			return true;
		}
	}
	return false;
}

// TODO: update enemy occupied region
void InformationManager::onUnitDestroy(BWAPI::Unit unit)
{
	// sparcraft
	if (unit->getPlayer() == _self || unit->getPlayer() == _enemy)
	{
		_unitData[unit->getPlayer()].removeUnit(unit);

		if (unit->getType().isResourceDepot())
		{
			baseLost(unit->getTilePosition());
		}
	}


	//original
	if (unit->getPlayer() == BWAPI::Broodwar->enemy())
	{
		if (unit->getType().isBuilding())
		{
			addUnitInfluenceMap(unit, false);

			destroyOccupiedRegionsDetail(BWTA::getRegion(unit->getPosition()), BWAPI::Broodwar->enemy(), unit);

			if (unit->getType().isResourceDepot())
			{
				BOOST_FOREACH(BWTA::BaseLocation* base, BWTA::getBaseLocations())
				{
					//check if it is not our base
					if (base->getTilePosition().getDistance(unit->getTilePosition()) < 3)
					{
						enemyAllBase.erase(unit);
						break;
					}
				}
			}
			if (enemyAllBuilding.find(unit->getType()) != enemyAllBuilding.end())
				enemyAllBuilding[unit->getType()].erase(unit);
		}
		else
		{
			if (enemyAllBattleUnit.find(unit->getType()) != enemyAllBattleUnit.end())
				enemyAllBattleUnit[unit->getType()].erase(unit);
		}
	}
	else if (unit->getPlayer() == BWAPI::Broodwar->self())
	{
		if (unit->getType().isBuilding())
		{
			destroyOccupiedRegionsDetail(BWTA::getRegion(unit->getPosition()), BWAPI::Broodwar->self(), unit);
			
			// erase the place position.
			eraseBuildPosition(unit->getTilePosition());

			if (unit->getType().isResourceDepot())
			{
				BOOST_FOREACH(BWTA::BaseLocation* base, BWTA::getBaseLocations())
				{
					int resource_distance = 32 * 10;
					//check if it is not our base
					if (base->getTilePosition().getDistance(unit->getTilePosition()) < resource_distance)
					{
						//for ucb1 simulation
						if (BWTA::getRegion(unit->getTilePosition()) == BWTA::getRegion(getOurNatrualLocation())
							&& BWAPI::Broodwar->getFrameCount() < 7000)
						{
							enemyEarlyRushSuccess = true;
						}

						selfAllBase.erase(unit);
						// find out another base 
						for (auto localUnit : unit->getUnitsInRadius(resource_distance))
						{
							if (localUnit->getType().isResourceDepot())
							{
								selfAllBase.insert(localUnit);
								break;
							}
						}

						break;
					}
				}
			}

			if (selfAllBuilding.find(unit->getType()) != selfAllBuilding.end())
				selfAllBuilding[unit->getType()].erase(unit);
		}
		else
		{
			if (selfAllBattleUnit.find(unit->getType()) != selfAllBattleUnit.end())
				selfAllBattleUnit[unit->getType()].erase(unit);
		}
	}
}


BWAPI::Unit InformationManager::GetOurBaseUnit()
{
	if (selfBaseUnit->exists())
		return selfBaseUnit;
	else
		return NULL;
}


InformationManager& InformationManager::Instance()
{
	static InformationManager instance;
	return instance;
}


void InformationManager::updateOccupiedRegions(BWTA::Region * region, BWAPI::Player player)
{
	// if the region is valid (flying buildings may be in NULL regions)
	if (region)
	{
		// add it to the list of occupied regions, 0 is ourselves.
		if (player == BWAPI::Broodwar->self())
			occupiedRegions[0].insert(region);
		else
			occupiedRegions[1].insert(region);

		_occupiedRegions[player].insert(region);
	}
}


void InformationManager::addOccupiedRegionsDetail(BWTA::Region * region, BWAPI::Player player, BWAPI::Unit building)
{
	if (region)
	{
		if (player == BWAPI::Broodwar->self())
		{
			occupiedRegions[0].insert(region);

			if (selfOccupiedDetail.find(region) != selfOccupiedDetail.end())
				selfOccupiedDetail[region][building] = buildingInfo(building, building->getTilePosition(), building->isCompleted(), building->getType());
			else
			{
				selfOccupiedDetail[region] = std::map<BWAPI::Unit, buildingInfo>();
				selfOccupiedDetail[region][building] = buildingInfo(building, building->getTilePosition(), building->isCompleted(), building->getType());
			}
		}
		else
		{
			occupiedRegions[1].insert(region);
			if (enemyOccupiedDetail.find(region) != enemyOccupiedDetail.end())
				enemyOccupiedDetail[region][building] = buildingInfo(building, building->getTilePosition(), building->isCompleted(), building->getType());
			else
			{
				enemyOccupiedDetail[region] = std::map<BWAPI::Unit, buildingInfo>();
				enemyOccupiedDetail[region][building] = buildingInfo(building, building->getTilePosition(), building->isCompleted(), building->getType());
			}
		}
	}
}


void InformationManager::destroyOccupiedRegionsDetail(BWTA::Region * region, BWAPI::Player player, BWAPI::Unit building)
 {
	if (region)
	{
		if (player == BWAPI::Broodwar->self())
		{
			if (selfOccupiedDetail.find(region) != selfOccupiedDetail.end())
			{	
				selfOccupiedDetail[region].erase(building);
				if (selfOccupiedDetail[region].size() == 0)
				{
					occupiedRegions[0].erase(region);
					selfOccupiedDetail.erase(region);
				}
				/*
				if (selfOccupiedDetail[region].size() == 1 && (*selfOccupiedDetail[region].begin()).second.unitType.isRefinery())
				{
					selfOccupiedDetail[region].erase(selfOccupiedDetail[region].begin());
					occupiedRegions[0].erase(region);
					selfOccupiedDetail.erase(region);
				}*/
			}
		}
		else
		{
			if (enemyOccupiedDetail.find(region) != enemyOccupiedDetail.end())
			{
				enemyOccupiedDetail[region].erase(building);
				if (enemyOccupiedDetail[region].size() == 0)
				{
					occupiedRegions[1].erase(region);
					enemyOccupiedDetail.erase(region);
				} 

				/*
				if (enemyOccupiedDetail[region].size() == 1 && (*enemyOccupiedDetail[region].begin()).second.unitType.isRefinery())
				{
					enemyOccupiedDetail[region].erase(enemyOccupiedDetail[region].begin());
					occupiedRegions[1].erase(region);
					enemyOccupiedDetail.erase(region);
				}*/
			}
		}
	}
}


std::set<BWTA::Region *> & InformationManager::getOccupiedRegions(BWAPI::Player player)
{
	// only return the player region
	return _occupiedRegions[player];
}

void InformationManager::addPlaceBuildPosition(BWAPI::TilePosition pos)
{
	bool is_in = false;
	for (std::vector<BWAPI::TilePosition>::iterator it = placeBuildPositions.begin(); it != placeBuildPositions.end(); it++)
	{
		if (*it == pos)
		{
			is_in = true;
			break;
		}
	}

	if (! is_in)
		placeBuildPositions.push_back(pos);
}

bool InformationManager::isPlaceEmpty(BWAPI::TilePosition pos)
{
	bool flag = true;
	for (std::vector<BWAPI::TilePosition>::iterator it = placeBuildPositions.begin(); it != placeBuildPositions.end(); it++)
	{
		if (*it == pos)
		{
			flag = false;
			break;
		}
	}

	return flag;
}

void InformationManager::eraseBuildPosition(BWAPI::TilePosition pos)
{
	for (std::vector<BWAPI::TilePosition>::iterator it = placeBuildPositions.begin(); it != placeBuildPositions.end(); it++)
	{
		if (*it == pos)
		{
			placeBuildPositions.erase(it);
			break;
		}
	}
}

void InformationManager::printInfo()
{
	for (auto & unit : BWAPI::Broodwar->self()->getUnits())
	{
		int sx = 440;
		int sy = 30;
		int w = 180;
		int h = 120;//+ 15 * unit->Count.size();

		BWAPI::Broodwar->drawBoxScreen(sx - 2, sy, sx + w + 2, sy + h, BWAPI::Colors::Black, true);
		BWAPI::Broodwar->drawTextScreen(sx + 4, sy, "\x03%s", unit->getType().c_str());
		BWAPI::Broodwar->drawLineScreen(sx, sy + 14, sx + w, sy + 14, BWAPI::Colors::Orange);

		BWAPI::Broodwar->drawTextScreen(sx + 2, sy + 15, "Id: \x11%d", unit->getID());
		BWAPI::Broodwar->drawTextScreen(sx + 2, sy + 30, "Goal: \x11(%d,%d)", unit->getTargetPosition().x, unit->getTargetPosition().y);

		std::string str1 = "Ground ";
		if (unit->getType().airWeapon()) str1 = "Air ";
		

		std::string str2 = "";
		/*if (isOffensive()) str2 = "Offensive";
		if (isDefensive()) str2 = "Defensive";
		if (isBunkerDefend()) str2 = "Bunker";
		if (isExplorer()) str2 = "Explorer";
		if (isRush()) str2 = "Rush";*/

		//Broodwar->drawTextScreen(sx + 2, sy + 45, "Type: \x11%s%s", str1.c_str(), str2.c_str());
		
//		Broodwar->drawLineScreen(sx, sy + 119 + 15 * no, sx + w, sy + 119 + 15 * no, Colors::Orange);
	}
}



void InformationManager::getNearbyForce(std::vector<UAlbertaBot::UnitInfo> & unitInfo, BWAPI::Position p, BWAPI::Player player, int radius)
{
	// for each unit we know about for that player
	for (const auto & kv : getUnitData(player).getUnits())
	{
		const UAlbertaBot::UnitInfo & ui(kv.second);

		// if it's a combat unit we care about
		// and it's finished! 
		if (isCombatUnit(ui.type) && ui.completed)
		{
			// Determine its attack range, plus a small fudge factor.
			// TODO find range using the same method as UnitUtil
			int range = 0;
			if (ui.type.groundWeapon() != BWAPI::WeaponTypes::None)
			{
				range = ui.type.groundWeapon().maxRange() + 40;
			}
			// NOTE Ignores air weapon! Units with air attack only must be inside the radius to be
			// included (except turrets and spores, because they are also detectors).

			// if it can attack into the radius we care about
			if (ui.lastPosition.getDistance(p) <= (radius + range))
			{
				// add it to the vector
				unitInfo.push_back(ui);
			}
		}
		else if (ui.type.isDetector() && ui.lastPosition.getDistance(p) <= (radius + 250))
		{
			unitInfo.push_back(ui);
		}
	}
}

bool InformationManager::isCombatUnit(BWAPI::UnitType type) const
{
	return
		type.canAttack() ||         // NOTE: excludes spellcasters
		type == BWAPI::UnitTypes::Terran_Medic ||
		type == BWAPI::UnitTypes::Terran_Bunker ||
		type.isDetector();
}

BWTA::BaseLocation * InformationManager::getMyMainBaseLocation()
{
	UAB_ASSERT(_mainBaseLocations[_self], "No base location");
	return _mainBaseLocations[_self];
}

BWTA::BaseLocation * InformationManager::getEnemyMainBaseLocation()
{
	return _mainBaseLocations[_enemy];
}

void InformationManager::chooseNewMainBase()
{
	BWTA::BaseLocation * oldMain = getMyMainBaseLocation();

	// Choose a base we own which is as far away from the old main as possible.
	// Maybe that will be safer.
	double newMainDist = 0.0;
	BWTA::BaseLocation * newMain = nullptr;

	for (BWTA::BaseLocation * base : BWTA::getBaseLocations())
	{
		if (_theBases[base].owner == _self)
		{
			double dist = base->getAirDistance(oldMain);
			if (dist > newMainDist)
			{
				newMainDist = dist;
				newMain = base;
			}
		}
	}

	// If we didn't find a new main base, we're in deep trouble. We may as well keep the old one.
	if (newMain)
	{
		_mainBaseLocations[_self] = newMain;
	}
}

void InformationManager::updateBaseLocationInfo()
{
	_occupiedRegions[_self].clear();
	_occupiedRegions[_enemy].clear();

	// if we haven't found the enemy main base location yet
	if (!_mainBaseLocations[_enemy])
	{
		// how many start locations have we explored
		int exploredStartLocations = 0;
		bool baseFound = false;

		// an undexplored base location holder
		BWTA::BaseLocation * unexplored = nullptr;

		for (BWTA::BaseLocation * startLocation : BWTA::getStartLocations())
		{
			if (isEnemyBuildingInRegion(BWTA::getRegion(startLocation->getTilePosition())))
			{
				updateOccupiedRegions(BWTA::getRegion(startLocation->getTilePosition()), _enemy);

				// On a competition map, our base and the enemy base will never be in the same region.
				// If we find an enemy building in our region, it's a proxy.
				if (startLocation == getMyMainBaseLocation())
				{
				}
				else
				{
					baseFound = true;
					_mainBaseLocations[_enemy] = startLocation;
					baseInferred(startLocation);
				}
			}

			// if it's explored, increment
			// TODO If the enemy is zerg, we can be a little quicker by looking for creep.
			// TODO If we see a mineral patch that has been mined, that should be a base.
			if (BWAPI::Broodwar->isExplored(startLocation->getTilePosition()))
			{
				exploredStartLocations++;

				// otherwise set the unexplored base
			}
			else
			{
				unexplored = startLocation;
			}
		}

		// if we've explored every start location except one, it's the enemy
		if (!baseFound && exploredStartLocations == ((int)BWTA::getStartLocations().size() - 1))
		{
		
			_mainBaseLocations[_enemy] = unexplored;
			baseInferred(unexplored);
			updateOccupiedRegions(BWTA::getRegion(unexplored->getTilePosition()), _enemy);
		}
		// otherwise we do know it, so push it back
	}
	else
	{
		updateOccupiedRegions(BWTA::getRegion(_mainBaseLocations[_enemy]->getTilePosition()), _enemy);
	}

	// The enemy occupies a region if it has a building there.
	for (const auto & kv : _unitData[_enemy].getUnits())
	{
		const UAlbertaBot::UnitInfo & ui(kv.second);
		BWAPI::UnitType type = ui.type;

		if (type.isBuilding())
		{
			updateOccupiedRegions(BWTA::getRegion(BWAPI::TilePosition(ui.lastPosition)), _enemy);
		}
	}

	// We occupy a region if we have a building there.
	for (const auto & kv : _unitData[_self].getUnits())
	{
		const UAlbertaBot::UnitInfo & ui(kv.second);
		BWAPI::UnitType type = ui.type;

		if (type.isBuilding() && ui.completed)
		{
			updateOccupiedRegions(BWTA::getRegion(BWAPI::TilePosition(ui.lastPosition)), _self);
		}
	}
}

bool InformationManager::isEnemyBuildingInRegion(BWTA::Region * region)
{
	// invalid regions aren't considered the same, but they will both be null
	if (!region)
	{
		return false;
	}

	for (const auto & kv : _unitData[_enemy].getUnits())
	{
		const UAlbertaBot::UnitInfo & ui(kv.second);
		if (ui.type.isBuilding())
		{
			if (BWTA::getRegion(BWAPI::TilePosition(ui.lastPosition)) == region)
			{
				return true;
			}
		}
	}

	return false;
}

// A base is inferred to exist at the given position, without having been seen.
// Only enemy bases can be inferred; we see our own.
// Adjust its BaseStatus to match. It is not reserved.
void InformationManager::baseInferred(BWTA::BaseLocation * base)
{
	if (_theBases[base].owner != _self)
	{
		_theBases[base] = BaseStatus(nullptr, _enemy, false);
	}
}

// The given resource depot has been created or discovered.
// Adjust its BaseStatus to match. It is not reserved.
// This accounts for the theoretical case that it might be neutral.
void InformationManager::baseFound(BWAPI::Unit depot)
{
	UAB_ASSERT(depot->getType().isResourceDepot(), "non-depot base");

	BWAPI::Player owner = BWAPI::Broodwar->neutral();

	if (depot->getPlayer() == _self || depot->getPlayer() == _enemy)
	{
		owner = depot->getPlayer();
	}

	for (BWTA::BaseLocation * base : BWTA::getBaseLocations())
	{
		if (closeEnough(base->getTilePosition(), depot->getTilePosition()))
		{
			_theBases[base] = BaseStatus(depot, owner, false);
			return;
		}
	}
}

// The two possible base positions are close enough together
// that we can say they are "the same place" for a base.
bool InformationManager::closeEnough(BWAPI::TilePosition a, BWAPI::TilePosition b)
{
	return abs(a.x - b.x) <= 4 && abs(a.y - b.y) <= 4;
}

// Something that may be a base was just destroyed.
// If it is, update the BaseStatus to match.
void InformationManager::baseLost(BWAPI::TilePosition basePosition)
{
	for (BWTA::BaseLocation * base : BWTA::getBaseLocations())
	{
		if (closeEnough(base->getTilePosition(), basePosition))
		{
			_theBases[base] = BaseStatus();
			if (base == getMyMainBaseLocation())
			{
				chooseNewMainBase();        // in case our main was destroyed
			}
			return;
		}
	}
}

// The natural base, whether it is taken or not.
// May be null on some maps.
BWTA::BaseLocation * InformationManager::getMyNaturalLocation()
{
	return _myNaturalBaseLocation;
}

// Self, enemy, or neutral.
BWAPI::Player InformationManager::getBaseOwner(BWTA::BaseLocation * base)
{
	return _theBases[base].owner;
}

const UAlbertaBot::UIMap & InformationManager::getUnitInfo(BWAPI::Player player) const
{
	return getUnitData(player).getUnits();
}

const UAlbertaBot::UnitData & InformationManager::getUnitData(BWAPI::Player player) const
{
	return _unitData.find(player)->second;
}


// We have complated combat units (excluding workers).
bool InformationManager::weHaveCombatUnits()
{
	// Latch: Once we have combat units, pretend we always have them.
	if (_weHaveCombatUnits)
	{
		return true;
	}

	for (const auto u : _self->getUnits())
	{
		if (!u->getType().isWorker() &&
			!u->getType().isBuilding() &&
			u->isCompleted() &&
			u->getType() != BWAPI::UnitTypes::Zerg_Larva &&
			u->getType() != BWAPI::UnitTypes::Zerg_Overlord)
		{
			_weHaveCombatUnits = true;
			return true;
		}
	}

	return false;
}

bool InformationManager::isBaseReserved(BWTA::BaseLocation *base)
{
	return _theBases[base].reserved;
}

void InformationManager::reserveBase(BWTA::BaseLocation * base)
{
	_theBases[base].reserved = true;
}