#include "Common.h"
#include "BuildingPlacer.h"
#include "MapGrid.h"
#include "MapTools.h"

using namespace UAlbertaBot;

BuildingPlacer::BuildingPlacer()
	: _boxTop(999999)
	, _boxBottom(std::numeric_limits<int>::lowest())
	, _boxLeft(999999)
	, _boxRight(std::numeric_limits<int>::lowest())
{
	_reserveMap = std::vector< std::vector<bool> >(BWAPI::Broodwar->mapWidth(), std::vector<bool>(BWAPI::Broodwar->mapHeight(), false));

	computeResourceBox();
}

BuildingPlacer & BuildingPlacer::Instance()
{
	static BuildingPlacer instance;
	return instance;
}

bool BuildingPlacer::isInResourceBox(int x, int y) const
{
	int posX(x * 32);
	int posY(y * 32);

	return (posX >= _boxLeft) && (posX < _boxRight) && (posY >= _boxTop) && (posY < _boxBottom);
}

void BuildingPlacer::computeResourceBox()
{
	BWAPI::Position start(InformationManager::Instance().getMyMainBaseLocation()->getPosition());
	BWAPI::Unitset unitsAroundNexus;

	for (auto & unit : BWAPI::Broodwar->getAllUnits())
	{
		// if the units are less than 400 away add them if they are resources
		if (unit->getDistance(start) < 300 && unit->getType().isMineralField())
		{
			unitsAroundNexus.insert(unit);
		}
	}

	for (auto & unit : unitsAroundNexus)
	{
		int x = unit->getPosition().x;
		int y = unit->getPosition().y;

		int left = x - unit->getType().dimensionLeft();
		int right = x + unit->getType().dimensionRight() + 1;
		int top = y - unit->getType().dimensionUp();
		int bottom = y + unit->getType().dimensionDown() + 1;

		_boxTop = top < _boxTop ? top : _boxTop;
		_boxBottom = bottom > _boxBottom ? bottom : _boxBottom;
		_boxLeft = left < _boxLeft ? left : _boxLeft;
		_boxRight = right > _boxRight ? right : _boxRight;
	}

	//BWAPI::Broodwar->printf("%d %d %d %d", boxTop, boxBottom, boxLeft, boxRight);
}

// makes final checks to see if a building can be built at a certain location
bool BuildingPlacer::canBuildHere(BWAPI::TilePosition position, const Building & b) const
{
	/*if (!b.type.isRefinery() && !InformationManager::Instance().tileContainsUnit(position))
	{
	return false;
	}*/

	if (!BWAPI::Broodwar->canBuildHere(position, b.type, b.builderUnit))
	{
		return false;
	}

	// check the reserve map
	for (int x = position.x; x < position.x + b.type.tileWidth(); x++)
	{
		for (int y = position.y; y < position.y + b.type.tileHeight(); y++)
		{
			if (_reserveMap[x][y])
			{
				return false;
			}
		}
	}

	// if it overlaps a base location return false
	if (tileOverlapsBaseLocation(position, b.type))
	{
		return false;
	}

	return true;
}

bool BuildingPlacer::tileBlocksAddon(BWAPI::TilePosition position) const
{

	for (int i = 0; i <= 2; ++i)
	{
		for (auto unit : BWAPI::Broodwar->getUnitsOnTile(position.x - i, position.y))
		{
			if (unit->getType() == BWAPI::UnitTypes::Terran_Command_Center ||
				unit->getType() == BWAPI::UnitTypes::Terran_Factory ||
				unit->getType() == BWAPI::UnitTypes::Terran_Starport ||
				unit->getType() == BWAPI::UnitTypes::Terran_Science_Facility)
			{
				return true;
			}
		}
	}

	return false;
}

// Can we build this building here with the specified amount of space?
// Space value is buildDist. horizontalOnly means only horizontal spacing.
bool BuildingPlacer::canBuildHereWithSpace(BWAPI::TilePosition position, const Building & b, int buildDist, bool horizontalOnly) const
{
	//if we can't build here, we of course can't build here with space
	if (!canBuildHere(position, b))
	{
		return false;
	}

	// height and width of the building
	int width(b.type.tileWidth());
	int height(b.type.tileHeight());

	//make sure we leave space for add-ons. These types of units can have addons:
	if (b.type == BWAPI::UnitTypes::Terran_Command_Center ||
		b.type == BWAPI::UnitTypes::Terran_Factory ||
		b.type == BWAPI::UnitTypes::Terran_Starport ||
		b.type == BWAPI::UnitTypes::Terran_Science_Facility)
	{
		width += 2;
	}

	// define the rectangle of the building spot
	int startx = position.x - buildDist;
	int starty = position.y - buildDist;
	int endx = position.x + width + buildDist;
	int endy = position.y + height + buildDist;

	if (b.type.isAddon())
	{
		const BWAPI::UnitType builderType = b.type.whatBuilds().first;

		BWAPI::TilePosition builderTile(position.x - builderType.tileWidth(), position.y + 2 - builderType.tileHeight());

		startx = builderTile.x - buildDist;
		starty = builderTile.y - buildDist;
		endx = position.x + width + buildDist;
		endy = position.y + height + buildDist;
	}

	if (horizontalOnly)
	{
		starty += buildDist;
		endy -= buildDist;
	}

	// if this rectangle doesn't fit on the map we can't build here
	if (startx < 0 || starty < 0 || endx > BWAPI::Broodwar->mapWidth() || endx < position.x + width || endy > BWAPI::Broodwar->mapHeight())
	{
		return false;
	}

	// if space is reserved, or it's in the resource box, we can't build here
	for (int x = startx; x < endx; x++)
	{
		for (int y = starty; y < endy; y++)
		{
			if (!b.type.isRefinery())
			{
				if (!buildable(b, x, y) || _reserveMap[x][y] || ((b.type != BWAPI::UnitTypes::Protoss_Photon_Cannon) && isInResourceBox(x, y)))
				{
					return false;
				}
			}
		}
	}

	return true;
}

//BWAPI::TilePosition BuildingPlacer::getBuildLocationNear(const Building & b, int buildDist, bool horizontalOnly) const
//{
//	UAlbertaBot::SparCraft::Timer t;
//	t.start();
//
//	// BWAPI::Broodwar->printf("Building Placer seeks position near %d, %d", b.desiredPosition.x, b.desiredPosition.y);
//
//	// get the precomputed vector of tile positions which are sorted closest to this location
//	const std::vector<BWAPI::TilePosition> & closestToBuilding = MapTools::Instance().getClosestTilesTo(BWAPI::Position(b.desiredPosition));
//
//	double ms1 = t.getElapsedTimeInMilliSec();
//
//	// iterate through the list until we've found a suitable location
//	for (size_t i(0); i < closestToBuilding.size(); ++i)
//	{
//		if (canBuildHereWithSpace(closestToBuilding[i], b, buildDist, horizontalOnly))
//		{
//			double ms = t.getElapsedTimeInMilliSec();
//			// BWAPI::Broodwar->printf("Building Placer took %d iterations, lasting %lf ms @ %lf iterations/ms, %lf setup ms", i, ms, (i / ms), ms1);
//			// BWAPI::Broodwar->printf("Building Placer took %d iterations, lasting %lf ms, finding %d, %d", i, ms, closestToBuilding[i].x, closestToBuilding[i].y);
//
//			return closestToBuilding[i];
//		}
//	}
//
//	double ms = t.getElapsedTimeInMilliSec();
//	// BWAPI::Broodwar->printf("Building Placer took %lf ms, found nothing", ms);
//
//	return  BWAPI::TilePositions::None;
//}

bool BuildingPlacer::tileOverlapsBaseLocation(BWAPI::TilePosition tile, BWAPI::UnitType type) const
{
	// if it's a resource depot we don't care if it overlaps
	if (type.isResourceDepot())
	{
		return false;
	}

	// dimensions of the proposed location
	int tx1 = tile.x;
	int ty1 = tile.y;
	int tx2 = tx1 + type.tileWidth();
	int ty2 = ty1 + type.tileHeight();

	// for each base location
	for (BWTA::BaseLocation * base : BWTA::getBaseLocations())
	{
		// dimensions of the base location
		int bx1 = base->getTilePosition().x;
		int by1 = base->getTilePosition().y;
		int bx2 = bx1 + BWAPI::Broodwar->self()->getRace().getCenter().tileWidth();
		int by2 = by1 + BWAPI::Broodwar->self()->getRace().getCenter().tileHeight();

		// conditions for non-overlap are easy
		bool noOverlap = (tx2 < bx1) || (tx1 > bx2) || (ty2 < by1) || (ty1 > by2);

		// if the reverse is true, return true
		if (!noOverlap)
		{
			return true;
		}
	}

	// otherwise there is no overlap
	return false;
}

bool BuildingPlacer::buildable(const Building & b, int x, int y) const
{
	BWAPI::TilePosition tp(x, y);

	if (!tp.isValid())
	{
		return false;
	}

	if (!BWAPI::Broodwar->isBuildable(x, y, true))
	{
		// Unbuildable according to the map, or because the location is blocked
		// by a visible building. Unseen buildings (even if known) are "buildable" on.
		return false;
	}

	if (BWAPI::Broodwar->self()->getRace() == BWAPI::Races::Terran && tileBlocksAddon(tp))
	{
		return false;
	}

	// getUnitsOnTile() only returns visible units, even if they are buildings.
	for (auto & unit : BWAPI::Broodwar->getUnitsOnTile(x, y))
	{
		if ((b.builderUnit != nullptr) && (unit != b.builderUnit))
		{
			return false;
		}
	}

	return true;
}

void BuildingPlacer::reserveTiles(BWAPI::TilePosition position, int width, int height)
{
	int rwidth = _reserveMap.size();
	int rheight = _reserveMap[0].size();
	for (int x = position.x; x < position.x + width && x < rwidth; x++)
	{
		for (int y = position.y; y < position.y + height && y < rheight; y++)
		{
			_reserveMap[x][y] = true;
		}
	}
}

void BuildingPlacer::drawReservedTiles()
{


	int rwidth = _reserveMap.size();
	int rheight = _reserveMap[0].size();

	for (int x = 0; x < rwidth; ++x)
	{
		for (int y = 0; y < rheight; ++y)
		{
			if (_reserveMap[x][y] || isInResourceBox(x, y))
			{
				int x1 = x * 32 + 8;
				int y1 = y * 32 + 8;
				int x2 = (x + 1) * 32 - 8;
				int y2 = (y + 1) * 32 - 8;

				BWAPI::Broodwar->drawBoxMap(x1, y1, x2, y2, BWAPI::Colors::Yellow, false);
			}
		}
	}
}

void BuildingPlacer::freeTiles(BWAPI::TilePosition position, int width, int height)
{
	int rwidth = _reserveMap.size();
	int rheight = _reserveMap[0].size();

	for (int x = position.x; x < position.x + width && x < rwidth; x++)
	{
		for (int y = position.y; y < position.y + height && y < rheight; y++)
		{
			_reserveMap[x][y] = false;
		}
	}
}

BWAPI::TilePosition BuildingPlacer::getRefineryPosition()
{
	BWAPI::TilePosition closestGeyser = BWAPI::TilePositions::None;
	int minGeyserDistanceFromHome = 100000;
	BWAPI::Position homePosition = InformationManager::Instance().getMyMainBaseLocation()->getPosition();

	// NOTE In BWAPI 4.2.1 getStaticGeysers() has a bug affecting geysers whose refineries
	// have been canceled or destroyed: They become inaccessible. https://github.com/bwapi/bwapi/issues/697
	// TODO still trying to work around the bug
	// TODO could rewrite this to use BWTA instead of BWAPI; would not need the slow nested loop
	//      loop through our bases (from InfoMan), then pick a geyser at that base
	// for (auto & geyser : BWAPI::Broodwar->getStaticGeysers())
	// for (auto & geyser : BWAPI::Broodwar->getAllUnits())
	for (auto & geyser : BWAPI::Broodwar->getGeysers())
	{
		//if (geyser->getType() != BWAPI::UnitTypes::Resource_Vespene_Geyser)
		//{
		//	continue;
		//}
		//BWAPI::Broodwar->printf("geyser (type %d) @ %d, %d", BWAPI::UnitTypes::Resource_Vespene_Geyser, geyser->getInitialTilePosition().x, geyser->getInitialTilePosition().y);

		// check to see if it's near one of our depots
		bool nearDepot = false;
		for (auto & unit : BWAPI::Broodwar->self()->getUnits())
		{
			if (unit->getType().isResourceDepot() && unit->getDistance(geyser) < 300)
			{
				nearDepot = true;
				break;
			}
		}

		if (nearDepot)
		{
			int homeDistance = geyser->getDistance(homePosition);
			//BWAPI::Broodwar->printf("our geyser @ %d, %d, dist %d", geyser->getInitialTilePosition().x, geyser->getInitialTilePosition().y, homeDistance);

			if (homeDistance < minGeyserDistanceFromHome)
			{
				minGeyserDistanceFromHome = homeDistance;
				closestGeyser = geyser->getInitialTilePosition();
			}
		}
	}

	return closestGeyser;
}

bool BuildingPlacer::isReserved(int x, int y) const
{
	int rwidth = _reserveMap.size();
	int rheight = _reserveMap[0].size();
	if (x < 0 || y < 0 || x >= rwidth || y >= rheight)
	{
		return false;
	}

	return _reserveMap[x][y];
}

