/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#pragma once

#include "upc.h"

namespace fairrsh {

class Module;
class State;

/**
 * Base class for UPC filters.
 *
 * UPC filters provide a `filter` method that takes a UPC and returns either the
 * same UPC, a new UPC or nullptr. The incoming UPC will never be modified. If
 * it returns nullptr, the corresponding UPC should not be posted in the
 * blackboard.
 *
 * Possible use cases for UPC filters are consistency checks (which would log a
 * warning if simple conditions are violated, for example) or enforcing unit
 * allocations according to a set of rules based on which tasks are currently
 * holding on to units. See below for a concrete application of the latter.
 */
class UPCFilter {
 public:
  UPCFilter() {}
  virtual ~UPCFilter() {}

  virtual std::shared_ptr<UPCTuple> filter(
      State* state,
      std::shared_ptr<UPCTuple> upc,
      Module* origin) = 0;
};

/**
 * Removes units from an UPC that are allocated to high-priority tasks.
 *
 * If the UPC will end up without any units, nullptr will be returned.
 */
class AssignedUnitsFilter : public UPCFilter {
 public:
  using UPCFilter::UPCFilter;
  virtual ~AssignedUnitsFilter() {}

  virtual std::shared_ptr<UPCTuple> filter(
      State* state,
      std::shared_ptr<UPCTuple> upc,
      Module *origin) override;
};

} // namespace fairrsh
