/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#pragma once

#include <unordered_map>

#ifdef WITH_ATEN
#include <ATen/ATen.h>
#endif // WITH_ATEN

#include "buildtype.h"
#include "fairrsh.h"

namespace fairrsh {

class State;
struct Area;
struct BuildType;
struct Unit;

/**
 * Abstract "meta" commands for UPCTuple below.
 */
enum Command {
  Create = 0,
  Move,
  Delete,
  Gather,
  Scout,
  Cancel,
  MAX,
};

/**
 * Who, where, what action tuple (unit, position, command).
 *
 * UPCTuples are used for module-to-module communication (via the Blackboard).
 * Posting a UPCTuple to the Blackboard is akin to requesting a specific action
 * from another module, and consuming a UPCTuple from the Blackboard implies
 * that the consumer implements this action. The implementation can also consist
 * of refining the UPCTuple so that a more lower-level module can actually
 * execute it.
 *
 * For example, a build order module might post a UPCTuple to create a certain
 * unit type, a builder module might wait until their are sufficient resources
 * before consuming it and then select a worker and a location. The
 * UPCToCommandModule takes care of translating executable UPCTuples (with sharp
 * unit, position and command entries) to actual game commands.
 */
struct UPCTuple {
  /// Maps units to probabilities of being selected.
  std::unordered_map<Unit*, float> unit;

#ifdef WITH_ATEN
  /// A distribution over the whole map.
  at::Tensor position;
#else // WITH_ATEN
  struct UndefTensor {
    bool defined() const {
      return false;
    }
    template <typename T>
    UndefTensor& fill_(T v) {
      return *this;
    }
  };
  UndefTensor position;
#endif // WITH_ATEN

  /// A distribution over units which can be used instead of position. Sometimes
  /// it is more efficient to specify target units directly (e.g. for resource
  /// mining or for attacks).
  std::unordered_map<Unit*, float> positionU;

  /// A single x/y position with one probability. This is more efficient then
  /// having a one-hot position tensor.
  Position positionS = {-1, -1};

  /// An area so that every position in this area has probability 1, and every
  /// position outside has probability 0.
  Area* positionA = nullptr;

  /// A distribution over possible abstract commands.
  float command[Command::MAX] = {0};

  /// A distribution over unit types relating to the Create command.
  std::unordered_map<BuildType const*, float> createType;

  /// Specifies the (inverse) scale of the position tensor. For efficiency
  /// reasons, positions can be specified at a coarser resolution.
  int scale = 1;

#ifdef WITH_ATEN
  /// An optional state tensor that can be used to provide additional
  /// information to downstream modules.
  at::Tensor state;
#endif // WITH_ATEN

  /* Methods */

  /// Returns the probability of a given position
  float positionProb(int x, int y) const;

#ifdef WITH_ATEN
  /// Creates a map-sized tensor filled with zeros
  static at::Tensor zeroPosition(State* state, int scale = 1);
#else // WITH_ATEN
  static UndefTensor zeroPosition(State* state, int scale = 1) {
    return UndefTensor();
  }
#endif // WITH_ATEN
};

} // namespace fairrsh
