/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#pragma once

#include "basetypes.h"

#include <unordered_map>
#include <vector>

namespace fairrsh {

class State;
struct Unit;
struct BuildType;

/**
 * Represents a tile on the map.
 *
 * buildable and height are static map data, the rest are updated throughout
 * the game. All fields might not be updated immediately every frame, but most
 * things should only have a few frames of delay in the worst case.
 */

struct Tile {
  /// X position of tile in walk tiles
  int x = 0;
  /// Y position of tile in walk tiles
  int y = 0;
  bool visible = false;
  bool buildable = false;
  /// Set by BuilderModule to help with planning building placement.
  bool reservedAsUnbuildable = false;
  bool hasCreep = false; // todo
  /// Indicates that this tile is in the mineral line and that buildings
  /// should not be placed here.
  bool reservedForGathering = false;
  /// Indicates that this tile is unbuildable for resource depots (too close
  /// to resources).
  bool resourceDepotUnbuildable = false;
  /// Indicates that this is a resource depot tile at an expansion, and should
  /// not be occupied by regular buildings.
  bool reservedForResourceDepot = false;
  /// Special field that can be set if a particular tile should not be used
  /// for buildings until this frame. Usually set by BuilderModule when trying
  /// and, for an unknown reason, failing to build something, such that we can
  /// try to build somewhere else.
  FrameNum blockedUntil = 0;
  /// The building that is currently occupying this tile. There might be other
  /// (stacked) buildings too.
  Unit* building = nullptr;
  /// Every walk tile within this tile is walkable. This usually means that
  /// any unit can pass through this tile, ignoring buildings or other units.
  bool entirelyWalkable = false;
  int height = 0;
  FrameNum lastSeen = 0;
};

/**
 * Manages and updates all of the tiles.
 *
 * That's what it does.
 */

class TilesInfo {
 public:
  TilesInfo(State* state_);

  void preUnitsUpdate();
  void postUnitsUpdate();

  unsigned mapTileWidth() const {
    return mapTileWidth_;
  }

  unsigned mapTileHeight() const {
    return mapTileHeight_;
  }

  static const unsigned tilesWidth = 256;
  static const unsigned tilesHeight = 256;

  Tile& getTile(int x, int y);
  const Tile& getTile(int x, int y) const;
  Tile* tryGetTile(int x, int y);
  const Tile* tryGetTile(int x, int y) const;

  /// This sets reservedAsUnbuildable in the tiles that would be occupied
  /// by the specified building at the specified build location (upper left
  /// corner coordinates). Should only be used by BuilderModule.
  void reserveArea(const BuildType* type, int x, int y);
  // Complements reserveArea.
  void unreserveArea(const BuildType* type, int x, int y);

  /// All the tiles. Prefer to use getTile, this is only here in case it is
  /// needed for performance.
  std::vector<Tile> tiles;

 protected:
  struct TileOccupyingBuilding {
    Unit* u;
    const BuildType* type;
    int pixelX;
    int pixelY;
    std::vector<Tile*> tiles;
  };

  unsigned mapTileWidth_ = 0;
  unsigned mapTileHeight_ = 0;

  std::unordered_map<const Unit*, TileOccupyingBuilding>
      tileOccupyingBuildings_;

  State* state_ = nullptr;
  FrameNum lastSlowTileUpdate_ = 0;
  FrameNum lastUpdateBuildings_ = 0;
  FrameNum lastFowCreepUpdate_ = 0;
};

} // namespace fairrsh
