/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#include "task.h"

#include "state.h"
#include "tracker.h"
#include "utils.h"

#include <glog/logging.h>

namespace fairrsh {

RTTR_REGISTRATION {
  rttr::registration::enumeration<TaskStatus>("TaskStatus")(
      rttr::value("Unknown", TaskStatus::Unknown),
      rttr::value("Ongoing", TaskStatus::Ongoing),
      rttr::value("Success", TaskStatus::Success),
      rttr::value("Cancelled", TaskStatus::Cancelled),
      rttr::value("Failure", TaskStatus::Failure));

  rttr::registration::class_<Task>("Task")(
      metadata("type", rttr::type::get<Task>()))
      .constructor<UpcId, std::unordered_set<Unit*>, tc::Resources>()
      .method("upcId", &Task::upcId)
      .method("status", &Task::status)
      .method(
          "units",
          rttr::select_overload<std::unordered_set<Unit*> const&() const>(
              &Task::units))
      .method("resources", &Task::resources);

  rttr::registration::class_<ProxyTask>("ProxyTask")(
      metadata("type", rttr::type::get<ProxyTask>()))
      .constructor<UpcId, UpcId>();

  rttr::registration::class_<TrackerTask>("TrackerTask")(
      metadata("type", rttr::type::get<TrackerTask>()))
      .constructor<
          std::shared_ptr<Tracker>,
          UpcId,
          std::unordered_set<Unit*>,
          tc::Resources>();
}

void Task::cancel(State* state) {
  VLOG(1) << "task " << upcId_ << " cancelled";
  setStatus(TaskStatus::Cancelled);
}

void Task::releaseMineralsAndGas(State* state) {
  tc::Resources res = {0};
  res.ore = resources_.ore;
  res.gas = resources_.gas;
  state->removePlannedResources(std::move(res));
  resources_.ore = 0;
  resources_.gas = 0;
}

void Task::releaseUsedPsi(State* state) {
  tc::Resources res = {0};
  res.used_psi = resources_.used_psi;
  state->removePlannedResources(std::move(res));
  resources_.used_psi = 0;
}

void Task::removeDeadOrReassignedUnits(State* state) {
  // We can't split this up really since the blackboard will automatically
  // unassign units that have died.
  auto board = state->board();
  for (auto it = units_.begin(); it != units_.end();) {
    if (board->taskWithUnit(*it).get() != this) {
      units_.erase(it++);
    } else {
      ++it;
    }
  }
}

ProxyTask::ProxyTask(UpcId targetUpcId, UpcId upcId)
    : Task(upcId), targetUpcId_(targetUpcId) {}

void ProxyTask::update(State* state) {
  if (target_ == nullptr) {
    target_ = state->board()->taskForId(targetUpcId_);
    if (target_ == nullptr) {
      return;
    } else {
      VLOG(1) << "Proxy: Found target task for UPC " << targetUpcId_;
    }
  }

  auto oldStatus = status();
  setStatus(target_->status());
  if (status() != oldStatus) {
    VLOG(2) << "Task for UPC " << upcId()
            << ": status changed: " << utils::enumAsInt(oldStatus) << " -> "
            << utils::enumAsInt(status());
  }
}

void ProxyTask::cancel(State* state) {
  if (target_) {
    VLOG(2) << "ProxyTask cancelled -> cancelling proxied task for UPC "
	    << target_->upcId();
    return target_->cancel(state);
  }
  VLOG(2) << "ProxyTask cancelled without proxy -> removing UPC " << targetUpcId_;
  state->board()->removeUPCs({targetUpcId_});
  Task::cancel(state);
}

std::unordered_set<Unit*> const& ProxyTask::proxiedUnits() const {
  if (target_ != nullptr) {
    return target_->proxiedUnits();
  }
  if (!units().empty()) {
    std::runtime_error("ProxyTask should not contain any units");
  }
  return units();
}

MultiProxyTask::MultiProxyTask(std::vector<UpcId> targetUpcIds, UpcId upcId)
    : Task(upcId), targetUpcIds_(std::move(targetUpcIds)) {
  targets_.insert(targets_.begin(), targetUpcIds_.size(), nullptr);
  policy_[TaskStatus::Unknown] = ProxyPolicy::ALL;
  policy_[TaskStatus::Ongoing] = ProxyPolicy::ANY;
  policy_[TaskStatus::Failure] = ProxyPolicy::ANY;
  policy_[TaskStatus::Cancelled] = ProxyPolicy::ALL;
  policy_[TaskStatus::Success] = ProxyPolicy::ALL;
}

void MultiProxyTask::setPolicyForStatus(TaskStatus status, ProxyPolicy policy) {
  policy_[status] = policy;
}

void MultiProxyTask::update(State* state) {
  auto board = state->board();
  for (size_t i = 0; i < targetUpcIds_.size(); i++) {
    auto& target = targets_[i];
    if (target == nullptr) {
      target = board->taskForId(targetUpcIds_[i]);
      if (target != nullptr) {
        VLOG(1) << "Multiproxy: found target task for UPC " << targetUpcIds_[i];
      }
    }
  }

  // Update status according to policy
  auto oldStatus = status();
  setStatus(TaskStatus::Unknown); // Fallback
  for (auto status : {TaskStatus::Success,
	              TaskStatus::Cancelled,
                      TaskStatus::Failure,
                      TaskStatus::Ongoing,
                      TaskStatus::Unknown}) {
    if (matchStatus(status)) {
      if (status != oldStatus) {
        VLOG(2) << "MultiProxy: change status to " << UpcId(status);
      }
      setStatus(status);
      break;
    }
  }

  // Update list of proxied units
  proxiedUnits_.clear();
  for (auto& target : targets_) {
    if (target != nullptr) {
      for (auto unit : target->proxiedUnits()) {
        proxiedUnits_.insert(unit);
      }
    }
  }
}

void MultiProxyTask::cancel(State* state) {
  auto board = state->board();
  VLOG(2) << "MultiProxy: canceling task with "
	  << targetUpcIds_.size() << " UPCs";
  for (size_t i = 0; i < targetUpcIds_.size(); i++) {
    auto& target = targets_[i];
    if (target) {
      target->cancel(state);
      VLOG(2) << "MultiProxyTask canceled -> canceling proxy task for upc "
	      << target->upcId();
    }
    else {
      board->removeUPCs({targetUpcIds_[i]});
      VLOG(2) << "MultiProxyTask canceled -> removing UPC without proxy task "
	      <<  targetUpcIds_[i];
    }
  }
  // mark unproxied tasks as canceled by default
  defaultTargetStatus_ = TaskStatus::Cancelled;
}

std::unordered_set<Unit*> const& MultiProxyTask::proxiedUnits() const {
  return proxiedUnits_;
}

bool MultiProxyTask::matchStatus(TaskStatus status) {
  auto policy = policy_[status];
  auto getStatus = [&](auto& task) {
    if (task == nullptr) {
      return defaultTargetStatus_;
    }
    return task->status();
  };

  if (policy == ProxyPolicy::ANY) {
    for (auto& target : targets_) {
      if (getStatus(target) == status) {
        return true;
      }
    }
    return false;
  } else if (policy == ProxyPolicy::MOST) {
    size_t n = 0;
    for (auto& target : targets_) {
      if (getStatus(target) == status) {
        n++;
      }
    }
    return n > targets_.size() / 2;
  } else if (policy == ProxyPolicy::ALL) {
    for (auto& target : targets_) {
      if (getStatus(target) != status) {
        return false;
      }
    }
    return true;
  }

  return false;
}

TrackerTask::TrackerTask(
    std::shared_ptr<Tracker> tracker,
    UpcId upcId,
    std::unordered_set<Unit*> units,
    tc::Resources resources)
    : Task(upcId, std::move(units), std::move(resources)),
      tracker_(std::move(tracker)),
      trackerStatus_(tracker_->status()) {}

void TrackerTask::update(State* state) {
  auto oldStatus = status();
  switch (tracker_->status()) {
    case TrackerStatus::Pending:
      setStatus(TaskStatus::Ongoing);
      break;
    case TrackerStatus::Ongoing:
      if (trackerStatus_ == TrackerStatus::Pending) {
        // Minerals and gas will be accounted for in the game as soon as an
        // order is being executed. The tracker switched from pending to
        // ongoing, so we assume the resources don't need to be reserved by the
        // task any more.
        releaseMineralsAndGas(state);
      }
      setStatus(TaskStatus::Ongoing);
      break;
    case TrackerStatus::Success:
      setStatus(TaskStatus::Success);
      break;
    case TrackerStatus::Timeout:
    case TrackerStatus::Failure:
      setStatus(TaskStatus::Failure);
      break;
    default:
      setStatus(TaskStatus::Unknown);
      break;
  }

  trackerStatus_ = tracker_->status();
  if (status() != oldStatus) {
    VLOG(2) << "Task for UPC " << upcId()
            << ": satus changed: " << utils::enumAsInt(oldStatus) << " -> "
            << utils::enumAsInt(status());

    if (finished()) {
      // Supply will be accounted for in the game when an order is finished. To
      // be safe, release everything now.
      state->removePlannedResources(resources());
    }
  }
}

} // namespace fairrsh
