/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#include "selfplayscenario.h"

#include "fairrsh.h"
#include "openbwprocess.h"

#ifndef WITHOUT_POSIX
#include <unistd.h>
#endif // !WITHOUT_POSIX

namespace fairrsh {

RTTR_REGISTRATION {
  rttr::registration::class_<SelfPlayScenario>("SelfPlayScenario")(
      metadata("type", rttr::type::get<SelfPlayScenario>()))
      .constructor<std::string, std::string, std::string>()
      .method("makeClient1", &SelfPlayScenario::makeClient1)(
          rttr::default_arguments(tc::Client::Options()))
      .method("makeClient2", &SelfPlayScenario::makeClient2)(
          rttr::default_arguments(tc::Client::Options()));
}

SelfPlayScenario::Pipe::Pipe() {
#ifdef WITHOUT_POSIX
  throw std::runtime_error("SelfPlayScenario::Pipe: Not implemented");
#else // WITHOUT_POSIX
  int p[2];
  if (pipe(p) != 0) {
    throw std::system_error(errno, std::system_category());
  }
  rfd = p[0];
  wfd = p[1];
#endif // WITHOUT_POSIX
}

 SelfPlayScenario::Pipe::~Pipe() {
//   close(wfd);
//   close(rfd);
 }

SelfPlayScenario::SelfPlayScenario(
    std::string const& map,
    std::string const& race1,
    std::string const& race2) {
  proc1_ = std::make_shared<OpenBwProcess>(
      AiModule::BWEnv,
      std::vector<OpenBwProcess::EnvVar>{
          {"OPENBW_LAN_MODE", "FD", true},
          {"OPENBW_FD_READ", std::to_string(pipe1_.rfd), true},
          {"OPENBW_FD_WRITE", std::to_string(pipe2_.wfd), true},
          {"BWAPI_CONFIG_AUTO_MENU__AUTO_MENU", "LAN", true},
          {"BWAPI_CONFIG_AUTO_MENU__GAME_TYPE", "MELEE", true},
          {"BWAPI_CONFIG_AUTO_MENU__MAP", map.c_str(), true},
          {"BWAPI_CONFIG_AUTO_MENU__RACE", race1.c_str(), true},
          {"BWAPI_CONFIG_AUTO_MENU__CHARACTER_NAME", "BWEnv1_" + race1, true},
      });
  proc2_ = std::make_shared<OpenBwProcess>(
      AiModule::BWEnv,
      std::vector<OpenBwProcess::EnvVar>{
          {"OPENBW_ENABLE_UI", "0", true},
          {"OPENBW_LAN_MODE", "FD", true},
          {"OPENBW_FD_READ", std::to_string(pipe2_.rfd), true},
          {"OPENBW_FD_WRITE", std::to_string(pipe1_.wfd), true},
          {"BWAPI_CONFIG_AUTO_MENU__AUTO_MENU", "LAN", true},
          {"BWAPI_CONFIG_AUTO_MENU__GAME_TYPE", "MELEE", true},
          {"BWAPI_CONFIG_AUTO_MENU__MAP", map.c_str(), true},
          {"BWAPI_CONFIG_AUTO_MENU__RACE", race2.c_str(), true},
          {"BWAPI_CONFIG_AUTO_MENU__CHARACTER_NAME", "BWEnv2_" + race2, true},
      });
}

// SelfPlayScenario::~SelfPlayScenario() {
//   proc1_.reset();
//   proc2_.reset();
// }

std::shared_ptr<tc::Client> SelfPlayScenario::makeClient1(
    tc::Client::Options opts) const {
  return makeClient(proc1_, std::move(opts));
}

std::shared_ptr<tc::Client> SelfPlayScenario::makeClient2(
    tc::Client::Options opts) const {
  return makeClient(proc2_, std::move(opts));
}

std::shared_ptr<torchcraft::Client> SelfPlayScenario::makeClient(
    std::shared_ptr<OpenBwProcess> proc,
    torchcraft::Client::Options opts) const {
  auto client = std::make_shared<tc::Client>();
  if (!client->connect("localhost", proc->port(), 1000)) {
    throw std::runtime_error(
        std::string("Error establishing connection: ") + client->error());
  }

  // Perform handshake
  std::vector<std::string> upd;
  if (!client->init(upd, opts)) {
    throw std::runtime_error(
        std::string("Error initializing connection: ") + client->error());
  }

  return client;
}

} // namespace fairrsh
