/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#include "scenario.h"

#include "openbwprocess.h"

namespace fairrsh {

RTTR_REGISTRATION {
  rttr::registration::class_<Scenario>("Scenario")(
      metadata("type", rttr::type::get<Scenario>()))
      .constructor<std::string, std::string>()
      .method("makeClient", &Scenario::makeClient)(
          rttr::default_arguments(tc::Client::Options()));

  rttr::registration::class_<MeleeScenario>("MeleeScenario")(
      metadata("type", rttr::type::get<MeleeScenario>()))
      .constructor<std::string, std::string, std::string>();
}

Scenario::Scenario(std::string map, std::string race)
    : map_(std::move(map)), race_(std::move(race)) {
  proc_ = startProcess();
}

std::shared_ptr<tc::Client> Scenario::makeClient(
    tc::Client::Options opts) const {
  auto client = std::make_shared<tc::Client>();
  if (!client->connect("localhost", proc_->port(), 1000)) {
    throw std::runtime_error(
        std::string("Error establishing connection: ") + client->error());
  }

  // Perform handshake
  std::vector<std::string> upd;
  if (!client->init(upd, opts)) {
    throw std::runtime_error(
        std::string("Error initializing connection: ") + client->error());
  }

  return client;
}

std::shared_ptr<OpenBwProcess> Scenario::startProcess() const {
  return std::make_shared<OpenBwProcess>(
      AiModule::BWEnv,
      std::vector<OpenBwProcess::EnvVar>{
          {"BWAPI_CONFIG_AUTO_MENU__GAME_TYPE", "USE_MAP_SETTINGS", true},
          {"BWAPI_CONFIG_AUTO_MENU__MAP", map_.c_str(), true},
          {"BWAPI_CONFIG_AUTO_MENU__RACE", race_.c_str(), true},
      });
}

MeleeScenario::MeleeScenario(
    std::string map,
    std::string myRace,
    std::string enemyRace)
    : Scenario(),
      map_(std::move(map)),
      myRace_(std::move(myRace)),
      enemyRace_(std::move(enemyRace)) {
  proc_ = startProcess();
}

std::shared_ptr<OpenBwProcess> MeleeScenario::startProcess() const {
  return std::make_shared<OpenBwProcess>(
      AiModule::BWEnv,
      std::vector<OpenBwProcess::EnvVar>{
          {"BWAPI_CONFIG_AUTO_MENU__GAME_TYPE", "MELEE", true},
          {"BWAPI_CONFIG_AUTO_MENU__MAP", map_.c_str(), true},
          {"BWAPI_CONFIG_AUTO_MENU__RACE", myRace_.c_str(), true},
          {"BWAPI_CONFIG_AUTO_MENU__ENEMY_RACE", enemyRace_.c_str(), true},
      });
}

} // namespace fairrsh
