/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#include "openbwprocess.h"

#include <array>
#include <cstdlib>
#include <fcntl.h>
#include <signal.h>
#ifndef WITHOUT_POSIX
#include <unistd.h>
#include <sys/wait.h>
#endif // WITHOUT_POSIX

#include <glog/logging.h>

namespace fairrsh {

namespace {

// Spawn a process similar to popen(3) but return its process ID.
// Thanks to https://stackoverflow.com/a/26852210
int popen2(const char* command, int* infp, int* outfp) {
#ifdef WITHOUT_POSIX
  throw std::runtime_error("pope2(): not implemented");
  return -1;
#else // WITHOUT_POSIX
  int p_stdin[2], p_stdout[2];
  pid_t pid;

  if (pipe(p_stdin) != 0 || pipe(p_stdout) != 0) {
    return -1;
  }

  pid = fork();

  if (pid < 0) {
    return pid;
  } else if (pid == 0) {
    close(p_stdin[0]);
    dup2(p_stdin[0], 0);
    close(p_stdout[0]);
    dup2(p_stdout[1], 1);

    execl("/bin/sh", "sh", "-c", command, NULL);
    perror("execl");
    exit(1);
  }

  if (infp == NULL) {
    close(p_stdin[1]);
  } else {
    *infp = p_stdin[1];
  }

  if (outfp == NULL) {
    close(p_stdout[0]);
  } else {
    *outfp = p_stdout[0];
  }

  return pid;
#endif // WITHOUT_POSIX
}

std::string modulePath(AiModule module) {
  std::string suffix = ".so";
#ifdef __APPLE__
  suffix = ".dylib"; // Don't bother fixing Makefiles...
#endif
  switch (module) {
    case AiModule::BWEnv:
      return "torchcraft/BWEnv/build/BWEnv" + suffix;
    case AiModule::Steamhammer_123:
      return "bw_bots/build/lib/Steamhammer-1.2.3" + suffix;
    case AiModule::UAlbertaBot_aiide_2015:
      return "bw_bots/build/lib/UAlbertaBot-aiide-2015" + suffix;
    case AiModule::BlackCrow:
      return "bw_bots/build/lib/BlackCrow" + suffix;
    default:
      throw std::runtime_error("Unknown AiModule");
  }
}

std::string moduleName(AiModule module) {
  switch (module) {
    case AiModule::BWEnv:
      return "BWEnv";
    case AiModule::Steamhammer_123:
      return "Steamhammer-1.2.3";
    case AiModule::UAlbertaBot_aiide_2015:
      return "UAlbertaBot-aiide-2015";
    case AiModule::BlackCrow:
      return "BlackCrow";
    default:
      throw std::runtime_error("Unknown AiModule");
  }
}

} // namespace

OpenBwProcess::OpenBwProcess(AiModule module, std::vector<EnvVar> const& vars) {
#ifdef WITHOUT_POSIX
  throw std::runtime_error("OpenBwProcess: Not implemented");
#else // WITHOUT_POSIX
  // Set a couple of default variables
  std::vector<EnvVar> defaultVars = {
      {"OPENBW_ENABLE_UI", "0", false},
      {"TORCHCRAFT_PORT", "0", true}, // automatically select port
      {"BWAPI_CONFIG_AI__AI", modulePath(module).c_str(), true},
      {"BWAPI_CONFIG_AUTO_MENU__CHARACTER_NAME", moduleName(module).c_str(), true},
      {"BWAPI_CONFIG_AUTO_MENU__AUTO_MENU", "SINGLE_PLAYER", true},
      {"BWAPI_CONFIG_AUTO_MENU__GAME_TYPE", "USE_MAP_SETTINGS", true},
      {"BWAPI_CONFIG_AUTO_MENU__AUTO_RESTART", "OFF", true},
  };

  std::string debugstr;
  for (auto& var : defaultVars) {
    setenv(var.key.c_str(), var.value.c_str(), var.overwrite ? 1 : 0);
    debugstr = debugstr + var.key + "=\"" + var.value + "\" ";
  }
  for (auto& var : vars) {
    setenv(var.key.c_str(), var.value.c_str(), var.overwrite ? 1 : 0);
    debugstr = debugstr + var.key + "=\"" + var.value + "\" ";
  }
  debugstr = debugstr + "BWAPILauncher";
  VLOG(4) << debugstr;

  // Launch OpenBW via BWAPILauncher
  int fd, flags;
  pid_ = popen2("BWAPILauncher", nullptr, &fd);
  if (pid_ < 0) {
    throw std::runtime_error("Failed to spawn process");
  }
  flags = fcntl(fd, F_GETFL, 0);
  flags |= O_NONBLOCK;
  fcntl(fd, F_SETFL, flags);
  pipe_ = fdopen(fd, "r");

  if (module == AiModule::BWEnv) {
    running_.store(true);
    portf_ = portp_.get_future();
    outputThread_ = std::async(std::launch::async,
                    &OpenBwProcess::redirectOutput, this);
  }
#endif // WITHOUT_POSIX
}

OpenBwProcess::~OpenBwProcess() {
#ifndef WITHOUT_POSIX
  fclose(pipe_);
  running_.store(false);

  // Wait a short time for BWAPILauncher to exit
  for (int i = 0; i < 5; i++) {
    if (waitpid(pid_, nullptr, WNOHANG) > 0) {
      return;
    }
    std::this_thread::sleep_for(std::chrono::milliseconds(10));
  }
  kill(pid_, SIGKILL);
#endif // !WITHOUT_POSIX
}

int OpenBwProcess::port() {
  if (port_ < 0 && portf_.valid()) {
    port_ = portf_.get();
  }
  return port_;
}

void OpenBwProcess::redirectOutput() {
  std::array<char, 256> buffer;
  int port;
  while (!feof(pipe_) && running_.load()) {
    if (fgets(buffer.data(), buffer.size(), pipe_) != NULL) {
      VLOG(2) << buffer.data();
      if (port_ < 0 && std::sscanf(
              buffer.data(), "TorchCraft server listening on port %d", &port) >
          0) {
        portp_.set_value(port);
      }
    }
  }
}
} // namespace fairrsh
