/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#include "state.h"
#include "task.h"
#include "utils.h"

#include "modules/upctocommand.h"

#include <bwem/map.h>
#include <glog/logging.h>

namespace fairrsh {

RTTR_REGISTRATION {
  rttr::registration::class_<UPCToCommandModule>("UPCToCommandModule")(
      metadata("type", rttr::type::get<UPCToCommandModule>()))
      .constructor();
}

namespace {

Unit* getPositionU(UPCTuple const& upc) {
  Unit* unit = nullptr;
  float maxP = std::numeric_limits<float>::lowest();
  for (auto& el : upc.positionU) {
    if (el.second > maxP) {
      unit = el.first;
      maxP = el.second;
    }
  }
  return unit;
}

Position getPosition(UPCTuple const& upc) {
#ifdef WITH_ATEN
  if (upc.position.defined()) {
    auto amax = utils::argmax(upc.position, upc.scale);
    return Position(std::get<0>(amax), std::get<1>(amax));
  }
#endif // WITH_ATEN
  if (!upc.positionU.empty()) {
    auto unit = getPositionU(upc);
    return Position(unit->x, unit->y);
  }
  if (upc.positionS.x >= 0 && upc.positionS.y >= 0) {
    if (upc.scale == 1) {
      return upc.positionS;
    } else {
      return Position(upc.positionS.x * upc.scale, upc.positionS.y * upc.scale);
    }
  }
  if (upc.positionA) {
    return Position(upc.positionA->x, upc.positionA->y);
  }

  return Position();
}

} // namespace

void UPCToCommandModule::checkDuplicateCommand(
    const Unit* unit,
    UPCToCommandState& upcToCommandState) {

  if (upcToCommandState.commandToUnit.find(unit) ==
      upcToCommandState.commandToUnit.end()) {
    upcToCommandState.commandToUnit.insert(unit);
  } else {
    LOG(WARNING) << "More than one command to unit " << unit->id;
  }
}

void UPCToCommandModule::registerGameCommand(
    State* state,
    const Unit* unit,
    int upcId,
    tc::Client::Command command,
    UPCToCommandState& upcToCommandState) {

  checkDuplicateCommand(unit, upcToCommandState);
  upcToCommandState.commands.emplace_back(command);
  upcToCommandState.upcIds.push_back(upcId);
  VLOG(1) << "Command from " << utils::upcString(upcId) << ": "
          << commandString(state, upcToCommandState.commands.back());
}

// TODO: better interface for drawing stuff?
void UPCToCommandModule::temporaryDebugDrawing(
    State* state,
    UPCToCommandState& upcToCommandState) {
  if (!VLOG_IS_ON(3))
    return;
  for (auto& area : state->map()->Areas()) {
    for (auto& base : area.Bases()) {
      utils::drawCircle(
          state, base.Location() * tc::BW::XYWalktilesPerBuildtile, 16, 254);
    }
  }

  for (Unit* u : state->unitsInfo().liveUnits()) {
    if (u->gone) {
      utils::drawCircle(state, u, 12);
    } else {
      utils::drawCircle(state, u, 8, tc::BW::Color::Yellow);
    }
  }

  auto forAllTiles = [&](TilesInfo& tt, auto&& f) {
    size_t stride = TilesInfo::tilesWidth - tt.mapTileWidth();
    Tile* ptr = tt.tiles.data();
    for (unsigned tileY = 0; tileY != tt.mapTileHeight();
         ++tileY, ptr += stride) {
      for (unsigned tileX = 0; tileX != tt.mapTileWidth(); ++tileX, ++ptr) {
        f(*ptr);
      }
    }
  };

  forAllTiles(state->tilesInfo(), [&](Tile& t) {
    if (t.reservedAsUnbuildable) {
      utils::drawCircle(state, Vec2(t) * 2, 16, tc::BW::Color::Red);
    }
  });
}

void UPCToCommandModule::postGameCommand(
    State* state,
    UPCToCommandState& upcToCommandState) {
  auto board = state->board();

  board->consumeUPCs(upcToCommandState.upcIds, this);
  board->postCommands(upcToCommandState.commands);
}

void UPCToCommandModule::step(State* state) {
  auto board = state->board();
  UPCToCommandState upcToCommandState;

  for (auto const& upct : board->upcs()) {
    auto upcId = upct.first;
    auto& upc = upct.second;

    // TODO: 1-epsilon?
    if (upc->unit.size() == 1 && upc->command[Command::Gather] == 1) {
      Unit* worker = upc->unit.begin()->first;
      auto dest = getPositionU(*upc);
      if (dest) {
        registerGameCommand(
            state,
            worker,
            upcId,
            tc::Client::Command(
                tc::BW::Command::CommandUnit,
                worker->id,
                tc::BW::UnitCommandType::Right_Click_Unit,
                dest->id),
            upcToCommandState);
      } else {
        auto pos = getPosition(*upc);
        for (Unit* target : state->unitsInfo().visibleUnits()) {
          if (target->x == pos.x && target->y == pos.y) {
            registerGameCommand(
                state,
                worker,
                upcId,
                tc::Client::Command(
                    tc::BW::Command::CommandUnit,
                    worker->id,
                    tc::BW::UnitCommandType::Right_Click_Unit,
                    target->id),
                upcToCommandState);
            break;
          }
        }
      }
    } else if (
        upc->unit.size() == 1 && upc->command[Command::Create] == 1 &&
        upc->createType.size() == 1) {
      const BuildType* type = upc->createType.begin()->first;
      Unit* unit = upc->unit.begin()->first;
      if (unit == nullptr) {
        LOG(WARNING) << "null unit";
        continue;
      }
      if (upc->unit[unit] < 1) {
        VLOG(4) << "Unit probability " << upc->unit[unit] << " < 1, skipping";
        continue;
      }

      if (unit->type->isWorker && type->isBuilding) {
        auto pos = getPosition(*upc);
        registerGameCommand(
            state,
            unit,
            upcId,
            tc::Client::Command(
                tc::BW::Command::CommandUnit,
                unit->id,
                tc::BW::UnitCommandType::Build,
                -1,
                pos.x,
                pos.y,
                type->unit),
            upcToCommandState);
      } else if (type->isAddon) {
        registerGameCommand(
            state,
            unit,
            upcId,
            tc::Client::Command(
                tc::BW::Command::CommandUnit,
                unit->id,
                tc::BW::UnitCommandType::Build_Addon,
                -1,
                0,
                0,
                type->unit),
            upcToCommandState);
      } else if (type->isUnit()) {
        if (type->isBuilding) {
          registerGameCommand(
              state,
              unit,
              upcId,
              tc::Client::Command(
                  tc::BW::Command::CommandUnit,
                  unit->id,
                  tc::BW::UnitCommandType::Morph,
                  -1,
                  0,
                  0,
                  type->unit),
              upcToCommandState);
        } else {
          if (type == buildtypes::Protoss_Archon ||
              type == buildtypes::Protoss_Dark_Archon) {
            LOG(WARNING) << " FIXME: morph archon!";
          }
          registerGameCommand(
              state,
              unit,
              upcId,
              tc::Client::Command(
                  tc::BW::Command::CommandUnit,
                  unit->id,
                  tc::BW::UnitCommandType::Train,
                  -1,
                  0,
                  0,
                  type->unit),
              upcToCommandState);
          // We need to keep track of the supply only since resources are
          // immediately account for in the game when issuing the training
          // order.
        }
      } else if (type->isUpgrade()) {
        registerGameCommand(
            state,
            unit,
            upcId,
            tc::Client::Command(
                tc::BW::Command::CommandUnit,
                unit->id,
                tc::BW::UnitCommandType::Upgrade,
                -1,
                0,
                0,
                type->upgrade),
            upcToCommandState);
      } else if (type->isTech()) {
        registerGameCommand(
            state,
            unit,
            upcId,
            tc::Client::Command(
                tc::BW::Command::CommandUnit,
                unit->id,
                tc::BW::UnitCommandType::Research,
                -1,
                0,
                0,
                type->tech),
            upcToCommandState);
      } else {
        LOG(WARNING) << "Cannot handle create command with "
                     << utils::unitString(unit);
      }
    } else if (upc->command[Command::Move] == 1) {
      for (auto& uprob : upc->unit) {
        if (uprob.second == 0) {
          continue;
        }
        auto pos = getPosition(*upc);
        registerGameCommand(
            state,
            uprob.first,
            upcId,
            tc::Client::Command(
                tc::BW::Command::CommandUnit,
                uprob.first->id,
                tc::BW::UnitCommandType::Move,
                -1,
                pos.x,
                pos.y),
            upcToCommandState);
      }
    } else if (upc->command[Command::Delete] == 1) {
      for (auto& uprob : upc->unit) {
        if (uprob.second == 0) {
          continue;
        }
        if (!upc->positionU.empty()) {
          if (upc->positionU.begin()->second == 1) {
            registerGameCommand(
                state,
                uprob.first,
                upcId,
                tc::Client::Command(
                    tc::BW::Command::CommandUnit,
                    uprob.first->id,
                    tc::BW::UnitCommandType::Attack_Unit,
                    upc->positionU.begin()->first->id),
                upcToCommandState);
          }
        } else {
          auto pos = getPosition(*upc);
          registerGameCommand(
              state,
              uprob.first,
              upcId,
              tc::Client::Command(
                  tc::BW::Command::CommandUnit,
                  uprob.first->id,
                  tc::BW::UnitCommandType::Attack_Move,
                  -1,
                  pos.x,
                  pos.y),
              upcToCommandState);
        }
      }
    } else if (upc->command[Command::Cancel] == 1) {
      for (auto& uprob : upc->unit) {
        if (uprob.second == 0) {
          continue;
        }
        registerGameCommand(
            state,
            uprob.first,
            upcId,
            tc::Client::Command(
                tc::BW::Command::CommandUnit,
                uprob.first->id,
                tc::BW::UnitCommandType::Cancel_Morph),
            upcToCommandState);
      }
    }
  }

  postGameCommand(state, upcToCommandState);

  temporaryDebugDrawing(state, upcToCommandState);
}

std::string UPCToCommandModule::commandString(
    State* state,
    tc::Client::Command const& cmd) {
  std::ostringstream oss;
  auto cc = tc::BW::Command::_from_integral_nothrow(cmd.code);
  oss << "{code=" << (cc ? cc->_to_string() : "???");
  if (!cmd.str.empty()) {
    oss << ", str='" << cmd.str << "'";
  }
  if (!cmd.args.empty()) {
    oss << ", args=[";
    for (size_t i = 0; i < cmd.args.size(); i++) {
      if (i > 0) {
        oss << ", ";
      }
      if (i == 0 && cmd.code == tc::BW::Command::CommandUnit) {
        // Unit type name
        oss << utils::unitString(state->unitsInfo().getUnit(cmd.args[i]));
      } else if (i == 1 && cmd.code == tc::BW::Command::CommandUnit) {
        // Command name
        auto uct = tc::BW::UnitCommandType::_from_integral_nothrow(cmd.args[i]);
        oss << "'" << (uct ? uct->_to_string() : "???") << "'";
      } else {
        oss << cmd.args[i];
      }
    }
    oss << "]";
  }
  oss << "}";

  return oss.str();
}

} // namespace fairrsh
