/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#include "ummhelper.h"

#include "models/units_mixer.h"
#include "player.h"
#include "state.h"

namespace fairrsh {

RTTR_REGISTRATION {
  rttr::registration::class_<UMMHelperModule>("UMMHelperModule")(
      metadata("type", rttr::type::get<UMMHelperModule>()))
      .constructor();
}

void UMMHelperModule::setPlayer(Player* player) {
  player->state()->board()->unitsMixer->onGameStart(player->state());
}

void UMMHelperModule::step(State* state) {
  state->board()->unitsMixer->onGameStep(state);

  int twoUnitsInOneEggCount = 0;

  auto newUnit = [&](Unit* u) {
    if (u->type->isNonUsable) {
      return;
    }
    if (u->type->isTwoUnitsInOneEgg) {
      ++twoUnitsInOneEggCount;
      if (twoUnitsInOneEggCount % 2 == 0) {
        return;
      }
    }
    VLOG(1) << " UMM new unit " << u->type->name;

    auto& input = state->board()->unitsMixerCurrentInput;
    auto& output = state->board()->unitsMixerCurrentOutput;
    if (!input.empty() && !output.empty()) {
      state->board()->unitsMixer->model_->inputs_.push_back(input);
      state->board()->unitsMixer->model_->outputs_.push_back(output);

      VLOG(1) << " there are now "
              << state->board()->unitsMixer->model_->inputs_.size()
              << " inputs and "
              << state->board()->unitsMixer->model_->outputs_.size()
              << " outputs";
    }

  };

  for (Unit* u : state->unitsInfo().getNewUnits()) {
    if (u->isMine) {
      newUnit(u);
    }
  }
  for (Unit* u : state->unitsInfo().getStartedMorphingUnits()) {
    if (u->isMine) {
      newUnit(u);
    }
  }
}

void UMMHelperModule::onGameEnd(State* state) {
  state->board()->unitsMixer->onGameEnd(state);
  model::UnitsMixer::save(state->board()->unitsMixer);
}

} // namespace fairrsh
