/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#include "tactics.h"

#include "combatsim.h"
#include "movefilters.h"
#include "player.h"
#include "state.h"
#include "task.h"
#include "utils.h"

#include <bwem/map.h>
#include <deque>
#include <glog/logging.h>
#include <memory>
#include <vector>

namespace fairrsh {

RTTR_REGISTRATION {
  rttr::registration::class_<TacticsModule>("TacticsModule")(
      metadata("type", rttr::type::get<TacticsModule>()))
      .constructor();
}

namespace {

class TacticsTask : public Task {
 public:
  TacticsTask(int upcId) : Task(upcId) {}

  virtual void update(State* state) override {
    removeDeadOrReassignedUnits(state);
  }

  std::vector<Unit*> myUnits;
  Position targetPos;
  bool isFighting = false;
};

template <typename Units>
double scoreTeam(Units&& units) {
  double score = 0.0;
  for (Unit* u : units) {
    score += u->type->gScore;
    if (u->type->isBuilding &&
        (u->type->hasGroundWeapon || u->type->hasAirWeapon)) {
      score += u->type->gScore;
    }
    if (u->type == buildtypes::Terran_Bunker) {
      score += u->type->gScore * 3;
    }
  }
  return score;
}

template <typename OurUnits, typename TheirUnits>
double winRatio(OurUnits&& ourUnits, TheirUnits&& theirUnits) {
  return scoreTeam(ourUnits) / scoreTeam(theirUnits);
}

} // namespace

void TacticsModule::process(State* state, int srcUpcId) {

  if (state->board()->hasKey("TacticsDisabled")) {
    for (auto& t : state->board()->tasksOfModule(this)) {
      auto task = std::static_pointer_cast<TacticsTask>(t);
      task->cancel(state);
      task->setUnits({});
      state->board()->updateTasksByUnit(task.get());
    }
    return;
  }

  bool attack = true;
  if (state->board()->hasKey("TacticsAttack")) {
    attack = state->board()->get<bool>("TacticsAttack");
  }

  // Two enemy units within this distance of each other will belong
  // to the same group
  const float groupingDistance = 4 * 20;
  // A group where all enemy units are greater than distance away
  // from one of our resource depots will have isAggressiveGroup set.
  const float aggressiveGroupDistance = 4 * 20;
  // The distance around each enemy unit that will be considered "inside" their
  // group. Any of our units in this area will be assigned to the group, and
  // this effectively ends up being the distance away from enemy units that
  // our units flee.
  const float insideGroupDistance = 4 * 16;
  // To avoid spending cpu searching the entire map, this is a limit to how far
  // away we will search for a position to flee to.
  const float maxFleeSearchDistance = 4 * 20;
  // Any unit within this distance of an enemy unit (and vice versa) will be
  // included in the combat simulation to determine fight or flight.
  const float nearbyUnitDistance = 4 * 20;

  struct Group {
    std::vector<Unit*> enemyUnits;
    std::vector<Unit*> myUnits;
    Unit* targetUnit = nullptr;
    Position targetPos;
    std::shared_ptr<TacticsTask> task;
    bool hasEnoughUnits = false;
    bool hasEnemyGroundUnits = false;
    bool hasEnemyAirUnits = false;
    bool hasEnemyBuildings = false;
    bool hasEnemyCloakedUnits = false;
    bool hasEnemyTanks = false;
    bool hasEnemyReavers = false;
    bool enemyIsOnlyWorkers = false;
    bool enemiesAreAttacking = false;
    bool enemiesInOurBase = false;
    bool isAggressiveGroup = true;
    double score = 0.0;
    bool searchAndDestroy = false;
    bool isIdleGroup = false;
    bool isScoutGroup = false;
  };

  std::list<Group> groups;

  int frame = state->currentFrame();

  const auto& enemyUnits = state->unitsInfo().enemyUnits();
  for (Unit* u : enemyUnits) {
    if (u->gone || (u->detected() && u->invincible())) {
      continue;
    }

    Group* thisGroup = nullptr;
    for (auto& g : groups) {
      for (Unit* u2 : g.enemyUnits) {
        if (utils::distance(u, u2) <= groupingDistance) {
          thisGroup = &g;
        }
      }
      if (thisGroup) {
        continue;
      }
    }
    if (!thisGroup) {
      groups.emplace_back();
      thisGroup = &groups.back();
      thisGroup->enemyIsOnlyWorkers = u->type->isWorker;
    }
    thisGroup->enemyUnits.push_back(u);
    if (!thisGroup->targetUnit ||
        u->type->mineralCost + u->type->gasCost >
            thisGroup->targetUnit->type->mineralCost +
                thisGroup->targetUnit->type->gasCost) {
      thisGroup->targetUnit = u;
      thisGroup->targetPos = Position(u->x, u->y);
    }
    if (u->flying()) {
      thisGroup->hasEnemyAirUnits = true;
    } else {
      thisGroup->hasEnemyGroundUnits = true;
    }
    if (!u->type->isWorker) {
      thisGroup->enemyIsOnlyWorkers = false;
    }
    if (u->type->isBuilding) {
      thisGroup->hasEnemyBuildings = true;
    }
    if (frame - u->lastAttacked <= 30) {
      thisGroup->enemiesAreAttacking = true;
    }
    if (u->cloaked() || u->burrowed()) {
      thisGroup->hasEnemyCloakedUnits = true;
    }
    if (u->type == buildtypes::Terran_Siege_Tank_Tank_Mode ||
        u->type == buildtypes::Terran_Siege_Tank_Siege_Mode) {
      thisGroup->hasEnemyTanks = true;
    }
    if (u->type == buildtypes::Protoss_Reaver) {
      thisGroup->hasEnemyReavers = true;
    }
    if (!thisGroup->enemiesInOurBase) {
      const Tile* tile = state->tilesInfo().tryGetTile(u->x, u->y);
      if (tile) {
        size_t index = tile - state->tilesInfo().tiles.data();
        thisGroup->enemiesInOurBase = inBaseArea[index] != 0;
      }
    }

    if (thisGroup->isAggressiveGroup) {
      auto ai = state->areaInfo();
      auto uArea = ai.tryGetArea(u);
      std::unordered_set<Area const*> areas;
      for (int i = 0; i < ai.numExpands() + 1; i++) {
        auto area = ai.myExpandArea(i);
        areas.insert(area);
        for (auto neighbor : area->neighbors) {
          areas.insert(neighbor);
        }
      }
      if (areas.find(uArea) != areas.end()) {
        thisGroup->isAggressiveGroup = false;
      }
      for (Unit* n : state->unitsInfo().myResourceDepots()) {
        if (utils::distance(u, n) <= aggressiveGroupDistance) {
          thisGroup->isAggressiveGroup = false;
        }
      }
    }

    if (u->type->isResourceDepot) {
      thisGroup->score -= 1000.0;
    } else if (u->type->isWorker) {
      thisGroup->score -= 100.0;
    } else {
      if (!u->type->isBuilding || u->type == buildtypes::Terran_Bunker ||
          u->type->hasAirWeapon || u->type->hasGroundWeapon) {
        thisGroup->score += u->type->mineralCost + u->type->gasCost;
      }
    }
  }

  bool anyGroupsWithBuildings = false;
  for (auto& g : groups) {
    if (g.hasEnemyBuildings) {
      anyGroupsWithBuildings = true;
      break;
    }
  }

  if (groups.empty() || !anyGroupsWithBuildings) {
    bool found = false;
    for (auto tilePos : state->map()->StartingLocations()) {
      Position pos(
          tilePos.x * tc::BW::XYWalktilesPerBuildtile,
          tilePos.y * tc::BW::XYWalktilesPerBuildtile);
      auto& tile = state->tilesInfo().getTile(pos.x, pos.y);
      if (tile.building && tile.building->isEnemy) {
        groups.emplace_back();
        groups.back().targetPos = pos;
        groups.back().hasEnemyGroundUnits = true;
        groups.back().hasEnemyBuildings = true;
        found = true;
        break;
      }
    }
    if (!found) {
      for (auto tilePos : state->map()->StartingLocations()) {
        Position pos(
            tilePos.x * tc::BW::XYWalktilesPerBuildtile,
            tilePos.y * tc::BW::XYWalktilesPerBuildtile);
        auto& tile = state->tilesInfo().getTile(pos.x, pos.y);
        if (tile.lastSeen == 0) {
          groups.emplace_back();
          groups.back().targetPos = pos;
          groups.back().hasEnemyGroundUnits = true;
          groups.back().hasEnemyBuildings = true;
        }
      }
    }
  }
  if (groups.empty()) {
    groups.emplace_back();
    groups.back().searchAndDestroy = true;
    groups.back().targetPos = Position(0, 1);
  }

  if (state->currentFrame() < 15 * 60 * 16) {
    for (Group& g : groups) {
      if (!g.hasEnemyTanks && !g.hasEnemyReavers && !g.hasEnemyAirUnits) {
        if (!g.enemiesInOurBase && !g.enemyUnits.empty()) {
          g.isAggressiveGroup = true;
        }
      }
    }
  }

  // Prioritize defending
  for (Group& g : groups) {
    if (!g.isAggressiveGroup) {
      g.score -= 100000.0;
    }
  }

  groups.sort([](Group& a, Group& b) { return a.score < b.score; });

  if (state->unitsInfo().myWorkers().size() >= 30) {
    groups.emplace_back();
    groups.back().isScoutGroup = true;
    groups.back().targetPos = Position(1, 0);
  }

  if (!attack) {
    groups.emplace_back();
    groups.back().isIdleGroup = true;
    groups.back().isAggressiveGroup = false;
    groups.back().targetPos = Position(1, 1);
  }

  struct MapNode {
    Tile* tile = nullptr;
    Group* group = nullptr;
    Unit* nearestEnemy = nullptr;
  };

  const int mapWidth = state->mapWidth();
  const int mapHeight = state->mapHeight();

  std::vector<MapNode> insideGroup(
      TilesInfo::tilesWidth * TilesInfo::tilesHeight);

  std::deque<MapNode> open;
  for (Group& g : groups) {
    for (Unit* e : g.enemyUnits) {
      if (!e->type->isWorker && e->type != buildtypes::Zerg_Overlord) {
        open.push_back({state->tilesInfo().tryGetTile(e->x, e->y), &g, e});
      }
    }
  }
  auto* tilesData = state->tilesInfo().tiles.data();
  while (!open.empty()) {
    MapNode curNode = open.front();
    open.pop_front();
    Tile* tile = curNode.tile;
    if (!tile) {
      continue;
    }
    size_t index = tile - tilesData;
    MapNode& n = insideGroup[index];
    if (n.group) {
      continue;
    }
    if (utils::distance(
            tile->x,
            tile->y,
            curNode.nearestEnemy->x,
            curNode.nearestEnemy->y) > insideGroupDistance) {
      continue;
    }
    n = curNode;

    if (tile->x > 0) {
      open.push_back({tile - 1, curNode.group, curNode.nearestEnemy});
    }
    if (tile->y > 0) {
      open.push_back(
          {tile - TilesInfo::tilesWidth, curNode.group, curNode.nearestEnemy});
    }
    if (tile->x < mapWidth - tc::BW::XYWalktilesPerBuildtile) {
      open.push_back({tile + 1, curNode.group, curNode.nearestEnemy});
    }
    if (tile->y < mapHeight - tc::BW::XYWalktilesPerBuildtile) {
      open.push_back(
          {tile + TilesInfo::tilesHeight, curNode.group, curNode.nearestEnemy});
    }
  }

  std::unordered_map<Unit*, Group*> hardAssignedUnits;
  std::unordered_map<Unit*, Group*> softAssignedUnits;

  std::unordered_set<Unit*> wasInAGroup;

  for (auto& t : state->board()->tasksOfModule(this)) {
    auto task = std::static_pointer_cast<TacticsTask>(t);

    for (Unit* u : task->myUnits) {
      wasInAGroup.insert(u);
    }

    Group* nearestGroup = utils::getBestScorePointer(
        groups,
        [&](Group& g) {
          if (g.task) {
            return std::numeric_limits<double>::infinity();
          }
          double d = task->targetPos.dist(g.targetPos);
          if (d > 4 * 4) {
            return std::numeric_limits<double>::infinity();
          }
          return d;
        },
        std::numeric_limits<double>::infinity());
    if (nearestGroup) {
      if (!nearestGroup->isIdleGroup) {
        for (Unit* u : task->myUnits) {
          if (u->dead || !u->isMine || !u->active()) {
            continue;
          }
          if (!softAssignedUnits.emplace(u, nearestGroup).second) {
            continue;
          }
          if (nearestGroup->isScoutGroup) {
            size_t index = &state->tilesInfo().getTile(u->x, u->y) - tilesData;
            MapNode& n = insideGroup[index];
            if (!n.group) {
              nearestGroup->myUnits.push_back(u);
              hardAssignedUnits[u] = n.group;
            }
          } else {
//            size_t index = &state->tilesInfo().getTile(u->x, u->y) - tilesData;
//            MapNode& n = insideGroup[index];
//            if (n.group && !u->type->isWorker && !n.group->enemyIsOnlyWorkers) {
//              if (u->type != buildtypes::Zerg_Zergling ||
//                  n.group->hasEnemyGroundUnits) {
//                n.group->myUnits.push_back(u);
//                hardAssignedUnits[u] = n.group;
//              }
//            }
          }
        }
      }
      nearestGroup->task = task;
    } else {
      task->cancel(state);
      task->setUnits({});
      state->board()->updateTasksByUnit(task.get());
    }
  }

  std::vector<Unit*> availableUnits;
  auto& myUnits = state->unitsInfo().myUnits();
  availableUnits.reserve(myUnits.size());
  for (Unit* u : myUnits) {
    if (!u->active() || u->type->isBuilding) {
      continue;
    }
    if (u->type->isNonUsable) {
      continue;
    }
    if (hardAssignedUnits.find(u) != hardAssignedUnits.end()) {
      continue;
    }

    TaskData d = state->board()->taskDataWithUnit(u);
    // Don't take units from Builder
    if (d.owner) {
      if (d.owner->name().find("Builder") != std::string::npos ||
          d.owner->name().find("MeanLingRush") != std::string::npos) {
        continue;
      }
    }

    size_t index = &state->tilesInfo().getTile(u->x, u->y) - tilesData;
    MapNode& n = insideGroup.at(index);
    if (n.group && !u->type->isWorker &&
        (u->type != buildtypes::Zerg_Zergling ||
         n.group->hasEnemyGroundUnits) &&
        !n.group->enemyIsOnlyWorkers) {
      n.group->myUnits.push_back(u);
    } else {
      if (u->type->isWorker) {
        continue;
      }

      availableUnits.push_back(u);
    }
  }

  for (Group& g : groups) {
    size_t nScouts = 1;
    if (state->unitsInfo().myWorkers().size() >= 60) {
      nScouts = 3;
    } else if (state->unitsInfo().myWorkers().size() >= 45) {
      nScouts = 2;
    }
    if (g.isScoutGroup && g.myUnits.size() < nScouts) {
      for (auto i = availableUnits.begin(); i != availableUnits.end();) {
        Unit* u = *i;
        if (u->type == buildtypes::Zerg_Zergling &&
            !insideGroup[&state->tilesInfo().getTile(u->x, u->y) - tilesData]
                 .group) {
          i = availableUnits.erase(i);
          g.myUnits.push_back(u);
          if (g.myUnits.size() >= nScouts) {
            break;
          }
        } else {
          ++i;
        }
      }
    }
  }

  auto aggressiveUnit = [&](Unit* u) { return attack; };

  for (Group& g : groups) {
    if (g.hasEnoughUnits) {
      continue;
    }
    if (g.enemyIsOnlyWorkers && !g.enemiesAreAttacking) {
      continue;
    }

    auto scoreUnit = [&](Unit* u) {
      if ((!g.hasEnemyAirUnits || !u->type->hasAirWeapon) &&
          (!g.hasEnemyGroundUnits || !u->type->hasGroundWeapon)) {
        return std::numeric_limits<double>::infinity();
      }
      if (g.isAggressiveGroup && !aggressiveUnit(u)) {
        return std::numeric_limits<double>::infinity();
      }
      double d = g.targetPos.dist(u);
      if (u->type->isWorker) {
        if (!g.enemyIsOnlyWorkers || !g.enemiesInOurBase) {
          return std::numeric_limits<double>::infinity();
        }
        d += 4 * 256;
      }
      if (softAssignedUnits[u] == &g) {
        d -= 4 * 16;
      }
      return d;
    };

    auto workersIt =
        std::remove_if(g.myUnits.begin(), g.myUnits.end(), [&](Unit* u) {
          return u->type->isWorker;
        });
    std::vector<Unit*> workers(workersIt, g.myUnits.end());
    g.myUnits.erase(workersIt, g.myUnits.end());
    if (!g.enemyIsOnlyWorkers) {
      workers.clear();
    }

    while (!g.hasEnoughUnits) {
      auto i = utils::getBestScore(
          availableUnits, scoreUnit, std::numeric_limits<double>::infinity());
      if (i == availableUnits.end()) {
        break;
      }
      Unit* u = *i;
      if (i != std::prev(availableUnits.end())) {
        std::swap(*i, availableUnits.back());
      }
      availableUnits.pop_back();
      g.myUnits.push_back(u);
      double desiredWinRatio = 2.0;
      if (g.enemyIsOnlyWorkers) {
        desiredWinRatio = 1.0;
      }
      g.hasEnoughUnits = winRatio(g.myUnits, g.enemyUnits) >= desiredWinRatio;
      if (g.enemyIsOnlyWorkers && g.enemyUnits.size() == 1) {
        break;
      }
    }

    if (!g.hasEnoughUnits) {
      g.myUnits.insert(g.myUnits.end(), workers.begin(), workers.end());
    }
  }

  auto assignDetectors = [&]() {
    std::deque<Group*> remainingGroups;
    for (auto& g : groups) {
      remainingGroups.push_back(&g);
    }
    std::sort(
        remainingGroups.begin(), remainingGroups.end(), [](Group* a, Group* b) {
          if (a->hasEnemyCloakedUnits != b->hasEnemyCloakedUnits) {
            return a->hasEnemyCloakedUnits;
          }
          return a->myUnits.size() > b->myUnits.size();
        });
    while (!remainingGroups.empty()) {
      Group* g = remainingGroups.front();
      remainingGroups.pop_front();
      if (!g->hasEnemyCloakedUnits) {
        continue;
      }
      bool hasDetector = false;
      for (Unit* u : g->myUnits) {
        if (u->type->isDetector) {
          hasDetector = true;
          break;
        }
      }
      if (!hasDetector) {
        auto i = utils::getBestScore(
            availableUnits,
            [&](Unit* u) {
              if (!u->type->isDetector) {
                return std::numeric_limits<double>::infinity();
              }
              return g->targetPos.dist(u);
            },
            std::numeric_limits<double>::infinity());
        if (i == availableUnits.end()) {
          break;
        }
        Unit* u = *i;
        if (i != std::prev(availableUnits.end())) {
          std::swap(*i, availableUnits.back());
        }
        availableUnits.pop_back();
        g->myUnits.push_back(u);
      }
    }
  };

  std::vector<Unit*> leftoverWorkers;

  int assignNOverlords = 0;
  if (state->unitsInfo().myWorkers().size() >= 45) {
    assignNOverlords = 2;
    for (auto& g : groups) {
      if (g.isAggressiveGroup) {
        for (Unit* u : g.myUnits) {
          if (u->type == buildtypes::Zerg_Overlord) {
            --assignNOverlords;
          }
        }
      }
    }
  }

  auto assignLeftovers = [&]() {
    if (groups.empty()) {
      return;
    }
    Group* airGroup = &groups.front();
    Group* groundGroup = &groups.front();
    for (auto& g : groups) {
      if (g.hasEnemyAirUnits) {
        airGroup = &g;
        break;
      }
    }
    for (auto& g : groups) {
      if (g.hasEnemyGroundUnits) {
        groundGroup = &g;
        break;
      }
    }
    Group* defAirGroup = &groups.back();
    Group* defGroundGroup = &groups.back();
    for (auto& g : groups) {
      if (!g.isAggressiveGroup && g.hasEnemyAirUnits) {
        defAirGroup = &g;
        break;
      }
    }
    for (auto& g : groups) {
      if (!g.isAggressiveGroup && g.hasEnemyGroundUnits) {
        defGroundGroup = &g;
        break;
      }
    }
    while (!availableUnits.empty()) {
      Unit* u = availableUnits.back();
      Group* g = aggressiveUnit(u)
          ? (u->type->hasAirWeapon ? airGroup : groundGroup)
          : (u->type->hasAirWeapon ? defAirGroup : defGroundGroup);
      if (u->type->isWorker) {
        leftoverWorkers.push_back(availableUnits.back());
      } else if (
          u->type != buildtypes::Zerg_Overlord ||
          (assignNOverlords && assignNOverlords--)) {
        g->myUnits.push_back(u);
      }
      availableUnits.pop_back();
    }
  };

  assignDetectors();

  assignLeftovers();

  if (VLOG_IS_ON(2)) {
    VLOG(2) << groups.size() << "groups";
    for (Group& g : groups) {
      VLOG(2) << "group at " << g.targetPos.x << " " << g.targetPos.y << ": "
              << g.myUnits.size() << " allies, " << g.enemyUnits.size()
              << " enemies"
              << " aggressive " << g.isAggressiveGroup;
      for (Unit* u : g.myUnits) {
        VLOG(2) << "  ally " << u->id << " (" << u->type->name << ")";
      }
      for (Unit* u : g.enemyUnits) {
        VLOG(2) << "  enemy " << u->id << " (" << u->type->name << ")";
      }
    }
  }

  auto move = [&](Unit* u, Position target) {
    if (noOrderUntil[u] > state->currentFrame()) {
      return;
    }
    if (u->burrowed()) {
      if (u->type != buildtypes::Zerg_Lurker ||
          lastTargetInRange[u] - state->currentFrame() > 30) {
        state->board()->postCommand(tc::Client::Command(
            tc::BW::Command::CommandUnit,
            u->id,
            tc::BW::UnitCommandType::Unburrow));
        noOrderUntil[u] = state->currentFrame() + 15;
      }
      return;
    }
    if (state->currentFrame() - lastMove[u] < 12) {
      if (!u->unit.orders.empty()) {
        auto& o = u->unit.orders.front();
        if (o.type == tc::BW::Order::Move && o.targetX == target.x &&
            o.targetY == target.y) {
          return;
        }
      }
    }
    if (VLOG_IS_ON(2)) {
      utils::drawLine(state, Position(u), target, tc::BW::Color::Green);
    }
    state->board()->postUPC(
        utils::makeSharpUPC(u, target, Command::Move), srcUpcId, this);
    lastMove[u] = state->currentFrame();
  };

  std::vector<uint8_t> visited(TilesInfo::tilesWidth * TilesInfo::tilesHeight);
  uint8_t visitedN = 0;

  std::vector<uint8_t> spotTaken(
      TilesInfo::tilesWidth * TilesInfo::tilesHeight);

  auto findRunPos = [&](Unit* u) {
    bool flying = u->flying();

    uint8_t visitedValue = ++visitedN;

    auto* tilesData = state->tilesInfo().tiles.data();

    Position startPos(u->x, u->y);

    int nFound = 0;
    int bestScore = std::numeric_limits<int>::max();
    Position bestPos;

    std::deque<const Tile*> open;
    open.push_back(&state->tilesInfo().getTile(u->x, u->y));
    while (!open.empty()) {
      const Tile* tile = open.front();
      open.pop_front();

      if (!insideGroup[tile - tilesData].group) {
        int score = fleeScore[tile - tilesData] + spotTaken[tile - tilesData];
        if (score < bestScore) {
          bestScore = score;
          bestPos = Position(tile->x, tile->y);
        }
        ++nFound;
        if (nFound >= 16) {
          break;
        }
        continue;
      }

      auto add = [&](const Tile* ntile) {
        if (!flying && !tile->entirelyWalkable) {
          return;
        }
        auto& v = visited[ntile - tilesData];
        if (v == visitedValue) {
          return;
        }
        v = visitedValue;
        if (startPos.dist(ntile) <= maxFleeSearchDistance) {
          open.push_back(ntile);
        }
      };
      if (tile->x > 0) {
        add(tile - 1);
      }
      if (tile->y > 0) {
        add(tile - TilesInfo::tilesWidth);
      }
      if (tile->x < mapWidth - tc::BW::XYWalktilesPerBuildtile) {
        add(tile + 1);
      }
      if (tile->y < mapHeight - tc::BW::XYWalktilesPerBuildtile) {
        add(tile + TilesInfo::tilesHeight);
      }
    }

    return bestPos;
  };

  auto findMoveAwayPos = [&](Unit* u, Position source, float distance) {
    bool flying = u->flying();

    uint8_t visitedValue = ++visitedN;

    auto* tilesData = state->tilesInfo().tiles.data();

    Position startPos(u->x, u->y);

    std::deque<const Tile*> open;
    open.push_back(&state->tilesInfo().getTile(u->x, u->y));
    while (!open.empty()) {
      const Tile* tile = open.front();
      open.pop_front();

      if (utils::distance(tile->x, tile->y, source.x, source.y) >= distance) {
        return Position(tile->x, tile->y);
      }

      auto add = [&](const Tile* ntile) {
        if (!flying && !tile->entirelyWalkable) {
          return;
        }
        auto& v = visited[ntile - tilesData];
        if (v == visitedValue) {
          return;
        }
        v = visitedValue;
        if (utils::distance(ntile->x, ntile->y, startPos.x, startPos.y) <=
            maxFleeSearchDistance) {
          open.push_back(ntile);
        }
      };

      if (tile->x > 0) {
        add(tile - 1);
      }
      if (tile->y > 0) {
        add(tile - TilesInfo::tilesWidth);
      }
      if (tile->x < mapWidth - tc::BW::XYWalktilesPerBuildtile) {
        add(tile + 1);
      }
      if (tile->y < mapHeight - tc::BW::XYWalktilesPerBuildtile) {
        add(tile + TilesInfo::tilesHeight);
      }
    }

    return Position();
  };

  bool posted = false;
  for (Group& g : groups) {
    if (!g.task && !posted) {
      auto task = std::make_shared<TacticsTask>(srcUpcId);
      task->targetPos = g.targetPos;
      state->board()->postTask(std::move(task), this, true);
      posted = true;
    }
    if (!g.task) {
      continue;
    }

    g.task->myUnits = g.myUnits;
    g.task->targetPos = g.targetPos;

    bool anythingInRange = false;
    for (Unit* u : g.myUnits) {
      for (Unit* e : g.enemyUnits) {
        if (u->inRangeOf(e, 4) || e->inRangeOf(u, 4)) {
          anythingInRange = true;
          break;
        }
      }
      if (anythingInRange) {
        break;
      }
    }

    // If this is true, we will issue individual commands to units, essentially
    // microing units from here instead of depending on SquadCombat.
    bool tacticsMicroesUnits = state->board()->hasKey("TacticsMicroesUnits");

    std::vector<Unit*> nearbyAllies;
    std::unordered_set<Unit*> nearbyEnemies;
    std::unordered_set<Unit*> enemiesInRangeOfOurStaticDefence;
    for (Unit* u : g.myUnits) {
      Unit* enemy = utils::getBestScoreCopy(
          g.enemyUnits,
          [&](Unit* e) {
            double d = utils::distance(u, e);
            if (d >= nearbyUnitDistance) {
              return std::numeric_limits<double>::infinity();
            }
            nearbyEnemies.insert(e);
            return d;
          },
          std::numeric_limits<double>::infinity());
      if (enemy) {
        nearbyAllies.push_back(u);
      }
    }
    for (auto u : state->unitsInfo().myBuildings()) {
      if (u->type->hasGroundWeapon || u->type->hasAirWeapon) {
        Unit* enemy = utils::getBestScoreCopy(
            g.enemyUnits,
            [&](Unit* e) {
              double d = utils::distance(u, e);
              auto range = e->flying() ? u->unit.groundRange : u->unit.airRange;
              if (d >= range) {
                return std::numeric_limits<double>::infinity();
              }
              nearbyEnemies.insert(e);
              enemiesInRangeOfOurStaticDefence.insert(e);
              return d;
            },
            std::numeric_limits<double>::infinity());
        if (enemy) {
          nearbyAllies.push_back(u);
        }
      }
    }
    bool airFight = true;
    bool groundFight = true;
    double score = 0;

    if (!nearbyAllies.empty() && !nearbyEnemies.empty()) {
      CombatSim sim;
      for (Unit* u : nearbyAllies) {
        sim.addUnit(u);
      }
      for (Unit* u : nearbyEnemies) {
        if (!u->type->isWorker && u->type != buildtypes::Zerg_Overlord) {
          sim.addUnit(u);
        }
      }
      sim.run(10 * 24);

      double myRatio = sim.teams[0].endHp / sim.teams[0].startHp;
      double enemyRatio = sim.teams[1].endHp / sim.teams[1].startHp;

      score = myRatio - enemyRatio;
      if (score != score) {
        score = sim.teams[0].endHp ? 1.0 : -1.0;
      }

      double mod = 0.0;
      if (anythingInRange) {
        mod -= 0.3;
      }
      if (!g.isAggressiveGroup) {
        mod -= 0.3;
      }
      if (g.enemiesInOurBase) {
        mod -= 0.3;
      }
      if (g.task->isFighting) {
        airFight = score >= (0.3 + mod);
        groundFight = score >= (0.0 + mod);
      } else {
        airFight = score >= (0.4 + mod);
        groundFight = score >= (0.4 + mod);
      }
    }
    g.task->isFighting = groundFight;

    bool anyAntiAir = false;
    for (Unit* u : g.enemyUnits) {
      if (u->type == buildtypes::Terran_Bunker || u->type->hasAirWeapon) {
        anyAntiAir = true;
        break;
      }
    }

    std::vector<Unit*> fightUnits;
    std::vector<Unit*> fleeUnits;

    std::vector<std::pair<float, Unit*>> mySortedUnits;

    for (Unit* u : g.myUnits) {
      float nearestDistance = std::numeric_limits<float>::infinity();
      for (Unit* e : g.enemyUnits) {
        if (e->flying() ? u->type->hasAirWeapon : u->type->hasGroundWeapon) {
          float d = utils::distance(u->x, u->y, e->x, e->y);
          if (d < nearestDistance) {
            nearestDistance = d;
          }
        }
      }
      mySortedUnits.emplace_back(nearestDistance, u);
    }
    std::sort(mySortedUnits.begin(), mySortedUnits.end());

    std::unordered_map<Unit*, int> meleeTargetCount;

    auto areaInfo = state->areaInfo();

    for (auto& v : mySortedUnits) {
      Unit* u = v.second;

      Unit* target = nullptr;

      target = utils::getBestScoreCopy(
          g.enemyUnits,
          [&](Unit* e) {
            if (e->flying() ? !u->type->hasAirWeapon
                            : !u->type->hasGroundWeapon) {
              return std::numeric_limits<double>::infinity();
            }
            double d =
                (double)utils::pxDistanceBB(u, e) * tc::BW::XYPixelsPerWalktile;
            double r = d;
            if (e->type->isWorker) {
              r -= 4 * 2;
            }
            if (e->type == buildtypes::Terran_Siege_Tank_Siege_Mode) {
              r -= 4 * 10;
            }
            if (u->type == buildtypes::Zerg_Zergling ||
                u->type == buildtypes::Zerg_Scourge) {
              int maxN = 2 + e->type->size;
              if (meleeTargetCount[e] >= maxN) {
                r += 4 * 6;
              }
              if (e->type == buildtypes::Terran_Missile_Turret) {
                r -= 4 * 10;
              }
              if (e->type == buildtypes::Terran_Vulture && d > 32) {
                r += 4 * 6;
              }
            }
            if (d > 4 * 2 && r < 4 * 2) {
              r = 4 * 2;
            }
            return r;
          },
          std::numeric_limits<double>::infinity());
      if (target) {
        if (u->type == buildtypes::Zerg_Zergling ||
            u->type == buildtypes::Zerg_Scourge) {
          ++meleeTargetCount[target];
        }
        if (target->inRangeOf(u)) {
          lastTargetInRange[u] = state->currentFrame();
        }
      }

      bool runAway = false;
      Position moveTo = g.targetPos;

      if (target && (target->cloaked() || target->burrowed()) &&
          !target->detected() && !u->type->isDetector &&
          u->inRangeOf(target, 16)) {
        Unit* detector = utils::getBestScoreCopy(
            g.myUnits,
            [&](Unit* n) {
              if (!n->type->isDetector) {
                return std::numeric_limits<float>::infinity();
              }
              return utils::distance(u, n);
            },
            std::numeric_limits<float>::infinity());
        if (!detector || utils::distance(u, detector) > 4 * 8) {
          runAway = true;
        }
      }

      if (g.isIdleGroup) {
        moveTo = Position(u);
        if (areaInfo.numExpands() <= 1) {
          auto chokes = areaInfo.myBaseArea()->area->ChokePoints();
          for (auto choke : chokes) {
            if (!choke->Blocked()) {
              moveTo = Position(choke->Pos(BWEM::ChokePoint::node::middle));
              if (VLOG_IS_ON(2)) {
                utils::drawLine(
                    state,
                    choke->Pos(BWEM::ChokePoint::node::end1),
                    choke->Pos(BWEM::ChokePoint::node::middle),
                    tc::BW::Color::Blue);
                utils::drawLine(
                    state,
                    choke->Pos(BWEM::ChokePoint::node::end2),
                    choke->Pos(BWEM::ChokePoint::node::middle),
                    tc::BW::Color::Red);
              }
              break;
            }
          }
        } else {
          Unit* hatch = utils::getBestScoreCopy(
              state->unitsInfo().myResourceDepots(),
              [&](Unit* n) { return utils::distance(u, n); });
          if (hatch) {
            moveTo = Position(hatch);
          }
        }
        Tile* tile = state->tilesInfo().tryGetTile(u->x, u->y);
        if (tile && inBaseArea[tile - state->tilesInfo().tiles.data()] != 0) {
          Unit* drone = utils::getBestScoreCopy(
              state->unitsInfo().myWorkers(),
              [&](Unit* n) { return utils::distance(u, n); });
          if (drone && utils::distance(u, drone) <= 4 * 4) {
            Position pos = findMoveAwayPos(u, Position(drone), 4 * 6);
            if (pos != Position()) {
              moveTo = pos;
            }
          }
        }
      }

      auto getrandomcoord = [&](int range) {
        double n = std::normal_distribution<>(range / 2, range / 2)(rngEngine);
        bool neg = n < 0.0;
        n = std::fmod(std::abs(n), range);
        return neg ? range - (int)n : (int)n;
      };

      if (g.isScoutGroup) {
        auto& target = scoutTarget[u];
        moveTo = target.second;
        if (state->currentFrame() - target.first >= 15 * 30 ||
            utils::distance(u->x, u->y, moveTo.x, moveTo.y) <= 4) {
          const BWEM::Area* sourceArea =
              state->map()->GetNearestArea(BWAPI::WalkPosition(u->x, u->y));
          std::vector<std::pair<double, Position>> destinations;
          if (std::uniform_int_distribution<>(0, 256)(rngEngine) <= 40) {
            moveTo.x = getrandomcoord(state->mapWidth());
            moveTo.y = getrandomcoord(state->mapHeight());
          } else if (
              std::uniform_int_distribution<>(0, 256)(rngEngine) <= 200) {
            for (auto& area : state->map()->Areas()) {
              if (area.AccessibleFrom(sourceArea)) {
                for (auto& base : area.Bases()) {
                  BWAPI::WalkPosition walkpos(base.Center());
                  Position pos(walkpos.x, walkpos.y);
                  const Tile& tile = state->tilesInfo().getTile(pos.x, pos.y);
                  double age =
                      std::min(state->currentFrame() - tile.lastSeen, 1003);
                  destinations.emplace_back(age, pos);
                }
              }
            }
          } else {
            for (auto& area : state->map()->Areas()) {
              if (area.AccessibleFrom(sourceArea)) {
                BWAPI::WalkPosition pos = area.Top();
                destinations.emplace_back(1.0, Position(pos.x, pos.y));
              }
            }
          }

          if (!destinations.empty()) {
            double sum = 0.0;
            for (auto& v : destinations) {
              sum += v.first;
            }
            double v = std::uniform_real_distribution<>(0, sum)(rngEngine);
            moveTo = destinations[0].second;
            for (size_t i = 1; i != destinations.size(); ++i) {
              if (v < destinations[i].first) {
                moveTo = destinations[i].second;
                break;
              } else {
                v -= destinations[i].first;
              }
            }
          }

          target.first = state->currentFrame();
          target.second = moveTo;
        }
        fleeUnits.push_back(u);
        move(u, moveTo);
        continue;

        fleeUnits.push_back(u);
        move(u, moveTo);
        continue;
      }

      if (g.searchAndDestroy) {
        auto& sndTarget = searchAndDestroyTarget[u];
        moveTo = sndTarget.second;
        if (state->currentFrame() - sndTarget.first >= 15 * 30 ||
            utils::distance(u->x, u->y, moveTo.x, moveTo.y) <= 4) {
          moveTo.x = getrandomcoord(state->mapWidth());
          moveTo.y = getrandomcoord(state->mapHeight());
          sndTarget.first = state->currentFrame();
          sndTarget.second = moveTo;
        }
        fleeUnits.push_back(u);
        move(u, moveTo);
        continue;
      }

      bool fight = u->flying() ? airFight : groundFight;

      if (!fight && (!u->flying() || anyAntiAir)) {
        if (!target || enemiesInRangeOfOurStaticDefence.find(target) == enemiesInRangeOfOurStaticDefence.end()) {
          if (insideGroup.at(&state->tilesInfo().getTile(u->x, u->y) - tilesData)
                  .group) {
            runAway = true;
          }
        }
      }

      std::shared_ptr<UPCTuple> upc;
      if (runAway) {
        Position runPos = findRunPos(u);
        if (runPos != Position()) {
          ++spotTaken
              [&state->tilesInfo().getTile(runPos.x, runPos.y) - tilesData];
          target = nullptr;
          moveTo = runPos;
        }
      }
      if (!tacticsMicroesUnits && moveTo == g.targetPos &&
          !g.enemyUnits.empty()) {
        fightUnits.push_back(u);
        continue;
      }
      if (target) {
        fightUnits.push_back(u);
        if (!tacticsMicroesUnits) {
          continue;
        }
        if (target->visible && target->isEnemy) {
          upc = utils::makeSharpUPC(u, target, Command::Delete);
        } else {
          upc = utils::makeSharpUPC(u, Position(target), Command::Move);
        }
      } else {
        fleeUnits.push_back(u);
        move(u, moveTo);
      }
      if (upc) {
        state->board()->postUPC(upc, srcUpcId, this);
      }
    }

    // We always directly control fleeting units, so assign them to the task.
    // Attacking units will be assigned to some micro module, unless
    // tacticsMicroesUnits is set.
    std::unordered_set<Unit*> unitSet;
    for (Unit* u : fleeUnits) {
      unitSet.insert(u);
    }
    if (tacticsMicroesUnits) {
      for (Unit* u : fightUnits) {
        unitSet.insert(u);
      }
    }
    if (!leftoverWorkers.empty()) {
      // In order to get workers to go back to mining, we need to grab the unit
      // so that SquadCombat releases it. We will release it next update.
      // It doesn't matter which task we assign it to, so we assign all workers
      // that were previously in a group to the first task we process.
      for (Unit* u : leftoverWorkers) {
        if (wasInAGroup.find(u) != wasInAGroup.end()) {
          unitSet.insert(u);
        }
      }
      leftoverWorkers.clear();
    }
    g.task->setUnits(std::move(unitSet));
    state->board()->updateTasksByUnit(g.task.get());

    if (!fightUnits.empty() && !tacticsMicroesUnits) {

      auto upc = std::make_shared<UPCTuple>();
      for (Unit* u : fightUnits) {
        upc->unit[u] = 1.0f;
      }
      upc->scale = 1;

      for (Unit* e : g.enemyUnits) {
        upc->positionU[e] = 1;
      }
      VLOG(3) << "SCORE FROM TACTICS: " << score;
      VLOG(3) << "My units " << utils::unitsString(fightUnits);
      VLOG(3) << "Their units " << utils::unitsString(nearbyEnemies);
      score = utils::clamp(score / 2 + 1, 0.11, 0.99);
      upc->command[Command::Delete] = score;
      upc->command[Command::Move] = 1 - score;

      state->board()->postUPC(upc, srcUpcId, this);
    }

    /*
    LOG(ERROR) << "Group has enemies " << utils::unitsString(g.enemyUnits)
      << "\n and  allies " << utils::unitsString(g.myUnits)
      << "\n with properties "
      << "hasEnoughUnits = " << g.hasEnoughUnits << ", "
      << "hasEnemyGroundUnits = " << g.hasEnemyGroundUnits << ", "
      << "hasEnemyAirUnits = " << g.hasEnemyAirUnits << ", "
      << "enemyIsOnlyWorkers = " << g.enemyIsOnlyWorkers << ", "
      << "enemiesAreAttacking = " << g.enemiesAreAttacking << ", "
      << "enemiesInOurBase = " << g.enemiesInOurBase << ", "
      << "isAggressiveGroup = " << g.isAggressiveGroup << ", "
      << "isIdleGroup = " << g.isIdleGroup << ", "
      << "score = " << g.score << ", "
      << "searchAndDestroy = " << g.searchAndDestroy << ", "
      <<  "\nshould fight: " << fight;
      */
  }
}

void updateInBaseArea(State* state, std::vector<uint8_t>& inBaseArea) {
  std::fill(inBaseArea.begin(), inBaseArea.end(), 0);

  auto& tilesInfo = state->tilesInfo();
  auto* tilesData = tilesInfo.tiles.data();

  const int mapWidth = state->mapWidth();
  const int mapHeight = state->mapHeight();

  struct OpenNode {
    const Tile* tile;
    const Tile* sourceTile;
    float maxDistance;
  };

  std::vector<Position> staticDefence;
  for (Unit* u :
       state->unitsInfo().myUnitsOfType(buildtypes::Zerg_Sunken_Colony)) {
    staticDefence.emplace_back(u->x, u->y);
  }

  const Tile* mainBaseTile = nullptr;
  std::deque<OpenNode> open;
  bool isMain = true;
  for (Unit* u : state->unitsInfo().myResourceDepots()) {
    auto* tile = tilesInfo.tryGetTile(u->x, u->y);
    if (tile) {
      float maxDistance = isMain ? 4 * 24 : 4 * 14;
      if (isMain) {
        mainBaseTile = tile;
      }
      isMain = false;
      open.push_back({tile, tile, maxDistance});
      inBaseArea.at(tile - tilesData) = 1;
    }
  }
  while (!open.empty()) {
    OpenNode curNode = open.front();
    open.pop_front();

    auto add = [&](const Tile* ntile) {
      if (!curNode.tile->entirelyWalkable) {
        return;
      }

      float sourceDistance = utils::distance(
          ntile->x, ntile->y, curNode.sourceTile->x, curNode.sourceTile->y);
      if (sourceDistance >= curNode.maxDistance) {
        return;
      }

      if (!mainBaseTile) {
        bool staticDefenceIsCloserThanHome = false;
        bool isInStaticDefenceRange = false;
        for (Position pos : staticDefence) {
          float d = utils::distance(ntile->x, ntile->y, pos.x, pos.y);
          if (d <= 4 * 6) {
            isInStaticDefenceRange = true;
          }
          if (d < sourceDistance) {
            staticDefenceIsCloserThanHome = true;
          }
        }
        if (staticDefenceIsCloserThanHome && !isInStaticDefenceRange) {
          return;
        }
      }

      auto& v = inBaseArea[ntile - tilesData];
      if (v) {
        return;
      }
      v = 1;
      open.push_back({ntile, curNode.sourceTile, curNode.maxDistance});
    };

    const Tile* tile = curNode.tile;

    if (tile->x > 0) {
      add(tile - 1);
      if (tile->y > 0) {
        add(tile - 1 - TilesInfo::tilesWidth);
        add(tile - TilesInfo::tilesWidth);
      }
      if (tile->y < mapHeight - tc::BW::XYWalktilesPerBuildtile) {
        add(tile - 1 + TilesInfo::tilesHeight);
        add(tile + TilesInfo::tilesHeight);
      }
    } else {
      if (tile->y > 0) {
        add(tile - TilesInfo::tilesWidth);
      }
      if (tile->y < mapHeight - tc::BW::XYWalktilesPerBuildtile) {
        add(tile + TilesInfo::tilesHeight);
      }
    }
    if (tile->x < mapWidth - tc::BW::XYWalktilesPerBuildtile) {
      add(tile + 1);
      if (tile->y > 0) {
        add(tile + 1 - TilesInfo::tilesWidth);
      }
      if (tile->y < mapHeight - tc::BW::XYWalktilesPerBuildtile) {
        add(tile + 1 + TilesInfo::tilesHeight);
      }
    }
  }
}

namespace {

void updateFleeScore(State* state, std::vector<uint16_t>& fleeScore) {
  std::fill(fleeScore.begin(), fleeScore.end(), 0xffff);

  auto& tilesInfo = state->tilesInfo();
  auto* tilesData = tilesInfo.tiles.data();

  const int mapWidth = state->mapWidth();
  const int mapHeight = state->mapHeight();

  struct OpenNode {
    const Tile* tile;
    uint16_t distance;
  };

  std::deque<OpenNode> open;
  for (Unit* u : state->unitsInfo().myResourceDepots()) {
    auto* tile = tilesInfo.tryGetTile(u->x, u->y);
    if (tile) {
      open.push_back({tile, 1});
      fleeScore.at(tile - tilesData) = 0;
    }
  }
  while (!open.empty()) {
    OpenNode curNode = open.front();
    open.pop_front();

    auto add = [&](const Tile* ntile) {
      if (!curNode.tile->entirelyWalkable) {
        return;
      }

      auto& v = fleeScore[ntile - tilesData];
      if (v != 0xffff) {
        return;
      }
      v = curNode.distance;
      open.push_back({ntile, (uint16_t)(curNode.distance + 1)});
    };

    const Tile* tile = curNode.tile;

    if (tile->x > 0) {
      add(tile - 1);
      if (tile->y > 0) {
        add(tile - 1 - TilesInfo::tilesWidth);
        add(tile - TilesInfo::tilesWidth);
      }
      if (tile->y < mapHeight - tc::BW::XYWalktilesPerBuildtile) {
        add(tile - 1 + TilesInfo::tilesHeight);
        add(tile + TilesInfo::tilesHeight);
      }
    } else {
      if (tile->y > 0) {
        add(tile - TilesInfo::tilesWidth);
      }
      if (tile->y < mapHeight - tc::BW::XYWalktilesPerBuildtile) {
        add(tile + TilesInfo::tilesHeight);
      }
    }
    if (tile->x < mapWidth - tc::BW::XYWalktilesPerBuildtile) {
      add(tile + 1);
      if (tile->y > 0) {
        add(tile + 1 - TilesInfo::tilesWidth);
      }
      if (tile->y < mapHeight - tc::BW::XYWalktilesPerBuildtile) {
        add(tile + 1 + TilesInfo::tilesHeight);
      }
    }
  }
}
} // namespace

void TacticsModule::step(State* state) {
  auto board = state->board();

  auto srcUpcId = findSourceUpc(state);
  if (srcUpcId < 0) {
    VLOG(4) << "No suitable source UPC";
    return;
  }

  board->consumeUPC(srcUpcId, this);

  if (lastUpdateInBaseArea == 0 ||
      state->currentFrame() - lastUpdateInBaseArea >= 60) {
    lastUpdateInBaseArea = state->currentFrame();
    updateInBaseArea(state, inBaseArea);
  }

  if (lastUpdateFleeScore == 0 ||
      state->currentFrame() - lastUpdateFleeScore >= 122) {
    lastUpdateFleeScore = state->currentFrame();
    updateFleeScore(state, fleeScore);
  }

  if (lastProcess == 0 || state->currentFrame() - lastProcess >= 3) {
    lastProcess = state->currentFrame();
    process(state, srcUpcId);
  }
}

UpcId TacticsModule::findSourceUpc(State* state) {
  // Find 'Delete' UPC with unspecified (empty) units
  for (auto& upcs : state->board()->upcsWithCommand(Command::Delete, 0.5)) {
    if (upcs.second->unit.empty()) {
      return upcs.first;
    }
  }
  return -1;
}

} // namespace fairrsh
