/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#include "blackboard.h"
#include "models/bandit.h"
#include "player.h"
#include "state.h"
#include "utils.h"

#include "strategy.h"

#include "../buildorders/registry.h"

#include <gflags/gflags.h>

DEFINE_string(
    build,
    "5pool",
    "Opening/build order to use, if string with spaces, set of build orders"
    "to choose from");
DEFINE_bool(
    bandit,
    false,
    "Use bandits openings/BOs or not? [TODO set true by default?]");

namespace fairrsh {

namespace autobuild {

class FivePoolABTask : public AutoBuildTask // TODO refactor and move
{
 public:
  FivePoolABTask(int upcId) : AutoBuildTask(upcId) {}
  virtual ~FivePoolABTask() {}
};

} // namespace autobuild

namespace strategy {

RTTR_REGISTRATION {
  rttr::registration::class_<StrategyModule>("StrategyModule")(
      metadata("type", rttr::type::get<StrategyModule>()))
      .constructor();
}

/*** selectBO() contextually selects the initial build order out of
 * a fixed set of build orders (going up to ~20-30 pop max).
 * We need to learn (playing against other bots, not in tournament)
 * P(BO | enemyRace, map) by looking ad win rates. Let's model it
 * Gaussian so that we have avg_wr_race_map and stddev_wr_bo_race_map
 * for each BO. We use that to seed (set the prior probabilities of)
 * the parameters of a Bandit (UCB1 or Beta law) over BOs against a
 * given enemyName (that we learn during the tournament).
 */
std::shared_ptr<AutoBuildTask> StrategyModule::selectBO(
    State* state,
    tc::BW::Race enemyRace,
    const std::string& mapName,
    const std::string& enemyName,
    UpcId srcUpcId) {
  using namespace autobuild;
  VLOG(2) << "(selectBO) playing against " << enemyName
          << ", race: " << enemyRace << ", on map: " << mapName;

  VLOG(2) << "Choosing our BO from"; // TODO remove
  // hardcoding FLAGS_build here: not taking a risk that we launch
  // it _with_ command line arguments
  FLAGS_build = "5pool_10hatchling_9poolspeedlingmuta_zvp10hatch_1hatchlurker_meanlingrush_12hatchhydras_2basemutas_3basepoollings_zvpmutas_12poolmuta_2hatchmuta";
  for (auto s : utils::stringSplit(FLAGS_build, '_')) // TODO remove
    VLOG(2) << s; // TODO remove

  if (state->board()->openingBandit != nullptr) {
    std::string buildorder = state->board()->openingBandit->getOpening(
        utils::stringSplit(FLAGS_build, '_'), enemyRace, mapName, enemyName);
    // ^ If you call this more than once per game, we need to talk about the
    // semantics that it has! TODO
    return buildorders::createTask(srcUpcId, buildorder);
  } else {
    return buildorders::createTask(srcUpcId, FLAGS_build);
  }
}

/*** step() chooses the BO and then chooses the units mix (Create UPCs)
 * we want for our army, conditionned on a model and the state.
 * It also should track where to send Delete UPCs to be interpreted by
 * the Tactics module. And contain all the one-off rules that are deemed
 * necessary.
 */
void StrategyModule::step(State* state) {
  using namespace buildtypes;
  using namespace autobuild;
  auto board = state->board();
  // auto enemyRace = tc::BW::Race::Zerg;
  tc::BW::Race enemyRace = tc::BW::Race::Unknown;
  auto enemyRaceBB = tc::BW::Race::_from_integral_nothrow(
      board->get<int>(Blackboard::kEnemyRaceKey));
  if (enemyRaceBB) {
    enemyRace = *enemyRaceBB;
  }

  auto mapName = state->tcstate()->map_name;
  auto enemyName = board->get<std::string>(Blackboard::kEnemyNameKey);
  if (!spawnedBOTask_) {
    spawnedBOTask_ = true;
    // TODO could maybe be in GenericAutoBuildModule so that I don't resort to
    // this trick:
    auto upc = std::make_shared<UPCTuple>();
    upc->scale = 1;
    upc->command[Command::Create] = 1;
    auto upcId = board->postUPC(std::move(upc), -1, this); // fake UPC

    auto boTask = selectBO(state, enemyRace, mapName, enemyName, upcId);
    board->postTask(std::move(boTask), nullptr); // hackety hack
    spawnedBOTask_ = true;
  } else {
    // can do additional checks here
  }
}

void StrategyModule::onGameEnd(State* state) {
  if (state->board()->openingBandit != nullptr) {
    state->board()->openingBandit->onGameEnd(state);
    model::OpeningBandit::save(
        state->board()->openingBandit,
        state->board()->get<std::string>(Blackboard::kEnemyNameKey));
  }
}

} // namespace strategy

} // namespace fairrsh
