/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#include "staticdefencefocusfire.h"

#include "state.h"
#include "unitsinfo.h"
#include "utils.h"

#include <unordered_map>

namespace fairrsh {

RTTR_REGISTRATION {
  rttr::registration::class_<StaticDefenceFocusFireModule>(
      "StaticDefenceFocusFireModule")(
      metadata("type", rttr::type::get<StaticDefenceFocusFireModule>()))
      .constructor();
}

void StaticDefenceFocusFireModule::step(State* state) {

  std::unordered_map<Unit*, double> targetDamage;

  auto run = [&](Unit* u) {
    Unit* target = nullptr;
    double lowestHp = std::numeric_limits<double>::infinity();
    for (Unit* e : state->unitsInfo().visibleEnemyUnits()) {
      if (utils::distance(u, e) < 4 * 12) {
        if (e->inRangeOf(u)) {
          double hp = e->unit.shield + e->unit.health;
          hp -= targetDamage[e];
          if (hp > 0.0 && hp < lowestHp) {
            lowestHp = hp;
            target = e;
          }
        }
      }
    }
    if (target) {
      targetDamage[target] += u->computeHPDamage(target);
      if (!u->unit.orders.empty()) {
        auto& o = u->unit.orders.front();
        if (o.type == tc::BW::Order::AttackUnit && o.targetId == target->id) {
          return;
        }
      }
      if (VLOG_IS_ON(2)) {
        utils::drawLine(
            state, Position(u), Position(target), tc::BW::Color::Red);
      }
      auto cmd = tc::Client::Command(
          tc::BW::Command::CommandUnit,
          u->id,
          tc::BW::UnitCommandType::Attack_Unit,
          target->id);
      state->board()->postCommand(cmd);
    }
  };

  for (Unit* u : state->unitsInfo().myCompletedUnitsOfType(
           buildtypes::Zerg_Sunken_Colony)) {
    run(u);
  }
  for (Unit* u : state->unitsInfo().myCompletedUnitsOfType(
           buildtypes::Zerg_Spore_Colony)) {
    run(u);
  }
  for (Unit* u : state->unitsInfo().myCompletedUnitsOfType(
           buildtypes::Protoss_Photon_Cannon)) {
    run(u);
  }
  for (Unit* u : state->unitsInfo().myCompletedUnitsOfType(
           buildtypes::Terran_Missile_Turret)) {
    run(u);
  }
}

} // namespace fairrsh
