/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#pragma once

#include "buildtype.h"
#include "module.h"
#include "task.h"

namespace fairrsh {

/**
 * Scout management
 */

enum class ScoutingGoal {
  FindEnemyBase = 0,
  ExploreEnemyBase,
  FindEnemyExpand,
  Automatic // decide depending on context
};

constexpr size_t vScoutingMaxNbOverlord = 1;
constexpr size_t vScoutingMaxNbWorkers = 1;
constexpr size_t vScoutingMaxNbExplorers = 1;

class ScoutingModule : public Module {
  RTTR_ENABLE(Module)

 public:
  virtual ~ScoutingModule() = default;

  virtual void step(State* s) override;

  // this should be accessible to the higher-level module
  // or decided in context
  void setScoutingGoal(ScoutingGoal);
  ScoutingGoal goal(State* state) const; // used for automatic decisions

  // that should be part of the higher-level module
  void
  setScoutingPolicy(int maxNbExplorers, int maxNbOverLords, int maxNbWorkers) {
    maxNbExplorers_ = maxNbExplorers;
    maxNbOverLords_ = maxNbOverLords;
    maxNbWorkers_ = maxNbWorkers;
  }

  // communication with other modules through the blackboard
  bool automaticScoutingPolicy(State*);

 protected:
  Unit* findUnit(
      State* state,
      std::unordered_map<Unit*, float> const&,
      Position const& pos);
  bool postTask(
      State* state,
      UpcId baseUpcId,
      Unit* unit,
      Position loc,
      ScoutingGoal goal);
  bool
  postMoveUPC(State* state, UpcId baseUpcId, Unit* unit, const Position& loc);
  Position nextScoutingLocation(
      State* state,
      Unit* unit,
      std::unordered_map<Position, int> const&);
  void updateLocations(
      State* state,
      std::unordered_map<Position, int>&,
      std::vector<Position> const&);
  std::unordered_map<Position, int> startingLocations_;

  ScoutingGoal scoutingGoal_ = ScoutingGoal::Automatic;

  // info that should be taken care of by higher-level module
  UpcId createUPC(
      State* state,
      BuildType const* type,
      std::vector<Position> const& locations);
  void CreateMainUPCs(State* state);

  size_t maxNbOverLords_ = vScoutingMaxNbOverlord;
  size_t maxNbExplorers_ = vScoutingMaxNbExplorers;
  size_t maxNbWorkers_ = vScoutingMaxNbWorkers;
  size_t nbOverlords_ = 0;
  size_t nbWorkers_ = 0;
  size_t nbExplorers_ = 0;
};

} // namespace fairrsh
