/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#include "modules/once.h"

namespace fairrsh {

RTTR_REGISTRATION {
  // Don't register a constructor for this class since its purpose is ad-hoc
  // instantation with a user-supplied function.
  rttr::registration::class_<OnceModule>("OnceModule")(
      metadata("type", rttr::type::get<OnceModule>()));
}

OnceModule::OnceModule(StepFunction fn, std::string name)
    : LambdaModule(std::move(fn), std::move(name)) {
  key_ = name + "_flag_" + std::to_string(reinterpret_cast<uintptr_t>(this));
}

void OnceModule::step(State* state) {
  auto board = state->board();
  if (!board->get<bool>(key_, false)) {
    fn_(state);
    board->post(key_, true);
  }
}

std::shared_ptr<Module> OnceModule::makeWithCommands(
    std::vector<tc::Client::Command> commands,
    std::string name) {
  return Module::make<OnceModule>(
      [commands{std::move(commands)}](State * state) {
        state->board()->postCommands(std::move(commands));
      },
      std::move(name));
}

std::shared_ptr<Module> OnceModule::makeWithSpawns(
    std::vector<SpawnInfo> spawns,
    std::string name) {
  return Module::make<OnceModule>(
      [spawns{std::move(spawns)}](State * state) {
        std::vector<tc::Client::Command> commands;
        for (auto& si : spawns) {
          commands.emplace_back(
              tc::BW::Command::CommandOpenbw,
              tc::BW::OpenBWCommandType::SpawnUnit,
              state->playerId(),
              si.type,
              si.x * tc::BW::XYPixelsPerWalktile,
              si.y * tc::BW::XYPixelsPerWalktile);
        }
        state->board()->postCommands(std::move(commands));
      },
      std::move(name));
}

std::shared_ptr<Module> OnceModule::makeWithEnemySpawns(
    std::vector<SpawnInfo> spawns,
    std::string name) {
  return Module::make<OnceModule>(
      [spawns{std::move(spawns)}](State * state) {
        std::vector<tc::Client::Command> commands;
        for (auto& si : spawns) {
          commands.emplace_back(
              tc::BW::Command::CommandOpenbw,
              tc::BW::OpenBWCommandType::SpawnUnit,
              1 - state->playerId(),
              si.type,
              si.x * tc::BW::XYPixelsPerWalktile,
              si.y * tc::BW::XYPixelsPerWalktile);
        }
        state->board()->postCommands(std::move(commands));
      },
      std::move(name));
}

} // namespace fairrsh
