/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#include "player.h"
#include "state.h"
#include "utils.h"

#include "modules/fivepool.h"

namespace fairrsh {

RTTR_REGISTRATION {
  rttr::registration::class_<FivePoolModule>("FivePoolModule")(
      metadata("type", rttr::type::get<FivePoolModule>()))
      .constructor();
}

FivePoolModule::FivePoolModule() : Module() {
  builds_.emplace_back(buildtypes::Zerg_Drone);
  builds_.emplace_back(buildtypes::Zerg_Spawning_Pool);
  builds_.emplace_back(buildtypes::Zerg_Drone);
  builds_.emplace_back(buildtypes::Zerg_Drone);
  builds_.emplace_back(buildtypes::Zerg_Zergling);
  builds_.emplace_back(buildtypes::Zerg_Zergling);
  builds_.emplace_back(buildtypes::Zerg_Zergling);
  builds_.emplace_back(buildtypes::Zerg_Overlord);
  for (int k = 0; k < 100; ++k) {
    builds_.emplace_back(buildtypes::Zerg_Zergling);
  }
}

void FivePoolModule::step(State* state) {
  auto board = state->board();

  // Find 'Create' UPC with unspecified (empty) createType
  int srcUPCId = -1;
  std::shared_ptr<UPCTuple> srcUPC = nullptr;
  for (auto& upcs : board->upcsWithSharpCommand(Command::Create)) {
    if (upcs.second->createType.empty()) {
      srcUPCId = upcs.first;
      srcUPC = upcs.second;
      break;
    }
  }
  if (srcUPCId < 0) {
    LOG(INFO) << "No suitable source UPC";
    return;
  }

  if (builds_.empty()) {
    VLOG(3) << "Build is done";
    return;
  }

  auto p = builds_.front();

  VLOG(1) << "Post new UPC for " << p->name;
  auto upc = std::make_shared<UPCTuple>();
  auto& builders = state->unitsInfo().myCompletedUnitsOfType(p->builder);
  // Avoid posting UPCs with probability 1 for a single builder since
  // UPCToCommand will directly issue a command.
  auto prob = std::min(1.0f / builders.size(), 0.5f);
  for (Unit* u : builders) {
    upc->unit[u] = prob;
  }
  upc->scale = 1;
  upc->position = UPCTuple::zeroPosition(state).fill_(
      1.0f / (state->mapWidth() * state->mapHeight()));
  upc->command[Command::Create] = 1;
  upc->createType[p] = 1.0f;

  board->consumeUPC(srcUPCId, this);
  board->postUPC(upc, srcUPCId, this);
  builds_.erase(builds_.begin());
}

} // namespace fairrsh
