/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#pragma once

#include "buildtype.h"
#include "fairrsh.h"

#include <functional>
#include <list>
#include <vector>

namespace fairrsh {

struct Unit;
struct UPCTuple;
struct Tile;
class State;

namespace builderhelpers {

/// Find location to construct the building
/// @param state Bot's state
/// @param startLocations Seed locations for building location search
/// @param type Type of the building
/// @param upc UPCTuple with possible restrictions regarding position
/// @return Location to construct the building; (-1, -1) if no suitable
///   location was found
Position findBuildLocation(
    State* state,
    std::vector<Position> const& startLocations,
    BuildType const* type,
    UPCTuple const& upc);

Position findBuildLocation(
    State* state,
    std::vector<Position> const& startLocations,
    BuildType const* type,
    UPCTuple const& upc,
    std::function<double(State*, const BuildType*, const Tile*)> scoreFunc);

/// Check whether the building can be constructed at specified location
/// @param state Bot's state
/// @param type Type of the building
/// @param pos Location to be checked
bool canBuildAt(
    State* state,
    const BuildType* type,
    const Position& pos,
    bool ignoreReserved = false,
    bool logFailure = false);

/// Find a free Vespene Geyser for a refinery
/// @param state Bot's state
/// @param type Building type of the refinery
/// @param upc UPCTuple with possible restrictions regarding position
/// @return Unit that corresponds to the Vespene Geyser on which to
/// build the refinery; nullptr if no suitable Vespene Geyser could be
/// found
Unit* findGeyserForRefinery(
    State* state,
    BuildType const* type,
    UPCTuple const& upc);

/// Find Vespene Geyser location for a refinery
/// @param state Bot's state
/// @param type Building type of the refinery
/// @param upc UPCTuple with possible restrictions regarding position
/// @return Location that corresponds to the Vespene Geyser on which to
/// build the refinery; (-1, -1) if no suitable location could be
/// found
Position
findRefineryLocation(State* state, BuildType const* type, UPCTuple const& upc);

/// Find location for a new resource depot
/// @param state Bot's state
/// @param type Type of the building
/// @param candidateLocations Candidate locations for resource depots,
///   sorted in the order of location preference - the most preferred
///   comes first
/// @return Proposed resource depot location (in walktiles) or (-1, -1),
///  if no suitable location was found
Position findResourceDepotLocation(
    State* state,
    const BuildType* type,
    const std::vector<Position>& candidateLocations,
    bool isExpansion = true);

/// Use map information to produce candidate resource depot locations
/// sorted by their proximity to the main base
std::vector<Position> candidateExpansionResourceDepotLocations(State* state);

/// Use map information to produce candidate resource depot locations
/// sorted by their proximity to the main base
std::list<std::pair<Position, int>>
candidateExpansionResourceDepotLocationsDistances(State* state);

/// Produce seed locations for the building
/// @param state Bot's state
/// @param type Building type
/// @param upc UPCTuple with possible restrictions regarding position
/// @param builder Unit selected to construct the building,
///   nullptr if not yet selected
/// @return Proposed seed locations (in walktiles)
std::vector<Position> buildLocationSeeds(
    State* state,
    BuildType const* type,
    UPCTuple const& upc,
    Unit* builder = nullptr);

/// Produces string representation of masks (buildable, building,
/// reservedAsUnbuildable, resourceDepotUnbuildable, reservedForResourceDepot)
/// around the provided build location. Used for verbose logs mostly for
/// debugging purposes.
/// @param state Bot's state
/// @param type Building type
/// @param pos Coordinates of an upper-left corner of the proposed location
///    (in walktiles)
/// @return String containing masks, where '1' (or '+' outside the building
///    area) stands for true, '0' (or '-') for false, 'X' for cells over
///    the map edge
std::string
buildLocationMasks(State* state, const BuildType* type, const Position& pos);

} // namespace builderhelpers
} // namespace fairrsh
