/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#include "builder.h"
#include "builderhelper.h"

#include "commandtrackers.h"
#include "movefilters.h"
#include "state.h"
#include "utils.h"

#include <cstdint>
#include <deque>
#include <glog/logging.h>

namespace fairrsh {

RTTR_REGISTRATION {
  rttr::registration::class_<BuilderModule>("BuilderModule")(
      metadata("type", rttr::type::get<BuilderModule>()))
      .constructor();
}

namespace {

auto const kMaxBuildAttempts = 3;

void fullReserve(
    TilesInfo& tt,
    const BuildType* type,
    int x,
    int y,
    bool reserve = true) {
  const BuildType* addon = nullptr;
  if (type == buildtypes::Terran_Command_Center) {
    addon = buildtypes::Terran_Comsat_Station;
  } else if (type == buildtypes::Terran_Factory) {
    addon = buildtypes::Terran_Machine_Shop;
  } else if (type == buildtypes::Terran_Starport) {
    addon = buildtypes::Terran_Control_Tower;
  } else if (type == buildtypes::Terran_Science_Facility) {
    addon = buildtypes::Terran_Physics_Lab;
  }
  if (addon) {
    int addonX = x + tc::BW::XYWalktilesPerBuildtile * type->tileWidth;
    int addonY = y +
        tc::BW::XYWalktilesPerBuildtile *
            (type->tileHeight - addon->tileHeight);
    if (reserve) {
      tt.reserveArea(type, x, y);
      tt.reserveArea(addon, addonX, addonY);
    } else {
      tt.unreserveArea(type, x, y);
      tt.unreserveArea(addon, addonX, addonY);
    }
  } else {
    if (reserve)
      tt.reserveArea(type, x, y);
    else
      tt.unreserveArea(type, x, y);
  }
}

void fullUnreserve(TilesInfo& tt, const BuildType* type, int x, int y) {
  fullReserve(tt, type, x, y, false);
}

} // namespace

class BuildTask : public Task {
 public:
  BuildTask(
      int upcId,
      BuildType const* type,
      std::shared_ptr<UPCTuple> sourceUpc)
      : Task(
            upcId,
            {},
            {static_cast<int32_t>(type->mineralCost),
             static_cast<int32_t>(type->gasCost),
             static_cast<int32_t>(type->supplyRequired)}),
        type(type),
        sourceUpc(std::move(sourceUpc)) {}

  virtual void update(State* state) override {
    removeDeadOrReassignedUnits(state);

    if (type->isBuilding && type->builder->isWorker) {
      if (moving && tracker) {
        // Tracking is moving to build location
        switch (tracker->status()) {
          case TrackerStatus::Success:
            // This task didn't succeed yet, we need to start the building now
            // (this is done in BuilderModule).
            lastUpdate = 0;
            moving = false;
            tracker = nullptr;
            VLOG(1) << "BuildTask, " << utils::buildTypeString(type)
                    << " movement tracker reported success, resetting";
            break;
          case TrackerStatus::Cancelled:
            VLOG(2) << "Builder: tracker cancelled but task not cancelled"
                    << " marking task " << upcId() << "as failed";
          case TrackerStatus::Timeout:
          case TrackerStatus::Failure:
            moving = false;
            tracker = nullptr;
            VLOG(1) << "BuildTask, " << utils::buildTypeString(type)
                    << " movement tracker reported timeout/failure";
            break;
          case TrackerStatus::Pending:
          case TrackerStatus::Ongoing:
            setStatus(TaskStatus::Ongoing);
            VLOG(2) << "BuildTask, " << utils::buildTypeString(type)
                    << " movement tracker reported pending/ongoing, "
                       "status->ongoing";
            break;
          default:
            break;
        }
      } else if (!moving && tracker) {
        updateWhileBuilding(state);
      }
    } else if (tracker) {
      updateWhileBuilding(state);
    }

    if (tracker) {
      trackerStatus = tracker->status();
    }

    if (finished()) {
      state->removePlannedResources(resources());
      setResources({});
    }
  }

  virtual void cancel(State* state) override {
    if (!building) {
      state->removePlannedResources(resources());
      setResources({});
      setStatus(TaskStatus::Cancelled);
      setUnits({});
      state->board()->updateTasksByUnit(this);
      if (tracker) {
        tracker->cancel();
        tracker = nullptr;
      }
      if (x != -1 || y != -1) {
        fullUnreserve(state->tilesInfo(), type, x, y);
        x = -1;
        y = -1;
      }
    }
  }

  void updateWhileBuilding(State* state) {
    if (tracker == nullptr) {
      LOG(ERROR) << "No tracker means we're not building??";
      return;
    }

    switch (tracker->status()) {
      case TrackerStatus::Pending:
        setStatus(TaskStatus::Ongoing);
        VLOG(2) << "BuildTask, " << utils::buildTypeString(type)
                << " tracker reported pending, status->ongoing";
        break;
      case TrackerStatus::Ongoing:
        if (trackerStatus == TrackerStatus::Pending) {
          // Minerals and gas will be accounted for in the game as soon as
          // an order is being executed. The tracker switched from pending
          // to ongoing, so we assume the resources don't need to be
          // reserved by the task any more.
          releaseMineralsAndGas(state);
          releaseUsedPsi(state);
          VLOG(2) << "BuildTask, " << utils::buildTypeString(type)
                  << " tracker pending->ongoing, release resources";
        }
        setStatus(TaskStatus::Ongoing);
        VLOG(2) << "BuildTask, " << utils::buildTypeString(type)
                << " tracker reported ongoing, status->ongoing";
        break;
      case TrackerStatus::Success:
        setStatus(TaskStatus::Success);
        VLOG(1) << "BuildTask, " << utils::buildTypeString(type)
                << ", success, finished task for UPC "
                << utils::upcString(upcId());
        state->removePlannedResources(resources());
        setResources({});
        building = false;
        break;
      case TrackerStatus::Timeout:
      case TrackerStatus::Failure:
        // Try to re-issue order on next update
        VLOG(1) << "BuildTask, " << utils::buildTypeString(type)
                << ", tracker timed out/failed, scheduling retry from "
                << utils::upcString(upcId()) << ", status->ongoing";
        tracker = nullptr;
        setStatus(TaskStatus::Ongoing);
        lastUpdate = 0;
        building = false;
        break;
      case TrackerStatus::Cancelled:
        if (status() != TaskStatus::Cancelled) {
          LOG(ERROR) << "BuildTask: canceled tracker without canceled task "
                     << "for UPC " << upcId();
          setStatus(TaskStatus::Failure);
        }
        break;
      default:
        setStatus(TaskStatus::Unknown);
        VLOG(1) << "BuildTask, " << utils::buildTypeString(type)
                << ", status->unknown";
        break;
    }
  }

  void resetLocation(State* state);
  bool setLocation(State* state);

  auto defaultUnitBuilderScore(State* state, BuilderModule* module);
  auto droneBuilderScore(State* state, BuilderModule* module);
  auto hatcheryTechBuilderScore(State* state, BuilderModule* module);
  bool findBuilder(
      State* state,
      BuilderModule* module,
      std::unordered_map<Unit*, float> const& unitProbs);

  const BuildType* type = nullptr;
  std::shared_ptr<UPCTuple> sourceUpc = nullptr;
  int x = -1;
  int y = -1;
  Unit* builder = nullptr;
  Unit* detector = nullptr;
  int lastUpdate = 0;
  int buildAttempts = 0;
  int moveAttempts = 0;
  std::shared_ptr<Tracker> tracker = nullptr;
  TrackerStatus trackerStatus;
  bool moving = false;
  bool building = false;
  int lastMoveUnitsInTheWay = 0;
  int lastSetLocation = 0;
};

// Tries to re-claim the building location
void BuildTask::resetLocation(State* state) {
  if (x < 0 && y < 0) {
    return;
  }

  fullUnreserve(state->tilesInfo(), type, x, y);
  if (!builderhelpers::canBuildAt(state, type, {x, y})) {
    VLOG(1) << "Build " << type->name << ": location is no longer valid";
    x = -1;
    y = -1;
  } else {
    fullReserve(state->tilesInfo(), type, x, y);
  }
}

bool BuildTask::setLocation(State* state) {
  UPCTuple const& upc = *(sourceUpc.get());
  bool tryAtPositionS = false;
  if (type->isBuilding && type->builder->isWorker) {
    if (upc.positionS.x != -1 || upc.positionS.y != -1) {
      // Sorry, but this function fails sometimes when trying to expand
      // with positionS set, and I don't have time to debug it.
      if (builderhelpers::canBuildAt(state, type, upc.positionS, false, true)) {
        fullReserve(state->tilesInfo(), type, upc.positionS.x, upc.positionS.y);
        x = upc.positionS.x;
        y = upc.positionS.y;
        return true;
      } else {
        LOG(INFO) << "Asked to build " << utils::buildTypeString(type) << " at "
                  << upc.positionS.x << "," << upc.positionS.y
                  << " but can't build there. Let's see if we can place it "
                     "somewhere closeby";
        tryAtPositionS = true;
      }
    }
    Position p(-1, -1);
    if (type->isRefinery) {
      p = builderhelpers::findRefineryLocation(state, type, upc);
    } else {
      if (tryAtPositionS) {
        auto spos = upc.positionS * upc.scale;
        // positionS is given but we couldn't build there (see above). Try to
        // find another valid position that's close.
        p = builderhelpers::findBuildLocation(
            state,
            {spos},
            type,
            upc,
            [&](State* state, const BuildType* type, const Tile* tile) {
              return utils::distance(tile->x, tile->y, spos.x, spos.y);
            });
      } else {
        auto seeds =
            builderhelpers::buildLocationSeeds(state, type, upc, builder);
        p = builderhelpers::findBuildLocation(state, seeds, type, upc);
      }
    }

    if (p.x == -1 && p.y == -1) {
      LOG(WARNING) << "Build " << utils::buildTypeString(type)
                   << ": failed to find build location";
      return false;
    } else {
      VLOG(3) << "build " << utils::buildTypeString(type)
              << ": found build location: " << p.x << ", " << p.y;
      fullReserve(state->tilesInfo(), type, p.x, p.y);
      x = p.x;
      y = p.y;
      return true;
    }
  }

  return true; // Don't need an x/y location
}

/// Returns scoring function for selecting a unit to build another
/// (non-building) unit
auto BuildTask::defaultUnitBuilderScore(State* state, BuilderModule* module) {
  return [this, state, module](Unit* u) {
    if (u->type != type->builder) {
      return std::numeric_limits<double>::infinity();
    }
    if (!u->active()) {
      return std::numeric_limits<double>::infinity();
    }
    if (type->isAddon && u->addon) {
      return std::numeric_limits<double>::infinity();
    }

    double r = 0.0;
    if (builder == u) {
      r -= 10.0;
    } else {
      auto tdata = state->board()->taskDataWithUnit(u);
      if (tdata.owner == module) {
        // We're already building something with this one
        return std::numeric_limits<double>::infinity();
      }
    }
    r += u->remainingBuildTrainTime + u->remainingUpgradeResearchTime;
    return r;
  };
}

/// Returns scoring function for selecting a unit to build a drone.
auto BuildTask::droneBuilderScore(State* state, BuilderModule* module) {
  // Compute hatchery counts of Larvae
  std::unordered_map<int, int> larvaCount;
  for (Unit* u :
       state->unitsInfo().myCompletedUnitsOfType(buildtypes::Zerg_Larva)) {
    if (u->associatedUnit) {
      larvaCount[u->associatedUnit->id] += 1;
    }
  }

  return [ this, state, module, larvaCount = std::move(larvaCount) ](Unit * u) {
    if (u->type != type->builder) {
      return std::numeric_limits<double>::infinity();
    }
    if (!u->active()) {
      return std::numeric_limits<double>::infinity();
    }
    if (type->isAddon && u->addon) {
      return std::numeric_limits<double>::infinity();
    }

    double r = 0.0;
    if (builder == u) {
      r -= 10.0;
    } else {
      auto tdata = state->board()->taskDataWithUnit(u);
      if (tdata.owner == module) {
        // We're already building something with this one
        return std::numeric_limits<double>::infinity();
      }

      // Better build at a hatchery with lots of larva so that we'll get new
      // ones.
      int constexpr kMaxLarva = 3;
      if (u->associatedUnit &&
          u->associatedUnit->type == buildtypes::Zerg_Hatchery) {
        auto it = larvaCount.find(u->associatedUnit->id);
        if (it != larvaCount.end()) {
          int count = it->second;
          r += (kMaxLarva - count); // (1,3)
        }
      }

      // Build at bases where we have low saturation
      auto& areaInfo = state->areaInfo();
      auto baseId = areaInfo.myClosestExpand(u);
      auto saturation = areaInfo.mineralSaturationAtMyExpand(baseId);
      r += saturation; // (0, 1)
    }
    r += u->remainingBuildTrainTime + u->remainingUpgradeResearchTime;
    return r;
  };
}

/// Returns scoring function for selecting a unit to morph a hatchery or lair
auto BuildTask::hatcheryTechBuilderScore(State* state, BuilderModule* module) {
  return [this, state, module](Unit* u) {
    if (u->type != type->builder) {
      return std::numeric_limits<double>::infinity();
    }
    if (!u->active()) {
      return std::numeric_limits<double>::infinity();
    }
    if (type->isAddon && u->addon) {
      return std::numeric_limits<double>::infinity();
    }

    double r = 0.0;
    if (builder == u) {
      r -= 10.0;
    } else {
      auto tdata = state->board()->taskDataWithUnit(u);
      if (tdata.owner == module) {
        // We're already building something with this one
        return std::numeric_limits<double>::infinity();
      }
    }

    // Prefer Lair and Hive in early bases
    auto baseNumber = state->areaInfo().myClosestExpand(u);
    r += baseNumber * 10;

    r += u->remainingBuildTrainTime + u->remainingUpgradeResearchTime;
    return r;
  };
}

bool BuildTask::findBuilder(
    State* state,
    BuilderModule* module,
    std::unordered_map<Unit*, float> const& unitProbs) {
  auto board = state->board();

  if (type->isBuilding && type->builder->isWorker) {
    if (x != -1 || y != -1) {
      auto builderScore = [&](Unit* u) {
        if (u->type != type->builder) {
          return std::numeric_limits<double>::infinity();
        }
        if (!u->active()) {
          return std::numeric_limits<double>::infinity();
        }

        double r = 0.0;
        if (builder == u) {
          r -= 10.0;
        } else {
          auto tdata = board->taskDataWithUnit(u);
          if (tdata.owner == module) {
            // We're already building something with this one
            return std::numeric_limits<double>::infinity();
          }
          if (tdata.task && tdata.owner &&
              (tdata.owner->name().find("Scouting") != std::string::npos ||
               tdata.owner->name().find("Harass") != std::string::npos)) {
            return std::numeric_limits<double>::infinity();
          }
          if (!u->idle() && !u->unit.orders.empty()) {
            if (u->unit.orders.front().type == tc::BW::Order::MoveToMinerals) {
              r += 15.0;
            } else if (
                u->unit.orders.front().type == tc::BW::Order::ReturnMinerals) {
              r += 60.0;
            } else if (
                u->unit.orders.front().type == tc::BW::Order::MoveToGas) {
              r += 75.0;
            } else if (
                u->unit.orders.front().type == tc::BW::Order::ReturnGas) {
              r += 90.0;
            } else {
              r += 150.0;
            }
          }
          auto i = module->recentAssignedBuilders.find(u);
          if (i != module->recentAssignedBuilders.end()) {
            if (std::get<1>(i->second) == type &&
                utils::distance(
                    std::get<2>(i->second).x, std::get<2>(i->second).y, x, y) <=
                    4 * 12) {
              r -= 1000.0;
            }
          }
        }
        r += utils::distance(u->x, u->y, x, y) / u->topSpeed;
        return r;
      };

      if (builder &&
          builderScore(builder) == std::numeric_limits<double>::infinity()) {
        builder->busyUntil = 0;
        builder = nullptr;
      }

      if (!builder) {
        std::vector<Unit*> candidates;
        for (auto it : unitProbs) {
          if (it.second > 0 && it.first->type == type->builder) {
            candidates.push_back(it.first);
          }
        }
        if (!candidates.empty()) {
          builder = utils::getBestScoreCopy(
              std::move(candidates),
              builderScore,
              std::numeric_limits<double>::infinity());
        } else {
          builder = utils::getBestScoreCopy(
              state->unitsInfo().myCompletedUnitsOfType(type->builder),
              builderScore,
              std::numeric_limits<double>::infinity());
        }
      }
    } else if (builder) {
      builder = nullptr;
    }

  } else {
    std::function<double(Unit*)> builderScore;
    if (type == buildtypes::Zerg_Drone) {
      // Build new drones at bases that contain lots of larva.
      builderScore = droneBuilderScore(state, module);
    } else if (type == buildtypes::Zerg_Lair || type == buildtypes::Zerg_Hive) {
      builderScore = hatcheryTechBuilderScore(state, module);
    } else {
      builderScore = defaultUnitBuilderScore(state, module);
    }

    if (builder &&
        builderScore(builder) == std::numeric_limits<double>::infinity()) {
      builder->busyUntil = 0;
      builder = nullptr;
    }

    if (!builder) {
      std::vector<Unit*> candidates;
      for (auto it : unitProbs) {
        if (it.second > 0 && it.first->type == type->builder) {
          candidates.push_back(it.first);
        }
      }

      if (!candidates.empty()) {
        builder = utils::getBestScoreCopy(
            std::move(candidates),
            builderScore,
            std::numeric_limits<double>::infinity());
      } else {
        builder = utils::getBestScoreCopy(
            state->unitsInfo().myCompletedUnitsOfType(type->builder),
            builderScore,
            std::numeric_limits<double>::infinity());
      }
    }
  }

  return builder != nullptr;
}

void BuilderModule::step(State* state) {
  if (state->currentFrame() - lastIncomeHistoryUpdate_ >= 8) {
    int framesSinceLastUpdate =
        state->currentFrame() - lastIncomeHistoryUpdate_;
    lastIncomeHistoryUpdate_ = state->currentFrame();

    static const int resourcePerFrameAverageSize = 15 * 4;
    auto calculateAverage = [&](double& dst, auto& hist, int current) {
      for (int i = 0; i != framesSinceLastUpdate; ++i) {
        if (hist.size() >= resourcePerFrameAverageSize) {
          hist.pop_front();
        }
        hist.push_back(current);
      }
      int totalDifference = 0;
      for (size_t i = 1; i < hist.size(); ++i) {
        totalDifference += std::max(hist[i] - hist[i - 1], 0);
      }
      dst = (double)totalDifference / hist.size();
    };
    auto res = state->resources();
    calculateAverage(currentMineralsPerFrame_, mineralsHistory_, res.ore);
    calculateAverage(currentGasPerFrame_, gasHistory_, res.gas);
  }

  auto board = state->board();

  int frame = state->currentFrame();

  for (auto const& upct : board->upcsWithSharpCommand(Command::Create)) {
    auto upcId = upct.first;
    auto& upc = *upct.second;

    // Do we know what we want?
    if (upc.createType.size() != 1) {
      VLOG(4) << "Not sure what we want? createType.size() ="
              << upc.createType.size();
      continue;
    }

    auto task = createTask(state, upct.second, upcId);
    if (task) {
      board->consumeUPC(upcId, this);
      auto btask = std::static_pointer_cast<BuildTask>(task);
      VLOG(1) << "New task for " << utils::upcString(upcId) << " for "
              << btask->type->name << " and planned resources "
              << utils::resourcesString(btask->resources());
      board->postTask(task, this, true);
      state->addPlannedResources(task->resources());
      VLOG(1) << "Resources now: "
              << utils::resourcesString(state->resources());
    }
  }

  for (auto i = recentAssignedBuilders.begin();
       i != recentAssignedBuilders.end();) {
    if (state->currentFrame() - std::get<0>(i->second) >= 15 * 25) {
      i = recentAssignedBuilders.erase(i);
    } else {
      ++i;
    }
  }

  std::vector<uint8_t> visited;
  uint8_t visitedN = 0;

  auto findMoveAwayPos = [&](Unit* u, Position source, float distance) {
    if (visited.empty()) {
      visited.resize(TilesInfo::tilesWidth * TilesInfo::tilesHeight);
    }
    const int mapWidth = state->mapWidth();
    const int mapHeight = state->mapHeight();
    bool flying = u->flying();

    uint8_t visitedValue = ++visitedN;

    auto* tilesData = state->tilesInfo().tiles.data();

    Position startPos(u->x, u->y);

    std::deque<const Tile*> open;
    open.push_back(&state->tilesInfo().getTile(u->x, u->y));
    while (!open.empty()) {
      const Tile* tile = open.front();
      open.pop_front();

      if (utils::distance(tile->x, tile->y, source.x, source.y) >= distance) {
        return Position(tile->x, tile->y);
      }

      auto add = [&](const Tile* ntile) {
        if (!flying && (!tile->entirelyWalkable || tile->building)) {
          return;
        }
        auto& v = visited[ntile - tilesData];
        if (v == visitedValue) {
          return;
        }
        v = visitedValue;
        if (utils::distance(ntile->x, ntile->y, startPos.x, startPos.y) <=
            4 * 20) {
          open.push_back(ntile);
        }
      };

      if (tile->x > 0) {
        add(tile - 1);
      }
      if (tile->y > 0) {
        add(tile - TilesInfo::tilesWidth);
      }
      if (tile->x < mapWidth - tc::BW::XYWalktilesPerBuildtile) {
        add(tile + 1);
      }
      if (tile->y < mapHeight - tc::BW::XYWalktilesPerBuildtile) {
        add(tile + TilesInfo::tilesHeight);
      }
    }

    return Position();
  };

  // Update tasks
  auto tasks = state->board()->tasksOfModule(this);
  std::sort(tasks.begin(), tasks.end(), [](auto& a, auto& b) {
    return a->upcId() < b->upcId();
  });
  auto tcs = state->tcstate();
  auto res = tcs->frame->resources[tcs->player_id];
  for (auto& v : tasks) {
    auto btask = std::static_pointer_cast<BuildTask>(v);
    auto type = btask->type;

    btask->builder = nullptr;
    btask->detector = nullptr;
    for (Unit* u : ((const Task*)&*btask)->units()) {
      if (u->type == type->builder) {
        btask->builder = u;
      } else if (u->type->isDetector) {
        btask->detector = u;
      }
    }

    if (btask->builder) {
      if (VLOG_IS_ON(2)) {
        utils::drawLine(state, btask->builder, btask.get());
      }
    }

    if (btask->lastMoveUnitsInTheWay &&
        frame - btask->lastMoveUnitsInTheWay >= 30) {
      btask->lastMoveUnitsInTheWay = 0;
      if (btask->detector && btask->builder) {
        btask->setUnits({btask->builder, btask->detector});
      } else if (btask->builder) {
        btask->setUnits({btask->builder});
      }
      state->board()->updateTasksByUnit(btask.get());
    }

    auto reserved = btask->resources();
    res.ore -= reserved.ore;
    res.gas -= reserved.gas;

    // Update required?
    if (btask->lastUpdate > 0 && frame - btask->lastUpdate < 4) {
      continue;
    }
    if (btask->finished()) {
      continue;
    }
    btask->lastUpdate = frame;

    bool buildNow = true;
    bool moveOnly = false;

    if (reserved.ore && res.ore < 0) {
      moveOnly = true;
    }
    if (reserved.gas && res.gas < 0) {
      moveOnly = true;
    }
    if (reserved.used_psi && res.used_psi + reserved.used_psi > res.total_psi) {
      moveOnly = true;
    }
    if (!moveOnly) {
      if (!utils::prerequisitesReady(state, type)) {
        moveOnly = true;
      }
    }

    // Re-claim building location if we haven't started to build yet, find new
    // one if needed
    if (!btask->building) {
      if (frame - btask->lastSetLocation >= 11) {
        btask->lastSetLocation = frame;
        btask->resetLocation(state);
        if (btask->x < 0 || btask->y < 0) {
          btask->setLocation(state);
        }
      }
    }

    if (btask->builder && (btask->x != -1 || btask->y != -1)) {
      recentAssignedBuilders[btask->builder] = std::make_tuple(
          state->currentFrame(), type, Position(btask->x, btask->y));
    }

    if (buildNow) {
      if (!btask->builder && (!moveOnly || type->builder->isWorker) &&
          !btask->building) {
        btask->findBuilder(state, this, btask->sourceUpc->unit);
        if (btask->builder) {
          if (moveOnly && (btask->x != -1 || btask->y != -1)) {
            double t = 0.0;
            if (type->mineralCost) {
              t = std::max(t, -res.ore / currentMineralsPerFrame_);
            }
            if (type->gasCost) {
              t = std::max(t, -res.gas / currentGasPerFrame_);
            }
            if (t >
                utils::distance(
                    btask->builder->x, btask->builder->y, btask->x, btask->y) /
                    btask->builder->type->topSpeed) {
              btask->builder = nullptr;
            }
          }
          if (btask->builder) {
            VLOG(1) << "Found builder: " << utils::unitString(btask->builder)
                    << " for " << type->name << " from "
                    << utils::upcString(btask->upcId());
            btask->setUnits({btask->builder});
            state->board()->updateTasksByUnit(btask.get());
          }
        } else {
          VLOG(1) << "Could not determine builder right now for " << type->name
                  << " from " << utils::upcString(btask->upcId());
          btask->setUnits({});
          state->board()->updateTasksByUnit(btask.get());
        }
      }
      if (type->isBuilding && type->builder->isWorker) {
        if (btask->builder && (btask->x != -1 || btask->y != -1)) {

          if (!btask->detector) {
            Unit* detector = utils::getBestScoreCopy(
                state->unitsInfo().myUnits(),
                [&](Unit* u) {
                  if (!u->type->isDetector || u->type->isBuilding ||
                      !u->active() || state->board()->taskWithUnit(u)) {
                    return std::numeric_limits<double>::infinity();
                  }
                  return (double)utils::distance(
                      u->x, u->y, btask->x, btask->y);
                },
                std::numeric_limits<double>::infinity());
            if (detector) {
              btask->setUnits({btask->builder, detector});
              state->board()->updateTasksByUnit(btask.get());
            }
          } else {
            auto tgt = movefilters::safeMoveTo(
                state, btask->detector, Position(btask->x, btask->y));
            if (tgt.x < 0 || tgt.y < 0) {
              VLOG(1) << "detector stuck";
              tgt = Position(btask->x, btask->y);
            } else if (tgt.dist(btask->detector->getMovingTarget()) > 4) {
              // condition made to avoid sending too many commands
              UPCTuple newUPC;
              newUPC.unit[btask->detector] = 1;
              newUPC.positionS = tgt;
              newUPC.command[Command::Move] = 1;
              board->postUPC(
                  std::make_shared<UPCTuple>(std::move(newUPC)),
                  btask->upcId(),
                  this);
            }
          }

          // hack: we should not rely on movement tracker
          // re-send move every now and then if not close to destination
          float distFromBuilderToDestThreshold = 4.0f * 2;
          if (type->isRefinery) {
            distFromBuilderToDestThreshold = 4.0f * 6;
          }
          Position targetPosition(
              std::min(
                  btask->x +
                      type->tileWidth * tc::BW::XYWalktilesPerBuildtile / 2,
                  state->mapWidth() - 1),
              std::min(
                  btask->y +
                      type->tileHeight * tc::BW::XYWalktilesPerBuildtile / 2,
                  state->mapHeight() - 1));
          auto distFromBuilderToDest = utils::distance(
              btask->builder->x,
              btask->builder->y,
              targetPosition.x,
              targetPosition.y);

          if (btask->tracker == nullptr ||
              distFromBuilderToDest >= distFromBuilderToDestThreshold) {
            UPCTuple newUPC;
            newUPC.unit[btask->builder] = 1;
            if (distFromBuilderToDest >= distFromBuilderToDestThreshold) {
              auto tgt = movefilters::safeMoveTo(
                  state, btask->builder, targetPosition);
              if (tgt.x < 0 || tgt.y < 0) {
                VLOG(1) << "builder stuck";
                tgt = targetPosition; // well, that won't do anything
              }
              if (tgt.dist(btask->builder->getMovingTarget()) > 4) {
                newUPC.positionS = tgt;
                newUPC.command[Command::Move] = 1;

                board->postUPC(
                    std::make_shared<UPCTuple>(std::move(newUPC)),
                    btask->upcId(),
                    this);
                if (!btask->tracker) {
                  btask->tracker = state->addTracker<MovementTracker>(
                      std::initializer_list<Unit*>{btask->builder},
                      targetPosition.x,
                      targetPosition.y,
                      distFromBuilderToDestThreshold);
                  btask->trackerStatus = btask->tracker->status();
                  btask->moving = true;
                  VLOG(3) << "BuildTask, " << utils::buildTypeString(type)
                          << ", using MovementTracker, distance="
                          << distFromBuilderToDest
                          << ", threshold=" << distFromBuilderToDestThreshold;
                }
              }
            } else if (!moveOnly) {

              Unit* killUnit = nullptr;
              int movedUnits = 0;
              for (Unit* e : state->unitsInfo().visibleUnits()) {
                if (!e->flying() && !e->invincible() && e != btask->builder &&
                    e->detected()) {
                  auto d = utils::pxDistanceBB(
                      e->unit.pixel_x - e->type->dimensionLeft,
                      e->unit.pixel_y - e->type->dimensionUp,
                      e->unit.pixel_x + e->type->dimensionRight,
                      e->unit.pixel_y + e->type->dimensionDown,
                      btask->x * tc::BW::XYPixelsPerWalktile,
                      btask->y * tc::BW::XYPixelsPerWalktile,
                      (btask->x +
                       type->tileWidth * tc::BW::XYWalktilesPerBuildtile) *
                          tc::BW::XYPixelsPerWalktile,
                      (btask->y +
                       type->tileHeight * tc::BW::XYWalktilesPerBuildtile) *
                          tc::BW::XYPixelsPerWalktile);
                  if (e->isMine && !e->type->isNonUsable && d <= 16) {
                    Position target =
                        findMoveAwayPos(e, Position(btask->builder), 16);

                    btask->lastMoveUnitsInTheWay = state->currentFrame();
                    auto units = ((const Task*)&*btask)->units();
                    units.insert(e);
                    btask->setUnits(units);
                    state->board()->updateTasksByUnit(btask.get());

                    board->postUPC(
                        utils::makeSharpUPC(e, target, Command::Move),
                        btask->upcId(),
                        this);

                    ++movedUnits;

                    continue;
                  }
                  if (d <= 0) {
                    killUnit = e;
                    break;
                  }
                }
              }

              if (killUnit) {
                if (btask->detector) {
                  btask->setUnits({btask->builder, btask->detector});
                } else {
                  btask->setUnits({btask->builder});
                }
                state->board()->updateTasksByUnit(btask.get());
                newUPC.positionU[killUnit] = 1;
                newUPC.command[Command::Delete] = 1;
              } else {
                newUPC.positionS = Position(btask->x, btask->y);
                newUPC.command[Command::Create] = 1;
                if (movedUnits) {
                  ++btask->moveAttempts;
                }
                if (!movedUnits || btask->moveAttempts >= 12) {
                  ++btask->buildAttempts;
                }
                if (btask->buildAttempts > kMaxBuildAttempts) {
                  btask->buildAttempts = 0;
                  for (int y = btask->y; y !=
                       btask->y +
                           tc::BW::XYWalktilesPerBuildtile * type->tileHeight;
                       ++y) {
                    for (int x = btask->x; x !=
                         btask->x +
                             tc::BW::XYWalktilesPerBuildtile * type->tileWidth;
                         ++x) {
                      Tile* t = state->tilesInfo().tryGetTile(x, y);
                      if (t) {
                        t->blockedUntil = std::max(
                            t->blockedUntil, state->currentFrame() + 15 * 30);
                      }
                    }
                  }
                }
                newUPC.createType[type] = 1;

                btask->tracker =
                    state->addTracker<BuildTracker>(btask->builder, type, 8);
                btask->trackerStatus = btask->tracker->status();
                btask->building = true;
              }

              board->postUPC(
                  std::make_shared<UPCTuple>(std::move(newUPC)),
                  btask->upcId(),
                  this);
              VLOG(3) << "BuildTask, using BuildTracker, distance="
                      << utils::distance(
                             btask->builder->x,
                             btask->builder->y,
                             btask->x,
                             btask->y);
            }
          }
        }
      } else if (!moveOnly) {
        if (btask->tracker == nullptr && btask->builder) {
          ++btask->buildAttempts;
          if (btask->buildAttempts > kMaxBuildAttempts) {
            VLOG(1) << "Tried to build " << type->name << " "
                    << btask->buildAttempts - 1 << " times; not giving up!";
            btask->buildAttempts = 0;
          } else {
            UPCTuple newUPC;
            newUPC.unit[btask->builder] = 1;
            newUPC.positionS = Position(btask->builder->x, btask->builder->y);
            newUPC.command[Command::Create] = 1;
            newUPC.createType[type] = 1;

            board->postUPC(
                std::make_shared<UPCTuple>(std::move(newUPC)),
                btask->upcId(),
                this);
            if (type->isUpgrade()) {
              btask->tracker = state->addTracker<UpgradeTracker>(
                  btask->builder, btask->type, 15);
              btask->trackerStatus = btask->tracker->status();
            } else if (type->isTech()) {
              btask->tracker = state->addTracker<ResearchTracker>(
                  btask->builder, btask->type, 15);
              btask->trackerStatus = btask->tracker->status();
            } else {
              btask->tracker = state->addTracker<BuildTracker>(
                  btask->builder, btask->type, 15);
              btask->trackerStatus = btask->tracker->status();
            }
            btask->building = true;
          }
        }
      }
    }
  }

  // Any tasks that couldn't be updated for some time?
  for (auto& v : state->board()->tasksOfModule(this)) {
    BuildTask* b = (BuildTask*)&*v;
    if (b->finished()) {
      // Unreserve area for finished tasks
      if (b->x != -1 || b->y != -1) {
        fullUnreserve(state->tilesInfo(), b->type, b->x, b->y);
        b->x = -1;
        b->y = -1;
      }
    }
  }

  Module::step(state);
}

std::shared_ptr<Task> BuilderModule::createTask(
    State* state,
    std::shared_ptr<UPCTuple> upc,
    int upcId) {
  const BuildType* type = upc->createType.begin()->first;
  if (!type || !type->builder) {
    VLOG(4) << "Don't know how to build " << type->name << ", skipping "
            << utils::upcString(upcId);
    return nullptr;
  }

  return std::make_shared<BuildTask>(upcId, type, upc);
}

} // namespace fairrsh
