/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#pragma once

#include "buildtype.h"
#include "fairrsh.h"
#include "module.h"
#include "state.h"

#include <array>
#include <deque>
#include <functional>
#include <unordered_map>
#include <unordered_set>
#include <vector>

namespace fairrsh {

namespace autobuild {

struct BuildStateUnit {
  const BuildType* type = nullptr;
  int busyUntil = 0;
  const BuildType* addon = nullptr;
  int larvaTimer = 0;
};

struct BuildEntry {
  const BuildType* type;
  Position pos;
  std::function<void()> builtCallback;

  bool operator==(BuildEntry n) const {
    return type == n.type && pos == n.pos;
  }
  bool operator!=(BuildEntry n) const {
    return type != n.type || pos != n.pos;
  }
};

struct BuildState {
  int frame = 0;
  int race = 0;
  double minerals = 0;
  double gas = 0;
  std::array<double, 3> usedSupply{};
  std::array<double, 3> maxSupply{};
  std::array<double, 3> inprodSupply{};
  std::unordered_map<const BuildType*, std::vector<BuildStateUnit>> units;
  std::unordered_set<const BuildType*> upgradesAndTech;
  std::deque<std::pair<int, const BuildType*>> production;
  std::vector<std::pair<int, BuildEntry>> buildOrder;

  int workers = 0;
  int refineries = 0;
  int availableGases = 0;

  bool autoBuildRefineries = true;
  bool autoBuildHatcheries = false;
  bool isExpanding = false;
};

BuildState getMyState(State* state);

bool hasUnit(const BuildState& st, const BuildType* type);
bool hasUpgrade(const BuildState& st, const BuildType* type);
bool hasTech(const BuildState& st, const BuildType* type);
bool has(const BuildState& st, const BuildType* type);
int countUnits(const BuildState& st, const BuildType* type);
bool isInProduction(const BuildState& st, const BuildType* type);
bool hasOrInProduction(const BuildState& st, const BuildType* type);
int countProduction(const BuildState& st, const BuildType* type);
int countPlusProduction(const BuildState& st, const BuildType* type);
int larvaCount(const BuildState& st, const BuildStateUnit& u);
}

/**
 * A task for AutoBuildModule.
 *
 * TODO: Describe
 */
class AutoBuildTask : public MultiProxyTask {
 public:
  int lastEvaluate = 0;
  // Each of this UPCs is being proxied by this task
  std::unordered_map<UpcId, autobuild::BuildEntry> scheduledUpcs;

  std::function<bool(autobuild::BuildState&)> queue;
  const autobuild::BuildState* currentBuildSt = nullptr;

  AutoBuildTask(int upcId) : MultiProxyTask({}, upcId) {
    setStatus(TaskStatus::Ongoing);
  }
  virtual ~AutoBuildTask() {}

  virtual void update(State* state) override;

  void evaluate(State* state, Module* module);

  virtual void preBuild(State*, Module*) {}
  virtual void buildStep(autobuild::BuildState& st) {}
  virtual void postBuild(State*) {}

  void build(
      const BuildType* type,
      Position pos,
      std::function<void()> builtCallback);
  void build(const BuildType* type, std::function<void()> builtCallback);
  void build(const BuildType* type, Position pos);
  void build(const BuildType* type);
  bool buildN(const BuildType* type, int n);
  bool upgrade(const BuildType* type);
};

class AutoBuildModule : public Module {
  RTTR_ENABLE(Module)

 public:
  virtual ~AutoBuildModule() = default;

  virtual std::shared_ptr<AutoBuildTask> createTask(int srcUpcId);

  virtual void checkForNewUPCs(State* state);

  virtual void step(State* state) override;
};

class DefaultAutoBuildTask : public AutoBuildTask {
 public:
  struct Target {
    BuildType const* type;
    int n;

    Target(BuildType const* type, int n = -1) : type(type), n(n) {}
  };

  std::vector<Target> targets;

  DefaultAutoBuildTask(int upcId, std::vector<Target> targets)
      : AutoBuildTask(upcId), targets(std::move(targets)) {}
  virtual ~DefaultAutoBuildTask() {}

  virtual void buildStep(autobuild::BuildState& st) override {
    for (auto& target : targets) {
      if (target.n == -1) {
        build(target.type);
      } else {
        buildN(target.type, target.n);
      }
    }
  }
};

} // namespace fairrsh
