/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#include "autobuild.h"
#include "builderhelper.h"
#include "state.h"
#include "utils.h"

#include <BWAPI.h>
#include <gflags/gflags.h>
#include <glog/logging.h>

#include <algorithm>
#include <functional>
#include <iterator>
#include <stdexcept>
#include <tuple>
#include <utility>

DEFINE_bool(
    verbose_build_logging,
    false,
    "Enable (very) verbose logging of the build steps");
DEFINE_bool(
    log_build_order,
    false,
    "Log the build order every time it is calculated");

namespace fairrsh {

RTTR_REGISTRATION {
  rttr::registration::class_<AutoBuildModule>("AutoBuildModule")(
      metadata("type", rttr::type::get<AutoBuildModule>()))
      .constructor();
}

namespace autobuild {

thread_local int buildLogDepth = 0;
std::string buildLogIndent() {
  return std::string(buildLogDepth * 2, ' ');
}

bool hasUnit(const BuildState& st, const BuildType* type) {
  auto i = st.units.find(type);
  if (i == st.units.end()) {
    return false;
  }
  return !i->second.empty();
}

bool hasUpgrade(const BuildState& st, const BuildType* type) {
  return st.upgradesAndTech.find(type) != st.upgradesAndTech.end();
}

bool hasTech(const BuildState& st, const BuildType* type) {
  return st.upgradesAndTech.find(type) != st.upgradesAndTech.end();
}

bool has(const BuildState& st, const BuildType* type) {
  if (type->isUnit()) {
    return hasUnit(st, type);
  } else {
    return st.upgradesAndTech.find(type) != st.upgradesAndTech.end();
  }
}

int countUnits(const BuildState& st, const BuildType* type) {
  auto i = st.units.find(type);
  if (i == st.units.end()) {
    return 0;
  }
  return i->second.size();
}

bool isInProduction(const BuildState& st, const BuildType* type) {
  for (auto& v : st.production) {
    if (v.second == type) {
      return true;
    }
  }
  return false;
}

bool hasOrInProduction(const BuildState& st, const BuildType* type) {
  return has(st, type) || isInProduction(st, type);
}

int countProduction(const BuildState& st, const BuildType* type) {
  int r = 0;
  for (auto& v : st.production) {
    if (v.second == type) {
      ++r;
    }
  }
  return r;
}
int countPlusProduction(const BuildState& st, const BuildType* type) {
  int r = 0;
  if (type->isUnit()) {
    r += countUnits(st, type);
  } else if (has(st, type)) {
    ++r;
  }
  r += countProduction(st, type);
  if (type == buildtypes::Zerg_Hatchery) {
    return r + countPlusProduction(st, buildtypes::Zerg_Lair);
  }
  if (type == buildtypes::Zerg_Lair) {
    return r + countPlusProduction(st, buildtypes::Zerg_Hive);
  }
  if (type == buildtypes::Zerg_Spire) {
    return r + countPlusProduction(st, buildtypes::Zerg_Greater_Spire);
  }
  return r;
}

int larvaCount(
    const autobuild::BuildState& st,
    const autobuild::BuildStateUnit& u) {
  return std::max(std::min((st.frame - u.larvaTimer) / 342, 3), 0);
}

BuildStateUnit& addUnit(BuildState& st, const BuildType* type) {
  auto& c = st.units[type];
  c.emplace_back();
  auto& r = c.back();
  r.type = type;
  if (type->isWorker) {
    ++st.workers;
  }
  if (type->isRefinery) {
    ++st.refineries;
  }
  st.usedSupply[type->race] += type->supplyRequired;
  st.maxSupply[type->race] += type->supplyProvided;
  return r;
}

void removeUnit(BuildState& st, BuildStateUnit& u) {
  st.usedSupply[u.type->race] -= u.type->supplyRequired;
  st.maxSupply[u.type->race] -= u.type->supplyProvided;
  if (u.type->isWorker) {
    --st.workers;
  }
  if (u.type->isRefinery) {
    --st.refineries;
  }
  auto& c = st.units[u.type];
  c.erase(c.begin() + (&u - c.data()));
}

template <typename list_T, typename T>
void emplaceProd(list_T& list, int frame, T t) {
  auto pred = [&](auto i) {
    if (i == list.begin()) {
      return true;
    }
    auto p = std::prev(i);
    return p->first <= frame;
  };
  for (auto i = list.end();; --i) {
    if (pred(i)) {
      list.emplace(i, frame, std::move(t));
      break;
    }
  }
}

BuildState getMyState(State* state) {
  BuildState st;
  st.frame = state->currentFrame();
  // auto res = state->resources();
  auto tcs = state->tcstate();
  auto res = tcs->frame->resources[tcs->player_id];
  st.minerals = (double)res.ore;
  st.gas = (double)res.gas;

  st.availableGases = 0;
  if (builderhelpers::findGeyserForRefinery(
          state, buildtypes::Zerg_Extractor, {})) {
    st.availableGases = 1;
  }

  std::unordered_map<const Unit*, int> larvaCount;
  for (const Unit* u :
       state->unitsInfo().myUnitsOfType(buildtypes::Zerg_Larva)) {
    if (u->associatedUnit) {
      ++larvaCount[u->associatedUnit];
    }
  }

  for (const Unit* u : state->unitsInfo().myUnits()) {
    const BuildType* t = u->type;
    if (t == buildtypes::Terran_Siege_Tank_Siege_Mode) {
      t = buildtypes::Terran_Siege_Tank_Tank_Mode;
    }
    if (u->upgrading() && u->upgradingType) {
      autobuild::emplaceProd(
          st.production,
          st.frame + u->remainingUpgradeResearchTime,
          u->upgradingType);
    }
    if (u->researching() && u->researchingType) {
      autobuild::emplaceProd(
          st.production,
          st.frame + u->remainingUpgradeResearchTime,
          u->researchingType);
    }
    if (t == buildtypes::Zerg_Larva) {
      continue;
    }
    if (t == buildtypes::Zerg_Egg || t == buildtypes::Zerg_Cocoon) {
      t = u->constructingType;
      if (!t) {
        continue;
      }
    }
    if (t == buildtypes::Zerg_Lurker_Egg) {
      t = buildtypes::Zerg_Lurker;
    }
    if (!u->completed() || u->morphing()) {
      autobuild::emplaceProd(
          st.production, st.frame + u->remainingBuildTrainTime, t);
      if (t->isTwoUnitsInOneEgg) {
        autobuild::emplaceProd(
            st.production, st.frame + u->remainingBuildTrainTime, t);
      }

      st.inprodSupply[t->race] += t->supplyProvided;
      continue;
    }
    auto& stU = addUnit(st, t);
    if (u->addon) {
      stU.addon = u->addon->type;
    }
    if (t == buildtypes::Terran_Nuclear_Silo && u->associatedCount) {
      stU.busyUntil = std::numeric_limits<int>::max();
    }
    if (t == buildtypes::Zerg_Hatchery || t == buildtypes::Zerg_Lair ||
        t == buildtypes::Zerg_Hive) {
      stU.busyUntil = st.frame + u->remainingUpgradeResearchTime;
      stU.larvaTimer = st.frame - 342 + u->remainingBuildTrainTime;
      auto i = larvaCount.find(u);
      if (i != larvaCount.end()) {
        stU.larvaTimer -= 342 * i->second;
      }
    } else {
      stU.busyUntil = st.frame +
          std::max(u->remainingBuildTrainTime, u->remainingUpgradeResearchTime);
    }
  }

  for (const BuildType* t : buildtypes::allUpgradeTypes) {
    if (state->getUpgradeLevel(t) >= t->level) {
      st.upgradesAndTech.insert(t);
    }
  }

  for (const BuildType* t : buildtypes::allTechTypes) {
    if (state->hasResearched(t)) {
      st.upgradesAndTech.insert(t);
    }
  }

  st.usedSupply.fill(res.used_psi / 2.0);
  st.maxSupply.fill(res.total_psi / 2.0);

  st.race = BWAPI::Races::Enum::Terran;
  int scvs = countUnits(st, buildtypes::Terran_SCV);
  int probes = countUnits(st, buildtypes::Protoss_Probe);
  int drones = countUnits(st, buildtypes::Zerg_Drone);
  int best = std::max(scvs, std::max(probes, drones));
  if (best == scvs) {
    st.race = BWAPI::Races::Enum::Terran;
  } else if (best == probes) {
    st.race = BWAPI::Races::Enum::Protoss;
  } else if (best == drones) {
    st.race = BWAPI::Races::Enum::Zerg;
  }

  return st;
}

const BuildType failedObject;
const BuildType* failed = &failedObject;
const BuildType timeoutObject;
const BuildType* timeout = &timeoutObject;
const BuildType builtdepObject;
const BuildType* builtdep = &builtdepObject;

// This function advances the state until build has been put into production,
// or endFrame is reached, whichever is first.
// Returns null on success, failed on failure, timeout if endFrame was reached,
// or some other BuildType* that needs to be built first.
const BuildType* advance(BuildState& st, BuildEntry thing, int endFrame) {
  const BuildType* const build = thing.type;
  if (st.frame >= endFrame) {
    if (FLAGS_verbose_build_logging) {
      LOG(INFO) << buildLogIndent() << "advance "
                << (build ? build->name : "null") << " -> instant timeout";
    }
    return timeout;
  }

  if (!build->builder) {
    throw std::runtime_error("autobuild::advance: build has no builder");
  }

  using namespace buildtypes;

  const BuildType* addonRequired = nullptr;
  bool prereqInProd = false;
  if (build) {
    for (const BuildType* prereq : build->prerequisites) {
      if (prereq == Zerg_Larva)
        continue;
      // If there is a required addon which has the same builder as this type,
      // then we assume
      // that the thing can only be built from a unit which has this addon.
      if (prereq->isAddon && prereq->builder == build->builder &&
          !build->builder->isAddon) {
        addonRequired = prereq;
      }
      if (!has(st, prereq)) {
        if (prereq == Zerg_Spire && hasOrInProduction(st, Zerg_Greater_Spire))
          continue;
        else if (prereq == Zerg_Hatchery && hasOrInProduction(st, Zerg_Hive))
          continue;
        else if (prereq == Zerg_Hatchery && hasOrInProduction(st, Zerg_Lair))
          continue;
        else if (prereq == Zerg_Lair && hasOrInProduction(st, Zerg_Hive))
          continue;
        if (isInProduction(st, prereq))
          prereqInProd = true;
        else {
          if (FLAGS_verbose_build_logging) {
            LOG(INFO) << buildLogIndent() << "advance "
                      << (build ? build->name : "null")
                      << " -> prereq: " << prereq->name;
          }
          return prereq;
        }
      }
    }
  }

  const BuildType* refinery = Terran_Refinery;
  if (build) {
    if (build->race == BWAPI::Races::Enum::Protoss) {
      refinery = Protoss_Assimilator;
    } else if (build->race == BWAPI::Races::Enum::Zerg) {
      refinery = Zerg_Extractor;
    }
  }

  const BuildType* supply = Terran_Supply_Depot;
  if (build) {
    if (build->race == BWAPI::Races::Enum::Protoss) {
      supply = Protoss_Pylon;
    } else if (build->race == BWAPI::Races::Enum::Zerg) {
      supply = Zerg_Overlord;
    }
  }

  while (true) {
    while (!st.production.empty() && st.frame >= st.production.front().first) {
      const BuildType* t = st.production.front().second;
      st.production.pop_front();
      if (t->isUnit()) {
        if (t->isAddon) {
          for (auto& v : st.units[t->builder]) {
            if (st.frame >= v.busyUntil && !v.addon) {
              v.addon = t;
              break;
            }
          }
        }
        st.inprodSupply[t->race] -= t->supplyProvided;
        st.usedSupply[t->race] -= t->supplyRequired;
        addUnit(st, t);
      } else {
        st.upgradesAndTech.insert(t);
      }
      if (prereqInProd) {
        prereqInProd = false;
        for (const BuildType* prereq : build->prerequisites) {
          if (prereq == Zerg_Larva)
            continue;
          if (!hasUnit(st, prereq)) {
            if (prereq == Zerg_Spire &&
                hasOrInProduction(st, Zerg_Greater_Spire)) {
              continue;
            } else if (
                prereq == Zerg_Hatchery && hasOrInProduction(st, Zerg_Hive)) {
              continue;
            } else if (
                prereq == Zerg_Hatchery && hasOrInProduction(st, Zerg_Lair)) {
              continue;
            } else if (
                prereq == Zerg_Lair && hasOrInProduction(st, Zerg_Hive)) {
              continue;
            }
            prereqInProd = true;
          }
        }
      }
    }

    if (build) {
      auto addBuilt = [&](const BuildType* t, bool subtractBuildTime) {
        emplaceProd(
            st.buildOrder,
            st.frame - (subtractBuildTime ? t->buildTime : 0),
            BuildEntry{t, {}});
        addUnit(st, t);
        st.minerals -= t->mineralCost;
        st.gas -= t->gasCost;
      };
      bool hasEnoughMinerals =
          build->mineralCost == 0 || st.minerals >= build->mineralCost;
      bool hasEnoughGas = build->gasCost == 0 || st.gas >= build->gasCost;

      if (st.autoBuildRefineries && st.availableGases && hasEnoughMinerals &&
          !hasEnoughGas) {
        if (st.minerals >= build->mineralCost + refinery->mineralCost) {
          addBuilt(refinery, true);
          ++st.refineries;
          --st.availableGases;
          if (FLAGS_verbose_build_logging) {
            LOG(INFO) << buildLogIndent() << "advance "
                      << (build ? build->name : "null")
                      << " -> prebuilt refinery (" << refinery->name << ") ("
                      << st.refineries << "/" << st.availableGases << ")";
          }
          return builtdep;
        }
      }

      bool hasSupply = true;
      if (build->isUnit()) {
        if (build->supplyRequired && build->builder != Zerg_Mutalisk &&
            build != Protoss_Archon && build != Protoss_Dark_Archon) {
          double nextSupply =
              st.usedSupply[build->race] + build->supplyRequired;
          if (nextSupply >= 200.0) {
            if (FLAGS_verbose_build_logging) {
              LOG(INFO) << buildLogIndent() << "advance "
                        << (build ? build->name : "null")
                        << " -> failed: maxed out";
            }
            return failed;
          }
          if (nextSupply > st.maxSupply[build->race]) {
            if ((st.maxSupply[build->race] > 10 || st.production.empty()) &&
                nextSupply > st.maxSupply[build->race] +
                        (st.maxSupply[build->race] < 16
                             ? st.inprodSupply[build->race]
                             : 0)) {
              if (st.maxSupply[build->race] < 16) {
                if (FLAGS_verbose_build_logging) {
                  LOG(INFO) << buildLogIndent() << "advance "
                            << (build ? build->name : "null") << " -> supply ("
                            << supply->name << ")";
                }
                return supply;
              } else {
                addBuilt(supply, true);
                if (FLAGS_verbose_build_logging) {
                  LOG(INFO) << buildLogIndent() << "advance "
                            << (build ? build->name : "null")
                            << " -> prebuilt supply (" << supply->name << ")";
                }
                return builtdep;
              }
            } else
              hasSupply = false;
          }
        }
      }
      if (hasEnoughMinerals && hasEnoughGas && hasSupply && !prereqInProd) {
        BuildStateUnit* builder = nullptr;
        BuildStateUnit* builder2 = nullptr;
        bool builderExists = false;
        if (build->builder == Zerg_Larva) {
          for (int i = 0; i < 3 && !builder; ++i) {
            for (auto& u :
                 st.units[i == 0 ? Zerg_Hive : i == 1 ? Zerg_Lair
                                                      : Zerg_Hatchery]) {
              if (!builderExists) {
                builderExists = true;
              }
              if (st.frame - u.larvaTimer >= 342) {
                builder = &u;
                break;
              }
            }
          }
          if (!builder && st.autoBuildHatcheries &&
              st.minerals >= 300 + 212 * countProduction(st, Zerg_Hatchery)) {
            bool prebuild = st.workers >= 26;
            addBuilt(Zerg_Hatchery, prebuild);
            if (FLAGS_verbose_build_logging) {
              LOG(INFO) << buildLogIndent() << "advance "
                        << (build ? build->name : "null") << " -> "
                        << (prebuild ? "prebuilt hatchery" : "build hatchery");
            }
            return builtdep;
          }
        } else {
          for (auto& u : st.units[build->builder]) {
            if (build->isAddon && u.addon) {
              continue;
            }
            if (addonRequired && u.addon != addonRequired) {
              continue;
            }
            if (!builderExists) {
              builderExists = true;
            }
            if (st.frame >= u.busyUntil) {
              builder = &u;
              break;
            }
          }
          if (builder &&
              (build == Protoss_Archon || build == Protoss_Dark_Archon)) {
            for (auto& u : st.units[build->builder]) {
              if (&u == builder) {
                continue;
              }
              builder2 = &u;
              break;
            }
            if (!builder2) {
              builder = nullptr;
              builderExists = false;
            }
          }
        }
        if (!builder && !builderExists) {
          if (build->builder == Zerg_Larva) {
            if (!hasOrInProduction(st, Zerg_Hatchery) &&
                !hasOrInProduction(st, Zerg_Lair) &&
                !hasOrInProduction(st, Zerg_Hive)) {
              if (FLAGS_verbose_build_logging) {
                LOG(INFO) << buildLogIndent() << "advance "
                          << (build ? build->name : "null")
                          << " -> builder hatchery";
              }
              return Zerg_Hatchery;
            }
          } else {
            if (!builderExists && !isInProduction(st, build->builder)) {
              const BuildType* bt =
                  addonRequired ? addonRequired : build->builder;
              if (FLAGS_verbose_build_logging) {
                LOG(INFO) << buildLogIndent() << "advance "
                          << (build ? build->name : "null") << " -> builder ("
                          << bt->name << ")";
              }
              return bt;
            }
          }
        }
        if (builder) {
          if (build->builder == Zerg_Larva) {
            if (st.frame - builder->larvaTimer >= 342 * 3) {
              builder->larvaTimer = st.frame - 342 * 2;
            } else {
              builder->larvaTimer += 342;
            }
          } else {
            builder->busyUntil = st.frame + build->buildTime;
          }
          if (build == Terran_Nuclear_Missile) {
            builder->busyUntil = std::numeric_limits<int>::max();
          }
          if (build->isResourceDepot && thing.pos != Position()) {
            st.isExpanding = true;
          }

          st.inprodSupply[build->race] += build->supplyProvided;
          st.usedSupply[build->race] += build->supplyRequired;
          emplaceProd(st.production, st.frame + build->buildTime, build);
          emplaceProd(st.buildOrder, st.frame, std::move(thing));
          if (build->isTwoUnitsInOneEgg) {
            st.inprodSupply[build->race] += build->supplyProvided;
            st.usedSupply[build->race] += build->supplyRequired;
            emplaceProd(st.production, st.frame + build->buildTime, build);
          }
          st.minerals -= build->mineralCost;
          st.gas -= build->gasCost;
          if (build->isAddon) {
            builder->addon = build;
          }
          if (builder->type->race == BWAPI::Races::Enum::Zerg &&
              build->builder != Zerg_Larva) {
            if (build->isUnit()) {
              removeUnit(st, *builder);
            }
          } else if (build == Protoss_Archon || build == Protoss_Dark_Archon) {
            removeUnit(st, *builder);
            removeUnit(st, *builder2);
          }
          if (FLAGS_verbose_build_logging) {
            LOG(INFO) << buildLogIndent() << "advance "
                      << (build ? build->name : "null") << " -> success";
          }
          return nullptr;
        }
      }
    }

    int f = std::min(15, endFrame - st.frame);

    // This is a super rough estimate.  TODO something more accurate?
    int gasWorkers = std::min(3 * st.refineries, st.workers / 4);
    int mineralWorkers = st.workers - gasWorkers;
    const double gasPerFramePerWorker = 0.2 / 3;
    const double mineralsPerFramePerWorker = 0.05;
    double mineralIncome = mineralsPerFramePerWorker * mineralWorkers;
    double gasIncome = gasPerFramePerWorker * gasWorkers;

    st.minerals += mineralIncome * f;
    st.gas += gasIncome * f;
    st.frame += f;

    if (st.frame >= endFrame) {
      if (FLAGS_verbose_build_logging) {
        LOG(INFO) << buildLogIndent() << "advance "
                  << (build ? build->name : "null") << " -> timeout";
      }
      return timeout;
    }
  }
}

bool depbuild(BuildState& st, const BuildState& prevSt, BuildEntry thing) {
  auto* type = thing.type;
  auto* initialType = type;
  if (FLAGS_verbose_build_logging) {
    LOG(INFO) << buildLogIndent() << "depbuild " << initialType->name;
  }
  int endFrame = st.frame + 15 * 60 * 10;
  while (true) {
    if (FLAGS_verbose_build_logging) {
      ++buildLogDepth;
    }
    auto* builtType = thing.type;
    type = advance(st, thing, endFrame);
    if (FLAGS_verbose_build_logging) {
      --buildLogDepth;
    }
    if (!type) {
      if (FLAGS_verbose_build_logging) {
        LOG(INFO) << buildLogIndent() << "depbuild " << initialType->name
                  << ": successfully built " << builtType->name;
      }
      return true;
    }
    if (type == builtdep) {
      if (FLAGS_verbose_build_logging) {
        LOG(INFO) << buildLogIndent() << "depbuild " << initialType->name
                  << ": successfully built some dependency";
      }
      return true;
    }
    if (st.frame != prevSt.frame)
      st = prevSt;
    if (type == failed) {
      if (FLAGS_verbose_build_logging)
        LOG(INFO) << buildLogIndent() << "depbuild " << initialType->name
                  << ": failed";
      return false;
    }
    if (type == timeout) {
      if (FLAGS_verbose_build_logging) {
        LOG(INFO) << buildLogIndent() << "depbuild " << initialType->name
                  << ": timed out";
      }
      return false;
    }
    if (type == builtType || (type->builder && type->builder->builder == type &&
                              !hasOrInProduction(st, type->builder)) ||
        (type == buildtypes::Zerg_Drone &&
         !hasOrInProduction(st, buildtypes::Zerg_Hatchery) &&
         !hasOrInProduction(st, buildtypes::Zerg_Lair) &&
         !hasOrInProduction(st, buildtypes::Zerg_Hive))) {
      if (FLAGS_verbose_build_logging) {
        LOG(INFO) << buildLogIndent() << "depbuild " << initialType->name
                  << ": failing because of unsatisfiable cyclic dependency";
      }
      return false;
    }
    thing = {type, {}};
  }
}

// st is the initial state, thingSt is the state after building thing.
template <typename F>
bool nodelayStage2(
    BuildState& st,
    BuildState thingSt,
    BuildEntry thing,
    F&& otherThing) {
  if (FLAGS_verbose_build_logging) {
    LOG(INFO) << buildLogIndent() << "nodelayStage2 " << thing.type->name;
    ++buildLogDepth;
  }
  // Then try to build the other thing..
  if (otherThing(st)) {
    if (FLAGS_verbose_build_logging) {
      --buildLogDepth;
    }
    // If the other thing took too long, just do the thing
    if (st.frame >= thingSt.frame) {
      if (FLAGS_verbose_build_logging)
        LOG(INFO) << buildLogIndent() << "nodelayStage2 " << thing.type->name
                  << ": too late; choose thing";
      st = std::move(thingSt);
      return true;
    }
    auto otherThingSt = st;
    if (FLAGS_verbose_build_logging) {
      ++buildLogDepth;
    }
    // Then do the thing again!
    if (depbuild(st, otherThingSt, thing)) {
      if (FLAGS_verbose_build_logging) {
        --buildLogDepth;
      }
      // If we could do the other thing then the thing at least
      // as fast as just doing the thing, then do all the things!
      if (st.frame <= thingSt.frame) {
        if (FLAGS_verbose_build_logging) {
          LOG(INFO) << buildLogIndent() << "nodelayStage2 " << thing.type->name
                    << ": no delay; choose otherThing";
        }
        st = std::move(otherThingSt);
        return true;
      } else {
        if (FLAGS_verbose_build_logging) {
          LOG(INFO) << buildLogIndent() << "nodelayStage2 " << thing.type->name
                    << ": would delay; choose thing";
        }
        // otherwise, just do the thing
        st = std::move(thingSt);
        return true;
      }
    } else {
      if (FLAGS_verbose_build_logging) {
        --buildLogDepth;
        LOG(INFO) << buildLogIndent() << "nodelayStage2 " << thing.type->name
                  << ": depbuild failed";
      }
      // Should only fail if it times out, so let's just return the thing since
      // it is higher priority
      st = std::move(thingSt);
      return true;
    }
  } else {
    if (FLAGS_verbose_build_logging) {
      --buildLogDepth;
      LOG(INFO) << buildLogIndent() << "nodelayStage2 " << thing.type->name
                << ": otherThing failed";
    }
    // If otherThing failed, we just do thing
    st = std::move(thingSt);
    return true;
  }
}

// This function tries to build thing, but if it can it will squeeze in
// a call to otherThing first. It evaluates whether it can do the call to
// otherThing without delaying the construction of thing.
template <typename F>
bool nodelay(BuildState& st, BuildEntry thing, F&& otherThing) {
  if (FLAGS_verbose_build_logging) {
    LOG(INFO) << buildLogIndent() << "nodelay " << thing.type->name;
    ++buildLogDepth;
  }
  auto prevSt = st;
  // First try to build the thing
  if (depbuild(st, prevSt, thing)) {
    if (FLAGS_verbose_build_logging) {
      --buildLogDepth;
    }
    // Copy the state, restore the original state in preparation
    // of doing the other thing
    auto thingSt = std::move(st);
    st = std::move(prevSt);
    return nodelayStage2(
        st, std::move(thingSt), thing, std::forward<F>(otherThing));
  } else {
    if (FLAGS_verbose_build_logging) {
      --buildLogDepth;
      LOG(INFO) << buildLogIndent() << "nodelay " << thing.type->name
                << ": depbuild failed";
    }
    // If it failed, then just do func
    st = std::move(prevSt);
    return std::forward<F>(otherThing)(st);
  }
}

} // namespace autobuild

void AutoBuildTask::build(
    const BuildType* type,
    Position pos,
    std::function<void()> builtCallback) {
  if (!type->isUnit() && autobuild::hasOrInProduction(*currentBuildSt, type)) {
    return;
  }
  queue = [
    queue = std::move(queue),
    type,
    pos,
    builtCallback = std::move(builtCallback)
  ](autobuild::BuildState & st) {
    return autobuild::nodelay(st, {type, pos, builtCallback}, queue);
  };
}

void AutoBuildTask::build(
    const BuildType* type,
    std::function<void()> builtCallback) {
  if (!type->isUnit() && autobuild::hasOrInProduction(*currentBuildSt, type)) {
    return;
  }
  queue = [
    queue = std::move(queue),
    type,
    builtCallback = std::move(builtCallback)
  ](autobuild::BuildState & st) {
    return autobuild::nodelay(st, {type, Position(), builtCallback}, queue);
  };
}

void AutoBuildTask::build(const BuildType* type, Position pos) {
  if (!type->isUnit() && autobuild::hasOrInProduction(*currentBuildSt, type)) {
    return;
  }
  queue = [ queue = std::move(queue), type, pos ](autobuild::BuildState & st) {
    return autobuild::nodelay(st, {type, pos}, queue);
  };
}

void AutoBuildTask::build(const BuildType* type) {
  if (!type->isUnit() && autobuild::hasOrInProduction(*currentBuildSt, type)) {
    return;
  }
  queue = [ queue = std::move(queue), type ](autobuild::BuildState & st) {
    return autobuild::nodelay(st, {type}, queue);
  };
}

bool AutoBuildTask::buildN(const BuildType* type, int n) {
  if (autobuild::countPlusProduction(*currentBuildSt, type) >= n) {
    return true;
  }
  build(type);
  return false;
}

bool AutoBuildTask::upgrade(const BuildType* type) {
  return buildN(type, 1);
}

void AutoBuildTask::update(State* state) {
  auto board = state->board();
  for (size_t i = 0; i < targetUpcIds_.size(); i++) {
    auto& target = targets_[i];
    if (target == nullptr) {
      target = board->taskForId(targetUpcIds_[i]);
      if (target) {
        VLOG(2) << "Found target task for "
                << utils::upcString(targetUpcIds_[i]);
      }
    }
  }
  auto newUnit = [&](Unit* u) {
    for (size_t i = 0; i < targetUpcIds_.size(); i++) {
      auto upcId = targetUpcIds_[i];
      auto it = scheduledUpcs.find(upcId);
      const BuildType* utype = u->type;
      if (utype == buildtypes::Zerg_Egg) {
        utype = u->constructingType;
      }
      if (it->second.type == utype) {
        if (it->second.builtCallback) {
          it->second.builtCallback();
        }
        if (targets_[i]) {
          // Cancel just in case this is a mismatch, this prevents a serious
          // bug where we build two buildings instead of one.
          // Usually this cancel won't do anything since construction has
          // started.
          targets_[i]->cancel(state);
        }
        targetUpcIds_.erase(targetUpcIds_.begin() + i);
        targets_.erase(targets_.begin() + i);
        scheduledUpcs.erase(it);
        break;
      }
    }
  };

  for (Unit* u : state->unitsInfo().getNewUnits()) {
    if (u->isMine) {
      newUnit(u);
    }
  }
  for (Unit* u : state->unitsInfo().getStartedMorphingUnits()) {
    if (u->isMine) {
      newUnit(u);
    }
  }

  // In contrast to the base class, don't determine status based on proxied
  // tasks.
  // TODO: Detect overall success/failure?
}

void AutoBuildTask::evaluate(State* state, Module* module) {
  autobuild::BuildState st = autobuild::getMyState(state);

  auto& mySt = st;
  VLOG(1) << "Frame " << mySt.frame << " minerals: " << mySt.minerals << " gas "
          << mySt.gas << " supply: " << mySt.usedSupply[0] << "/"
          << mySt.maxSupply[0];

  int endFrame = st.frame + 15 * 60 * 4;

  currentBuildSt = &st;

  preBuild(state, module);
  while (st.frame < endFrame) {
    queue = [](autobuild::BuildState&) { return false; };
    buildStep(st);
    if (!queue(st)) {
      break;
    }
//  std::string str;
//  LOG(INFO) << "frame " << st.frame << " minerals " << st.minerals << " gas "
//            << st.gas << " supply " << st.usedSupply[0] << "/"
//            << st.maxSupply[0];
//  str = "at " + std::to_string(st.frame) + " - ";
//  for (auto& v : st.units) {
//    if (v.second.empty()) {
//      continue;
//    }
//    str += " " + std::to_string(v.second.size()) + "x" + v.first->name;
//  }
//  LOG(INFO) << str;
//  str = "inprod - ";
//  for (auto& v : st.production) {
//    str += " " + v.second->name;
//  }
//  LOG(INFO) << str;
  }
  postBuild(state);

  currentBuildSt = nullptr;

  if (FLAGS_log_build_order) {
    std::string str;
    for (auto& v : st.buildOrder) {
      if (!str.empty())
        str += ", ";
      str += std::to_string(v.first) + " ";
      str += v.second.type->name;
    }
    LOG(INFO) << "Build order: " << str;
  }

  auto* board = state->board();
  for (auto& v : board->upcsFrom(module)) {
    for (size_t i = 0; i != targetUpcIds_.size(); ++i) {
      if (targetUpcIds_[i] == v.first) {
        targetUpcIds_.erase(targetUpcIds_.begin() + i);
        targets_.erase(targets_.begin() + i);
        board->consumeUPC(v.first, module);
        break;
      }
    }
  }

  for (size_t i = 0; i != targetUpcIds_.size();) {
    if (!targets_[i] || targets_[i]->finished()) {
      auto upcId = targetUpcIds_[i];
      targetUpcIds_.erase(targetUpcIds_.begin() + i);
      targets_.erase(targets_.begin() + i);
      auto it = scheduledUpcs.find(upcId);
      if (it != scheduledUpcs.end()) {
        scheduledUpcs.erase(it);
      }
    } else {
      ++i;
    }
  }

  size_t i = 0;
  bool isInOrder = true;

  int frame = state->currentFrame();

  std::vector<UpcId> newUpcs;

  for (auto& v : st.buildOrder) {
    if (v.first >= frame + 15 * 30) {
      break;
    }
    autobuild::BuildEntry thing = v.second;
    const BuildType* type = v.second.type;
    Position pos = v.second.pos;

    if (isInOrder) {
      if (i == targetUpcIds_.size()) {
        isInOrder = false;
      } else {
        auto upcId = targetUpcIds_[i];
        auto it = scheduledUpcs.find(upcId);
        if (it->second == thing) {
          ++i;
          continue;
        } else {
          isInOrder = false;
        }
      }
    }

    UPCTuple upc;
    upc.scale = 1;
    if (pos != Position()) {
      upc.positionS = pos;
    }
//    upc.position = UPCTuple::zeroPosition(state).fill_(
//        1.0f / (state->mapWidth() * state->mapHeight()));
    upc.command[Command::Create] = 1;
    upc.createType[type] = 1.0f;

    // Post a new UPC and proxy it
    auto id = board->postUPC(
        std::make_shared<UPCTuple>(std::move(upc)), upcId(), module);
    if (id != kFilteredUpcId) {
      scheduledUpcs[id] = std::move(thing);
      newUpcs.push_back(id);

      VLOG(2) << "Build " << type->name;
    }
  }

  while (i != targetUpcIds_.size()) {
    if (targets_[i]) {
      targets_[i]->cancel(state);
    }
    auto upcId = targetUpcIds_[i];
    targetUpcIds_.erase(targetUpcIds_.begin() + i);
    targets_.erase(targets_.begin() + i);
    auto it = scheduledUpcs.find(upcId);
    if (it != scheduledUpcs.end()) {
      scheduledUpcs.erase(it);
    }
  }

  for (auto id : newUpcs) {
    targetUpcIds_.push_back(id);
    targets_.push_back(nullptr);
  }
}

std::shared_ptr<AutoBuildTask> AutoBuildModule::createTask(int srcUpcId) {
  // TODO: How to determine build targets from UPC in the future?
  std::vector<DefaultAutoBuildTask::Target> targets;
  targets.emplace_back(buildtypes::Zerg_Hydralisk);
  targets.emplace_back(buildtypes::Zerg_Drone, 60);
  targets.emplace_back(buildtypes::Zerg_Hydralisk, 20);
  targets.emplace_back(buildtypes::Zerg_Drone, 20);

  return std::make_shared<DefaultAutoBuildTask>(srcUpcId, std::move(targets));
}

void AutoBuildModule::checkForNewUPCs(State* state) {
  auto board = state->board();
  auto tasks = board->tasksOfModule(this);
  if (tasks.empty()) {
    int srcUpcId = -1;
    for (auto& upcs : state->board()->upcsWithSharpCommand(Command::Create)) {
      if (upcs.second->createType.empty()) {
        srcUpcId = upcs.first;
        break;
      }
    }
    if (srcUpcId < 0) {
      return;
    }

    auto task = createTask(srcUpcId);
    board->postTask(task, this);
    tasks.push_back(task);
  }
}

void AutoBuildModule::step(State* state) {
  checkForNewUPCs(state);

  // Update tasks
  int frame = state->currentFrame();
  auto tasks = state->board()->tasksOfModule(this);
  for (auto& task : tasks) {
    auto abtask = std::static_pointer_cast<AutoBuildTask>(task);

    // TODO: Check task status

    if (abtask->lastEvaluate == 0 || frame - abtask->lastEvaluate >= 15) {
      // Time to replan!
      abtask->evaluate(state, this);
      abtask->lastEvaluate = frame;
    }
  }
}

} // namespace fairrsh
