/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#include "attackheuristic.h"

#include "state.h"
#include "utils.h"

#include <glog/logging.h>

namespace fairrsh {

RTTR_REGISTRATION {
  rttr::registration::class_<AttackHeuristicModule>("AttackHeuristicModule")(
      metadata("type", rttr::type::get<AttackHeuristicModule>()))
      .constructor();
}

void AttackHeuristicModule::step(State* state) {
  auto board = state->board();

  // Simple "attack closest" heuristic
  std::set<Unit const*> assigned;
  for (auto const& upct : board->upcsWithSharpCommand(Command::Delete)) {
    auto upcId = upct.first;
    auto& upc = upct.second;

    // Do we know where to attack?
    auto pos = utils::argmax(upc->position, upc->scale);
    bool hasPos = std::get<2>(pos) == 1;

    bool foundUnit = false;
    for (auto& uprob : upc->unit) {
      if (uprob.second == 0) {
        continue;
      }
      if (assigned.find(uprob.first) != assigned.end()) {
        continue;
      }

      Unit* unit = uprob.first;
      auto newUPC = std::make_shared<UPCTuple>();
      newUPC->unit[unit] = 1;
      newUPC->command[Command::Delete] = 1;

      if (!hasPos) {
        // Find closest enemy unit
        const Unit* target = utils::getBestScoreCopy(
            state->unitsInfo().visibleEnemyUnits(),
            [&](const Unit* target) { return utils::distance(unit, target); });
        if (!target) {
          continue;
        }

        newUPC->positionS = target;
      } else {
        newUPC->position = upc->position;
      }
      board->postUPC(std::move(newUPC), upcId, this);

      assigned.insert(unit);
      foundUnit = true;
    }

    if (foundUnit) {
      board->consumeUPC(upcId, this);
    }
  }
}

} // namespace fairrsh
