/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#include "module.h"

#include "state.h"
#include "utils.h"

#include <algorithm>
#include <cctype>

namespace fairrsh {

RTTR_REGISTRATION {
  rttr::registration::class_<Module>("Module")(
      metadata("type", rttr::type::get<Module>()))
      .property("name", &Module::name, &Module::setName);
}

Module::Module() {}

std::shared_ptr<Module> Module::make(std::string const& typeName) {
  auto lowerName = utils::stringToLower(typeName);

  for (auto typ : rttr::type::get<Module>().get_derived_classes()) {
    auto ctors = typ.get_constructors();
    if (ctors.empty()) {
      // This can be empty for base classes or special modules that don't
      // declare a constructor in their RTTR_REGISTRATION.
      continue;
    }

    auto lowerType = utils::stringToLower(typ.get_name());
    if (lowerType == lowerName || lowerType == lowerName + "module") {
      auto var = ctors.begin()->invoke();
      if (!var.is_valid()) {
        LOG(WARNING) << "Failed to construct module: " << typeName;
        return nullptr;
      }

      auto m = var.get_value<std::shared_ptr<Module>>();
      if (m->name().empty()) {
        m->setName(makeName(typ));
      }
      return m;
    }
  }

  LOG(WARNING) << "No such module: " << typeName;
  return nullptr;
}

void Module::setName(std::string name) {
  name_ = std::move(name);
}

std::string Module::name() {
  return name_;
}

void Module::step(State* state) {
  auto board = state->board();

  auto it = commands_.begin();
  while (it != commands_.end()) {
    if (it->second <= 0) {
      commands_.erase(it++);
    } else {
      board->postCommand(it->first);
      --it->second;
      ++it;
    }
  }
}

void Module::postCommandForSteps(tc::Client::Command cmd, int steps) {
  commands_.emplace_back(std::move(cmd), steps);
}

std::string Module::makeName(rttr::type const& type) {
  // TODO: Naming scheme with automatic numbering, for example?
  return type.get_name().to_string();
}

} // namespace fairrsh
