/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#pragma once

#include "fairrsh.h"
#include "model.h"
#include "utils.h"

#include <cereal/types/polymorphic.hpp>
#include <cereal/types/memory.hpp>
#include <cereal/archives/json.hpp>


#include <random>

DECLARE_bool(max_exploit);

namespace fairrsh {

namespace model {

using OpeningBanditPtr = std::shared_ptr<OpeningBandit>;

/**
 * An Opening Bandit that decides which opening to follow
 */
class OpeningBandit {
  RTTR_ENABLE()
  friend class cereal::access;


  public:
    OpeningBandit();
    virtual ~OpeningBandit();

    static OpeningBanditPtr make(
        tc::BW::Race enemyRace,
        const std::string& mapName,
        const std::string& enemyName);
    // load will load by default from bwapi-data/read/openings_$OPPONENT
    static OpeningBanditPtr load(std::string enemyName = std::string());
    // save will write by default to bwapi-data/write/openings_$OPPONENT
    static void save(
        OpeningBanditPtr bandit,
        std::string enemyName = std::string());

    virtual std::string getOpening(
        const std::vector<std::string>& canConsiderOpenings,
        tc::BW::Race enemyRace,
        const std::string& mapName,
        const std::string& enemyName);

    virtual void onGameEnd(State* s);

    template <class Archive>
    void serialize(Archive& ar, const std::uint32_t version) {
      // TODO change for Cereal's preferred syntax
      // TODO refactor, e.g. http://uscilab.github.io/cereal/pimpl.html
      LOG(INFO) << "serializing OpeningBandit";
      ar& loadedEnemyName;
      ar& loadedEnemyRace;
      ar& openingToId_; // for convenience when editing
      ar& probas_;
      ar& wonGames_;
      ar& totalGames_;
    }

   protected:
    size_t decidedOpening_;
    std::string loadedEnemyName;
    std::string loadedEnemyRace;
    std::unordered_map<std::string, size_t> openingToId_;
    std::vector<float> probas_; // can be used for init probas!
    std::vector<float> wonGames_;
    std::vector<float> totalGames_;
};

} // namespace model

} // namespace fairrsh

// XXX Use rttr for all this stuff, including serialization
// CEREAL_REGISTER_TYPE(fairrsh::OpeningBandit)
