/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#ifdef WITHOUT_POSIX
#include <windows.h>
#include <tchar.h>
// ERROR macro is defined in windows headers, hack for glog:
#define GLOG_NO_ABBREVIATED_SEVERITIES
#endif

#include "bandit.h"
#include "../buildorders/base.h"
#include <fstream>
#include <glog/logging.h>

#ifdef WITHOUT_POSIX
// TODO
// https://msdn.microsoft.com/en-us/library/t49t9ds1.aspx
// https://msdn.microsoft.com/en-us/library/ms235631.aspx
// TODO: the current implementation assumes the Unicode option
// to be disabled
std::vector<std::string> findFiles(std::string const& path) {
  std::vector<std::string> files;
  WIN32_FIND_DATAA fileFindData;
  HANDLE hFind = FindFirstFileA((path + "/*").c_str(), &fileFindData);
  if (hFind == INVALID_HANDLE_VALUE) {
	  return files;
  }
  do {
	std::string fnameStr(fileFindData.cFileName);
	files.push_back(path + "/" + fnameStr);
  } while (FindNextFileA(hFind, &fileFindData) != 0);
  FindClose(hFind);
  return files;
}
#else
#include <dirent.h>
std::vector<std::string> findFiles(std::string const& path) {
  std::vector<std::string> files;
  DIR* dir;
  struct dirent* ent;
  if ((dir = opendir(path.c_str())) != nullptr) {
    while ((ent = readdir(dir)) != nullptr) {
      if (ent->d_type == DT_REG) {
        files.emplace_back(path + "/" + ent->d_name);
      }
    }
    closedir(dir);
  } else {
    throw std::runtime_error("Error opening directory");
  }
  return files;
}
#endif

std::string findNameIn(const std::string& ename, const std::string& basepath) {
  std::string fname = "";
  float matchRatio = 0.f;
  for (auto file : findFiles(basepath)) {
    auto f = file.find("openings_");
    if (f != std::string::npos && file.find(ename, f) != std::string::npos) {
      // heuristic in case there are several matches...
      float ratio = ename.length () * 1.f / (file.substr(f).length() + 1.e-6f);
      if (ratio > matchRatio) {
        matchRatio = ratio;
        fname = file;
      }
    }
  }
  return fname;
}

// DEFINE_double(bandit_prior_weight, 0.1, "importance to give to the priors")
DEFINE_bool(
    max_exploit,
    false,
    "exploit maximally a 100% WR opening instead of using vanilla UCB1");
// ^ TODO weight in initial probas_

namespace fairrsh {

namespace model {

RTTR_REGISTRATION {
  rttr::registration::class_<OpeningBandit>("OpeningBandit")(
      metadata("type", rttr::type::get<OpeningBandit>()))
      .constructor();
}

/**
 * Either bwapi-data/read/openings_${enemyName}.json exists and we use it
 * or we kick it up to make() by throwing a runtime_error
 */
OpeningBanditPtr OpeningBandit::load(std::string enemyName) {
  auto vs = utils::stringSplit(enemyName, '/');
  auto ename = vs[vs.size() - 1];
#ifdef WITHOUT_POSIX
  std::string path = findNameIn(ename, "bwapi-data\\read");
#else
  std::string path = findNameIn(ename, "bwapi-data/read");
#endif
  if (path == "") {
#ifdef WITHOUT_POSIX
    path = findNameIn(ename, "bwapi-data\\AI");
#else
    path = findNameIn(ename, "bwapi-data/AI");
#endif
  }
  std::ifstream ifs(path);
  if (!ifs.good() || path == "") { // defensive ||
    throw std::runtime_error("Cannot read from " + path);
  }
  cereal::JSONInputArchive ia(ifs);
  OpeningBanditPtr bandit;
  ia(bandit);
  // TODO some checks
  return bandit;
}

/**
 * We save at bwapi-data/write/openings_${bandit->loadedEnemyName}.json
 */
void OpeningBandit::save(OpeningBanditPtr bandit, std::string enemyName) {
  // TODO check on windows with '\' !
  std::string path =
      "bwapi-data/write/openings_" + bandit->loadedEnemyName + ".json";
  std::ofstream ofs(path);
  if (!ofs.good()) {
    LOG(ERROR) << "Cannot write to " << path;
    return;
  } else {
    LOG(INFO) << "Got |" << enemyName << "| saving bandit to " << path;
  }
  cereal::JSONOutputArchive oa(ofs);
  oa(bandit);
}

/**
 * We create a new bandit for this enemyName.split('/')[-1],
 * being it by loading existing probas_ from openings_RACE_*
 * or just calling OpeningBandit().
 */
OpeningBanditPtr OpeningBandit::make(
    tc::BW::Race enemyRace,
    const std::string& mapName,
    const std::string& enemyName) {
  // TODO use mapName! and maybe enemyName for rule based init?
  // It helps to have a openings_RACE_{Random|Zerg|Terran|Protoss} file with
  // initial probas_ to play against unknown bots. I suggest we weight higher
  // the openings with consistent win-rate accross the board in a given match-up
  std::string race = enemyRace._to_string();
  auto vs = utils::stringSplit(enemyName, '/');
  std::string ename = vs[vs.size() - 1];
  OpeningBanditPtr bandit;
  try {
    bandit = load("RACE_" + race);
    // not toLower(), no clash possible with bot names should be possible...
  } catch (std::exception& ex) {
    LOG(ERROR) << "No openings bandit found for race: " << race
               << ", initializing a new one for player: " << ename;
    bandit = std::make_shared<OpeningBandit>();
  }

  bandit->loadedEnemyName = ename;

  if (!bandit) {
    LOG(DFATAL) << "Failed to instantiate or load OpeningBandit!";
    return nullptr;
  }
  return bandit;
}

OpeningBandit::OpeningBandit() {
  size_t id = 0;
  for (auto typ : rttr::type::get<ABBOBase>().get_derived_classes()) {
    auto lowerType = utils::stringToLower(typ.get_name());
    openingToId_[lowerType.substr(4)] = id++; // removing "abbo"
    probas_.push_back(1.);
    wonGames_.push_back(0);
    totalGames_.push_back(1.e-6);
  }
}

OpeningBandit::~OpeningBandit() {}

/***
 * This draws UCB1 style, play the action j with maximum:
 * (win_j / total_j) + sqrt(2 * log(sum(total)) / total_j)
 * The assumption is that it is called once per game,
 * or at least acted upon based on the last call!
 * TODO a rule that makes us not-really-UCB as we keep exploiting as long as
 * we're undefeated with a given opening!
 */
std::string OpeningBandit::getOpening(
    const std::vector<std::string>& canConsiderOpenings,
    tc::BW::Race enemyRace,
    const std::string& mapName,
    const std::string& enemyName) {

  LOG(INFO) << "We are considering the following openings:";
  for (auto s : canConsiderOpenings)
    LOG(INFO) << s;
  LOG(INFO) << "Out of the following openings:";
  for (const auto& opi : openingToId_)
    LOG(INFO) << opi.first;

  std::string eRace = enemyRace._to_string();
  if (eRace != loadedEnemyRace) {
    // probably we play against random and we scouted them now
    // we could reload an openings_${name}_${race} here if we wanted,
    // see messenger discussion: too few data IMHO!
    LOG(INFO) << "called getOpening() with different race, enemyRace: |"
              << eRace << "|, loadedEnemyRace: |" << loadedEnemyRace << "|";
    // TODO use it?
    loadedEnemyRace = eRace;
    // TODO Test our handling of random properly
  }

  if (canConsiderOpenings.size() == 0) {
    LOG(DFATAL) << "no openings to choose from!";
    return "5pool"; // TODO CHANGE THAT
    // ^ absolute default if all hell breaks lose. Should never happen.
    // Here for when we replace FATAL by ERROR for Release/Tournament...
  }
  std::map<size_t, std::string> idToOpening;
  std::vector<size_t> mask;
  mask.resize(canConsiderOpenings.size());
  size_t i = 0;
  for (const auto& op_s : canConsiderOpenings) {
    mask[i++] = openingToId_[op_s];
    idToOpening[openingToId_[op_s]] = op_s;
  }
  decidedOpening_ = mask[0];
  float N = 0;
  for (const auto& i : mask)
    N += totalGames_[i];

  if (N < 1) {
    float max_ = -1.;
    for (const auto& i : mask) {
      if (probas_[i] > max_) {
        decidedOpening_ = i;
        max_ = probas_[i];
      }
    }
  } else {
    // TODO weight in probas_
    float max_ = -1.;
    float score = 0;
    for (const auto& i : mask) {
      if (FLAGS_max_exploit) {
        if (probas_[i] < 0.001) { // we do not want to do this build, ever!
          score = -1;
        } else {
          if (wonGames_[i] / totalGames_[i] > 0.999) { // exploit!
            score = std::numeric_limits<float>::infinity();
          } else if (totalGames_[i] < 1.) { // explore in order of ranking
            score = 10000.0*probas_[i];
          } else { // hedge, UCB1
            score = wonGames_[i] / (totalGames_[i] + 1e-6f) + // defensive
                std::sqrt(2 * std::log(N) / totalGames_[i]);
          }
        }
      } else { // vanilla UCB1
        if (probas_[i] < 0.001) { // we do not want to do this build, ever!
          score = -1;
        } else {
          if (totalGames_[i] < 1.) // explore in order of ranking
            score = 10000.0*probas_[i];
          else // UCB1
            score = wonGames_[i] / totalGames_[i] +
                std::sqrt(2 * std::log(N) / totalGames_[i]);
        }
      }
      if (score > max_) {
        decidedOpening_ = i;
        max_ = score;
      }
    }
  }
  return idToOpening[decidedOpening_];
}

void OpeningBandit::onGameEnd(State* s) {
  totalGames_[decidedOpening_] += 1;
  if (s->won())
    wonGames_[decidedOpening_] += 1;
}

} // namespace model

} // namespace fairrsh
