/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#include "fairrsh.h"
#include "modules.h"
#include "player.h"
#include "utils.h"

#include <gflags/gflags.h>
#include <glog/logging.h>
#include <torchcraft/client.h>

#include <string>
#include <vector>

using namespace fairrsh;

namespace {

// Top and bottom modules that are always present
char constexpr kAutoTopModule[] = "Top";
char constexpr kAutoBottomModule[] = "UPCToCommand";

// The default set of modules
char constexpr kDefaultModules[] =
    "CreateGatherAttack,"
    "Strategy,"
    "GenericAutoBuild,"
    "Builder,"
    "MeanLingRush,"
    "Tactics,"
    "SquadCombat,"
    "Scouting,"
    "Gatherer,"
    "WorkerDefence,"
    "Harass,"
    "StaticDefenceFocusFireModule";
} // namespace

// CLI flags
DEFINE_string(hostname, "localhost", "Host running torchcraft server");
DEFINE_int32(port, 11111, "Port for torchcraft");
DEFINE_string(modules, kDefaultModules, "Comma-separated list of bot modules");
DEFINE_int32(frameskip, 1, "Frame skip for screen updates");
DEFINE_int32(timeout, 120000, "Timeout for TorchCraft connection");
DEFINE_int32(
    realtime_factor,
    -1,
    "Delay execution to achieve desired realtime factor");
DEFINE_bool(
    warn_if_slow,
    true,
    "Warn if stepping through modules takes too long");
DEFINE_bool(
    blocking,
    false,
    "Run bot step in main thread, possibly blocking game execution");
DEFINE_bool(logsinktostderr, true, "Log sink to stderror.");
DEFINE_string(logsinkdir, "", "Optional directory to write sink log files");
DEFINE_bool(
    non_tournament,
    false,
    "Run bot with diagnostics and graphical debugging");
DECLARE_string(umm_path);

namespace {

// Establish connection and perform initial handshake
std::unique_ptr<tc::Client> makeClient() {
  // Establish connection
  auto client = std::unique_ptr<tc::Client>(new tc::Client());
  if (!client->connect(FLAGS_hostname, FLAGS_port, FLAGS_timeout)) {
    throw std::runtime_error(
        std::string("Error establishing connection: ") + client->error());
  }
  LOG(INFO) << "Connected to torchraft server at " << FLAGS_hostname << ":"
            << FLAGS_port;

  // Perform handshake
  tc::Client::Options opts;
  std::vector<std::string> upd;
  if (!client->init(upd, opts)) {
    throw std::runtime_error(
        std::string("Error initializing connection: ") + client->error());
  }
  if (client->state()->replay) {
    throw std::runtime_error("Expected non-replay map");
  }

  return client;
}

} // namespace

int main(int argc, char** argv) {
  fairrsh::init();
  google::InitGoogleLogging(argv[0]);
  gflags::ParseCommandLineFlags(&argc, &argv, true);

  // We need to init the logging after we have parsed the command
  // line flags since it depends on flags set by it
  fairrsh::initLogging(argv[0], FLAGS_logsinkdir, FLAGS_logsinktostderr);

  try {
    auto cl = makeClient();
    Player bot(std::move(cl));
    bot.setFrameskip(FLAGS_frameskip);
    bot.setRealtimeFactor(FLAGS_realtime_factor);
    bot.setWarnIfSlow(FLAGS_non_tournament && FLAGS_warn_if_slow);
    bot.setNonBlocking(!FLAGS_blocking);
    bot.setCollectStats(FLAGS_non_tournament);
    bot.setCollectTimers(FLAGS_non_tournament);

    // Add bot modules
    bot.addModule(Module::make(kAutoTopModule));
    for (auto name : utils::stringSplit(FLAGS_modules, ',')) {
      if (!name.empty()) {
        bot.addModule(Module::make(name));
      }
    }
    bot.addModule(Module::make(kAutoBottomModule));
    if (FLAGS_umm_path != "") {
      bot.addModule(Module::make("UMMHelper"));
    }

    bot.run();

    if (bot.state()->won()) {
      LOG(WARNING) << "Victory!!";
    } else {
      LOG(WARNING) << "Oh noes we lost :( -- with "
                   << bot.state()->unitsInfo().myBuildings().size()
                   << " buildings left";
    }
  } catch (std::exception& e) {
    LOG(DFATAL) << "Exception: " << e.what();
    // FATAL terminates the program, though
    fairrsh::shutdown(FLAGS_logsinktostderr);
    return EXIT_FAILURE;
  }

  fairrsh::shutdown(FLAGS_logsinktostderr);
  return EXIT_SUCCESS;
}
