/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#include "fairrsh.h"

#include "buildtype.h"
#include "fonf.h"
#include "modules.h"

#include <glog/logging.h>
#include <torchcraft/client.h>

#include <ctime>
#include <fstream>
#include <iomanip>
#include <iostream>
#include <sstream>
#include <string>

namespace fairrsh {

namespace {

#ifdef WITHOUT_POSIX
struct tm* localtime_r(time_t* _clock, struct tm* _result) {
  struct tm* p = localtime(_clock);
  if (p) {
    *(_result) = *p;
  }
  return p;
}
#endif // WITHOUT_POSIX

std::ofstream logFilesForSink[google::NUM_SEVERITIES];

// Thread-local logging prefix, usually using 5 digits for the frame but keeping
// space for 6 just to be on the safe side.
// TODO: This does not work too well; need to check what glog does when used
// from multiple threads.
thread_local char logPrefix[7] = "XXXXX";

/// A custom log sink
class PrefixLogSink : public google::LogSink {
 public:
  virtual void send(
      google::LogSeverity severity,
      const char* fullFilename,
      const char* baseFilename,
      int line,
      const struct ::tm* tmTime,
      const char* message,
      size_t messageLen) override {
    // Log to stderr or file depending on flag
    auto& chan = (logToStderr_) ? std::cerr : logFilesForSink[severity];

    switch (severity) {
      case google::GLOG_INFO:
        chan << "I";
        break;
      case google::GLOG_WARNING:
        chan << "W";
        break;
      case google::GLOG_ERROR:
        chan << "E";
        break;
      case google::GLOG_FATAL:
        chan << "F";
        break;
      default:
        chan << "?";
        break;
    }

    chan << logPrefix << " [" << baseFilename << ":" << line << "] ";
    chan.write(message, messageLen);
    chan << std::endl;
  }

  void setSinkDestination(bool logToStderr) {
    logToStderr_ = logToStderr;
  }

  bool isSinkToStderr() const {
    return logToStderr_;
  }

 private:
  bool logToStderr_ = true;
};

PrefixLogSink logSink;

std::string createLogFileName(
    const char* argv0,
    std::string logSinkDir,
    google::LogSeverity severity) {
  // Logic shamelessly stolen from glog for compatibility
  const char* slash = strrchr(argv0, '/');
#if defined(_WIN32) || defined(_WIN64)
  if (!slash) {
    slash = strrchr(argv0, '\\');
  }
#endif

  std::string logFileName(slash ? slash + 1 : argv0);

  // Append time to log file's name to make it unique
  // Same format as glog
  std::time_t timeStamp =
      std::chrono::system_clock::to_time_t(std::chrono::system_clock::now());
  struct std::tm tmTime;
  localtime_r(&timeStamp, &tmTime);

  std::ostringstream timeStringStream;
  timeStringStream.fill('0');
  timeStringStream << 1900 + tmTime.tm_year << std::setw(2) << 1 + tmTime.tm_mon
                   << std::setw(2) << tmTime.tm_mday << '-' << std::setw(2)
                   << tmTime.tm_hour << std::setw(2) << tmTime.tm_min
                   << std::setw(2) << tmTime.tm_sec;

  logFileName += "." + timeStringStream.str();

  // Should I also append a process id?
  switch (severity) {
    case google::GLOG_INFO:
      logFileName += ".INFO";
      break;
    case google::GLOG_WARNING:
      logFileName += ".WARNING";
      break;
    case google::GLOG_ERROR:
      logFileName += ".ERROR";
      break;
    case google::GLOG_FATAL:
      logFileName += ".FATAL";
      break;
    default:
      logFileName += ".UNKNOWN";
      break;
  }

  if (!logSinkDir.empty()) {
    logFileName = logSinkDir + "/" + logFileName;
  }
  return logFileName;
}

} // namespace

void init() {
  ::torchcraft::init();
  buildtypes::initialize();
  fonf::initialize();

  // Needed for timers in our bot
  static_assert(
      hires_clock::is_steady, "Steady high-resolution clock required");
}

void initLogging(
    const char* execName,
    std::string logSinkDir,
    bool logSinkToStderr) {
  // Setup all the file handles for logging for our sink
  if (!logSinkToStderr) {
    for (int i = 0; i < google::NUM_SEVERITIES; ++i) {
      logFilesForSink[i] =
          std::ofstream(createLogFileName(execName, logSinkDir, i));
    }
    logSink.setSinkDestination(logSinkToStderr);
  }

  // Logging setup
  google::InstallFailureSignalHandler();
  google::AddLogSink(&logSink);
}

void shutdown(bool logSinkToStderr) {
  if (!logSinkToStderr) {
    for (int i = 0; i < google::NUM_SEVERITIES; ++i) {
      logFilesForSink[i].close();
    }
  }

  google::ShutdownGoogleLogging();
}

void setLoggingFrame(int frame) {
  snprintf(logPrefix, sizeof(logPrefix), "%05d", frame);
}

void unsetLoggingFrame() {
  strcpy(logPrefix, "XXXXX");
}

} // namespace fairrsh
