/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#include "base.h"
#include "registry.h"

namespace fairrsh {

class ABBO10HatchLing : public ABBOBase {
  RTTR_ENABLE(ABBOBase)
 public:
  using ABBOBase::ABBOBase;

  bool buildExtractor = false;
  bool hasBuiltExtractor = false;
  bool hasmadelings = false;

  Position nextSunkenPos;
  virtual void preBuild2(State* state, Module* module) override {
    using namespace buildtypes;
    using namespace autobuild;

    state->board()->post("kMinScoutFrame", 0);

    auto st = getMyState(state);
    if (!hasBuiltExtractor && countPlusProduction(st, Zerg_Drone) == 9 &&
        countPlusProduction(st, Zerg_Overlord) == 1) {
      buildExtractor = true;
      for (Unit* u : state->unitsInfo().myBuildings()) {
        if (u->type == Zerg_Extractor && !u->completed()) {
          state->board()->postUPC(
              utils::makeSharpUPC(u, Command::Cancel), -1, module);
          hasBuiltExtractor = true;
        }
      }
    } else {
      buildExtractor = false;
    }

    bool attack = true;
    if (st.frame < 24 * 60 * 15) {
      if (enemyMutaliskCount && !has(st, Zerg_Spire) &&
          armySupply < enemyArmySupply) {
        attack = false;
      }
    }
    if (weArePlanningExpansion) {
      attack = true;
    }
    state->board()->post("TacticsAttack", attack);

    nextSunkenPos = findSunkenPos(state, Zerg_Sunken_Colony, false, true);

    if (!hasmadelings) {
      if (countPlusProduction(st, Zerg_Zergling) >= 6) {
        hasmadelings = true;
      }
    }
  }

  virtual void buildStep2(autobuild::BuildState& st) override {
    using namespace buildtypes;
    using namespace autobuild;

    st.autoBuildRefineries = false;
    autoBuildHatcheries = st.frame >= 15 * 60 * 5;

    if (enemyRace != +tc::BW::Race::Terran &&
        enemyRace != +tc::BW::Race::Protoss) {
      if (!hasmadelings) {
        buildN(Zerg_Zergling, 6);

        buildN(Zerg_Spawning_Pool, 1);
        buildN(Zerg_Drone, 9);
        return;
      }
    }

    if (hasOrInProduction(st, Zerg_Creep_Colony)) {
      build(Zerg_Spore_Colony);
      return;
    }

    auto placeSpores = [&](int n) {
      if (countPlusProduction(st, Zerg_Spore_Colony) < n) {
        build(Zerg_Creep_Colony, nextSunkenPos);
      }
      buildN(Zerg_Evolution_Chamber, 1);
    };

    if (has(st, Zerg_Spawning_Pool)) {
      build(Zerg_Zergling);
      if (st.frame >= 24 * 60 * 7 &&
          armySupply >= enemyArmySupply + 4.0 - enemyStaticDefenceCount) {
        buildN(Zerg_Evolution_Chamber, 1);
      }
      if (countPlusProduction(st, Zerg_Zergling) >= 40 || has(st, Zerg_Spire)) {
        build(Zerg_Mutalisk);
      }
      if (st.frame >= 15 * 60 * 7) {
        if (countProduction(st, Zerg_Drone) == 0 &&
            armySupply > enemyArmySupply + (st.workers >= 16 ? 8 : 0)) {
          build(Zerg_Drone);
        }
        if (st.workers >= 12) {
          buildN(Zerg_Spire, 1);
        }
      }
      if (st.workers >= 11) {
        build(Metabolic_Boost);
        buildN(Zerg_Extractor, 1);
      }
      buildN(Zerg_Zergling, 6);

      if (countPlusProduction(st, Zerg_Hydralisk) +
              countPlusProduction(st, Zerg_Scourge) <
          enemyAirArmySupply) {
        if (has(st, Zerg_Spire)) {
          buildN(
              Zerg_Scourge,
              (int)(enemyAirArmySupply - std::max(enemyMutaliskCount - countPlusProduction(st, Zerg_Mutalisk), 0) * 2));
        } else {
          if (enemyMutaliskCount) {
            buildN(Zerg_Spire, 1);
            placeSpores(std::max(enemyMutaliskCount / 3, 2));
          } else {
            build(Zerg_Hydralisk);
          }
        }
      }
    }
    buildN(Zerg_Overlord, 2);
    buildN(Zerg_Spawning_Pool, 1);
    if (countPlusProduction(st, Zerg_Hatchery) == 1) {
      build(Zerg_Hatchery, nextBase);
      if (!hasBuiltExtractor && buildExtractor) {
        buildN(Zerg_Extractor, 1);
      }
      buildN(Zerg_Drone, 9);
    }
  }
};

RTTR_REGISTRATION {
  rttr::registration::class_<ABBO10HatchLing>("ABBO10HatchLing")(
      metadata("type", rttr::type::get<ABBO10HatchLing>()))
      .constructor<UpcId>();
}
}
