/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#pragma once
#include <iostream>

#include <cmath>
#include <functional>
#include <iostream>
#include <utility>

namespace fairrsh {

typedef int PlayerId;
typedef int FrameNum;
typedef int UpcId;
UpcId constexpr kRootUpcId = -1;
UpcId constexpr kInvalidUpcId = -2;
UpcId constexpr kFilteredUpcId = -3;

double constexpr DEG_PER_RAD = 180 / M_PI;

// Source https://gist.github.com/acidleaf/8957147
template <typename T>
class Vec2T {
 public:
  T x, y;

  Vec2T() : x(0), y(0) {}
  Vec2T(T x, T y) : x(x), y(y) {}
  template <typename U>
  Vec2T(const U* const u) : x(u->x), y(u->y) {}
  template <typename U>
  Vec2T(U* const u) : x(u->x), y(u->y) {}
  template <typename U>
  Vec2T(const U& p) : x(p.x), y(p.y) {}
  template <typename U, typename V>
  Vec2T(const std::pair<U, V>& p) : x(p.first), y(p.second) {}

  bool operator==(Vec2T const& v) const {
    return x == v.x && y == v.y;
  }
  bool operator!=(Vec2T const& v) const {
    return x != v.x || y != v.y;
  }

  Vec2T& operator=(Vec2T const& v) {
    x = v.x;
    y = v.y;
    return *this;
  }

  // For use in std::map as a key, since it's ordered and keeps a compare
  bool operator<(Vec2T<T> const& p) const {
    return x < p.x || (x == p.x && y < p.y);
  }

  Vec2T operator+(Vec2T const& v) const {
    return Vec2T(x + v.x, y + v.y);
  }
  Vec2T operator-(Vec2T const& v) const {
    return Vec2T(x - v.x, y - v.y);
  }

  Vec2T& operator+=(Vec2T const& v) {
    x += v.x;
    y += v.y;
    return *this;
  }
  Vec2T& operator-=(Vec2T const& v) {
    x -= v.x;
    y -= v.y;
    return *this;
  }

  Vec2T operator+(T s) const {
    return Vec2T(x + s, y + s);
  }
  Vec2T operator-(T s) const {
    return Vec2T(x - s, y - s);
  }
  Vec2T operator*(T s) const {
    return Vec2T(x * s, y * s);
  }
  Vec2T operator/(T s) const {
    return Vec2T(x / s, y / s);
  }

  Vec2T& operator+=(T s) {
    x += s;
    y += s;
    return *this;
  }
  Vec2T& operator-=(T s) {
    x -= s;
    y -= s;
    return *this;
  }
  Vec2T& operator*=(T s) {
    x *= s;
    y *= s;
    return *this;
  }
  Vec2T& operator/=(T s) {
    x /= s;
    y /= s;
    return *this;
  }

  void set(T x, T y) {
    this->x = x;
    this->y = y;
  }

  Vec2T& rotate(double deg) {
    double theta = deg / DEG_PER_RAD;
    double c = std::cos(theta);
    double s = std::sin(theta);
    double tx = x * c - y * s;
    double ty = x * s + y * c;
    x = tx;
    y = ty;
    return *this;
  }

  Vec2T& normalize() {
    if (length() == 0)
      return *this;
    *this *= (1.0 / length());
    return *this;
  }

  double dist(Vec2T const& v) const {
    Vec2T d(v.x - x, v.y - y);
    return d.length();
  }
  double length() const {
    return std::sqrt(x * x + y * y);
  }
  void truncate(double length) {
    double angle = std::atan2(y, x);
    x = length * std::cos(angle);
    y = length * std::sin(angle);
  }

  Vec2T ortho() const {
    return Vec2T(y, -x);
  }

  static double cos(Vec2T const& v1, Vec2T const& v2) {
    double l1 = v1.length();
    double l2 = v2.length();
    double l12 = l1 * l2;
    if (l12 == 0) {
      return 0;
    }
    return Vec2T::dot(v1, v2) / l12;
  }
  static T dot(Vec2T const& v1, Vec2T const& v2) {
    return v1.x * v2.x + v1.y * v2.y;
  }

  T dot(Vec2T const& v2) {
    return dot(*this, v2);
  }
  static T cross(Vec2T const& v1, Vec2T const& v2) {
    return (v1.x * v2.y) - (v1.y * v2.x);
  }
};

typedef Vec2T<float> Vec2;
typedef Vec2T<int> Position;

} // namespace fairrsh

namespace std {
template <typename T>
inline ostream& operator<<(ostream& strm, fairrsh::Vec2T<T> const& p) {
  return strm << " (" << p.x << ", " << p.y << ")";
}

template <typename T>
struct hash<fairrsh::Vec2T<T>> {
  size_t operator()(fairrsh::Vec2T<T> const& pos) const {
    return hashT(pos.x * pos.y) ^ hashT(pos.y);
  }

 private:
  function<size_t(T)> hashT = hash<T>();
};
} // namespace std
