/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 */

#pragma once

#include "basetypes.h"

#include <tuple>
#include <unordered_map>
#include <unordered_set>
#include <vector>

namespace BWEM {
class Map;
class Area;
class ChokePoint;
} // namespace BWEM

namespace fairrsh {

class AreaInfo;
class State;
struct Unit;

/**
 * Represents an area on the map.
 *
 * Areas are regions determined by static map analysis using BWEM. They
 * correspond to the respective BWEM::Area. This struct is used to aggregate all
 * game state information (e.g. units, visibility) local to the respective area.
 */
struct Area {
  /// ID of BWEM area. This corresponds to the index in the areas vector (+1)
  int id = -1;
  /// Center in the area's bounding box in walk tiles.
  int x = 0;
  /// Center of the area's bounding box in walk tiles.
  int y = 0;
  /// Area size in walk tiles.
  int size = 0;
  /// Possible base locations.
  std::vector<Position> baseLocations;
  /// All units in this area that are not dead. This includes gone units.
  std::vector<Unit*> liveUnits;
  /// All units in this area that are currently visible.
  std::vector<Unit*> visibleUnits;

  BWEM::Area const* area;

  /// Pointer to container object
  AreaInfo* areaInfo = nullptr;
  /// Accessible neighbors
  std::vector<Area*> neighbors;

  FrameNum lastExplored = 0;

  bool isMyBase = false;
  bool isEnemyBase = false;
  bool isPossibleEnemyBase = false;
  /// True if we have a completed resource depot at one of the area's
  /// baseLocations.
  bool isMyExpand = false;
  /// True if the enemy has a building in this area.
  bool isEnemyExpand = false;
  /// True if we ever had a completed resource depot at one of the area's
  /// baseLocations.
  bool wasMyExpand = false;
  /// True if the enemy ever had a building in this area.
  bool wasEnemyExpand = false;
  bool hasMyBuildings = false;
  bool hasEnemyBuildings = false;
  /// Strength of all ground units in this area.
  double myGndStrength = 0;
  /// Strength of all air units in this area.
  double myAirStrength = 0;
  /// Strength of all detector units in this area.
  double myDetStrength = 0;
  double enemyGndStrength = 0;
  double enemyAirStrength = 0;
  double enemyDetStrength = 0;
};

/**
 * Access point for area and base information.
 *
 * Beside providing access to area information, this class also provides a few
 * convenience wrappers for functionality in BWEM::Map.
 */
class AreaInfo {
 public:
  AreaInfo(State* state);

  void update();

  std::vector<Area> const& areas() const;
  Area& getArea(int id);
  Area const& getArea(int id) const;
  Area& getArea(Position p);
  Area const& getArea(Position p) const;
  Area* tryGetArea(int id);
  Area const* tryGetArea(int id) const;
  Area* tryGetArea(Position p);
  Area const* tryGetArea(Position p) const;

  Unit* myBase() const;
  Area const* myBaseArea() const;
  Position myBasePosition() const;
  int numExpands() const;
  /// Returns the depot of our nth expansion (starting from 1) or myBase (if n
  /// is 0).
  Unit* myExpand(int n) const;
  Area const* myExpandArea(int n) const;
  Position myExpandPosition(int n) const;
  /// Returns closest base/expand according to euclidean distance.
  int myClosestExpand(Position const& p) const;
  std::vector<Unit*> myExpandResources(int n) const;
  float mineralSaturationAtMyExpand(int n) const;
  void setMineralSaturationAtMyExpand(int n, float sat);

  bool foundEnemyBase() const;
  Unit* enemyBase() const;
  Area const& enemyBaseArea() const;
  Position enemyBasePosition() const;
  std::vector<Position> const& candidateEnemyStartLoc() const;
  std::tuple<size_t, size_t, size_t> getCacheStats() const;

 private:
  struct BaseInfo {
    int areaId = -1;
    int baseId = -1;
    float mineralSaturation = 0.0f;
  };

  void initialize();
  void updateUnits();
  void updateEnemyStartingLocations();
  void updateStrengths();
  void updateNeighbors();
  void resetCacheStats();
  Area* getCachedArea(Position p);

  State* state_ = nullptr;
  BWEM::Map* map_ = nullptr;
  std::vector<Area> areas_;
  std::vector<Position> candidateEnemyStartLoc_;
  Unit* enemyBase_ = nullptr;
  std::vector<Unit*> myBases_;
  std::unordered_map<Unit*, BaseInfo> myBasesInfo_;
  std::unordered_set<Unit*> macroDepots_;
  std::unordered_map<int, int> neighborAreaCache_;

  size_t cacheAccess_ = 0;
  size_t totalAccess_ = 0;
};

} // namespace fairrsh
