#include "Common.h"
#include "WorkerManager.h"
#include "StrategyManager.h"
#include "Micro.h"
#include "UnitUtil.h"

using namespace UAlbertaBot;

WorkerManager::WorkerManager() 
{
    previousClosestWorker = nullptr;
}

WorkerManager & WorkerManager::Instance() 
{
	static WorkerManager instance;
	return instance;
}

void WorkerManager::update() 
{
	//every 30 seconds look wheter workers need to be rebalanced
	//getLeastSaturatedDepot(nullptr, true);
	if (StrategyManager::Instance().strategy.HydraliskDeflation +
		StrategyManager::Instance().strategy.LurkerDeflation +
		StrategyManager::Instance().strategy.MutaliskDeflation +
		StrategyManager::Instance().strategy.UltraliskDeflation +
		StrategyManager::Instance().strategy.GuardianDeflation == 0
		&& UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Lair) + UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Hive) == 0
		&& ((BWAPI::Broodwar->self()->isUpgrading(BWAPI::UpgradeTypes::Metabolic_Boost) || BWAPI::Broodwar->self()->getUpgradeLevel(BWAPI::UpgradeTypes::Metabolic_Boost) > 0)
			&& UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Evolution_Chamber) == 0))
	{
		gasBank = 0;
	}
	else
	{
		gasBank = std::min(BWAPI::Broodwar->self()->minerals() + 300, BWAPI::Broodwar->self()->minerals() * 2);
		gasBank = std::max(gasBank, 100);
	}
	bool alsoUpdateIdleBuilders = false;
	if (BWAPI::Broodwar->getFrameCount() % 720 == 0)
	{
		rebalanceWorkers(1);
	}
	if (BWAPI::Broodwar->getFrameCount() % 72 == 0)
	{
		alsoUpdateIdleBuilders = true;
	}

	updateWorkerStatus(alsoUpdateIdleBuilders);
	handleGasWorkers();
	handleIdleWorkers();
	handleMoveWorkers();
	handleCombatWorkers();

	drawResourceDebugInfo();
	drawWorkerInformation(450,20);

	workerData.drawDepotDebugInfo();

    handleRepairWorkers();
}

void WorkerManager::updateWorkerStatus(bool updateBuilders) 
{
	int maxGasWorkers = (workerData.getNumWorkers() / 3.0);
	if (BWAPI::Broodwar->self()->gas() > gasBank)
	{
		maxGasWorkers = 0;
	}
	//BWAPI::Broodwar->drawTextScreen(BWAPI::Position(30, 300), "GasWorkers: %d Wanted GasWorkers: %d Gas to bank: %d", workerData.getNumGasWorkers(), maxGasWorkers, gasBank);
	// for each of our Workers
	for (auto & worker : workerData.getWorkers())
	{
		if (!worker->isCompleted())
		{
			continue;
		}
		//getLeastSaturatedDepot(worker, true);
		//BWAPI::Broodwar->drawTextMap(worker->getPosition(), "%d", workerData.getWorkerJob(worker));
		//BWAPI::Broodwar->drawTextMap(worker->getPosition(), "Job: %d", workerData.getWorkerJob(worker));

		// if it's idle
		if (worker->isIdle() &&	workerData.getWorkerJob(worker) != WorkerData::Scout)
		{
			if (workerData.getWorkerJob(worker) != WorkerData::Build || updateBuilders)
			{
				workerData.setWorkerJob(worker, WorkerData::Idle, nullptr);
			}
		}
		// if its job is gas
		if (workerData.getWorkerJob(worker) == WorkerData::Gas)
		{
			//BWAPI::Broodwar->drawTextScreen(BWAPI::Position(30, 280), "There are gas-workers!");
			if (worker->isCarryingMinerals()) //I might have had to fight as gasworker and was sent to mineral-block
			{
				setMineralWorker(worker);
			}
			
			BWAPI::Unit refinery = workerData.getWorkerResource(worker);

			// if the refinery doesn't exist anymore
			if (!refinery || !refinery->exists() ||	refinery->getHitPoints() <= 0)
			{
				setMineralWorker(worker);
			}
			if (workerData.getNumGasWorkers() > maxGasWorkers
				&& !worker->isCarryingGas()
				&& worker->isMoving()
				&& BWAPI::Broodwar->self()->gas() > gasBank)
			{
				setMineralWorker(worker);
				break; //doing one per cycle is enough
			}
		}
	}
}

void WorkerManager::setRepairWorker(BWAPI::Unit worker, BWAPI::Unit unitToRepair)
{
    workerData.setWorkerJob(worker, WorkerData::Repair, unitToRepair);
}

void WorkerManager::stopRepairing(BWAPI::Unit worker)
{
    workerData.setWorkerJob(worker, WorkerData::Idle, nullptr);
}

void WorkerManager::handleGasWorkers() 
{
	int maxGasWorkers = (workerData.getNumWorkers() / 3.0);
	if (BWAPI::Broodwar->self()->gas() > gasBank)
	{
		maxGasWorkers = 0;
	}
	if (workerData.getNumGasWorkers() >= maxGasWorkers)
	{
		return;
	}
	// for each unit we have
	for (auto & unit : BWAPI::Broodwar->self()->getUnits())
	{
		// if that unit is a refinery
		if (unit->getType().isRefinery() && unit->isCompleted())
		{
			// get the number of workers currently assigned to it
			int numAssigned = workerData.getNumAssignedWorkers(unit);

			// if it's less than we want it to be, fill 'er up
			for (int i=0; i<(std::min(Config::Macro::WorkersPerRefinery - numAssigned, maxGasWorkers)); ++i)
			{
				BWAPI::Unit gasWorker = getGasWorker(unit);
				if (gasWorker)
				{
					workerData.setWorkerJob(gasWorker, WorkerData::Gas, unit);
					return;
				}
			}
		}
	}
}

void WorkerManager::handleIdleWorkers() 
{
	// for each of our workers
	for (auto & worker : workerData.getWorkers())
	{
        UAB_ASSERT(worker != nullptr, "Worker was null");

		// if it is idle
		if (workerData.getWorkerJob(worker) == WorkerData::Idle) 
		{
			// send it to the nearest mineral patch
			setMineralWorker(worker);
		}
	}
}

void WorkerManager::handleRepairWorkers()
{
    if (BWAPI::Broodwar->self()->getRace() != BWAPI::Races::Terran)
    {
        return;
    }

    for (auto & unit : BWAPI::Broodwar->self()->getUnits())
    {
        if (unit->getType().isBuilding() && (unit->getHitPoints() < unit->getType().maxHitPoints()))
        {
            BWAPI::Unit repairWorker = getClosestMineralWorkerTo(unit);
            setRepairWorker(repairWorker, unit);
            break;
        }
    }
}

// bad micro for combat workers
void WorkerManager::handleCombatWorkers()
{
	for (auto & worker : workerData.getWorkers())
	{
        UAB_ASSERT(worker != nullptr, "Worker was null");
		if (workerData.getWorkerJob(worker) == WorkerData::Scout && !Config::Strategy::ScoutHarassEnemy)
		{
			continue;
		}
		if (workerData.getWorkerJob(worker) == WorkerData::Build && !Config::Strategy::ScoutHarassEnemy)
		{
			continue;
		}
		if (workerData.getWorkerJob(worker) == WorkerData::Move && !Config::Strategy::ScoutHarassEnemy)
		{
			continue;
		}
		BWAPI::Unit target = getClosestEnemyUnit(worker);
		BWAPI::Unit mineralField = nullptr;
		double biggestDistance = 0;
		for each (auto patch in workerData.getMineralPatchesNearDepot(worker))
		{
			double distance = patch->getDistance(worker->getPosition());
			if (distance > biggestDistance)
			{
				mineralField = patch;
				biggestDistance = distance;
			}
		}
		if (target)
		{
			if (!target->isMoving() && worker->getHitPoints() > worker->getType().maxHitPoints() / 2)
			{
				BWAPI::Broodwar->drawCircleMap(worker->getPosition().x, worker->getPosition().y, 4, BWAPI::Colors::Yellow, true);
				Micro::SmartAttackMove(worker, target->getPosition());
			}
			else
			{
				if (mineralField)
				{
					Micro::SmartRightClick(worker, mineralField);
				}
				else
				{
					workerData.setWorkerJob(worker, WorkerData::Idle, nullptr);
				}
			}
		}
		else
		{
			if (worker->isAttacking())
			{
				workerData.setWorkerJob(worker, WorkerData::Idle, nullptr);
			}
		}
		BWAPI::Unit scarab = getClosestEnemyUnit(worker, 400);
		if (scarab != nullptr && (scarab->getType() == BWAPI::UnitTypes::Protoss_Scarab || scarab->getType() == BWAPI::UnitTypes::Protoss_Reaver || scarab->getType() == BWAPI::UnitTypes::Protoss_Shuttle))
		{
			Micro::SmartAvoid(worker, scarab->getPosition(), MapTools::Instance().getBaseCenter(), scarab->getPosition());
		}
	}
}

BWAPI::Unit WorkerManager::getClosestEnemyUnit(BWAPI::Unit worker, int range)
{
    UAB_ASSERT(worker != nullptr, "Worker was null");

	BWAPI::Unit closestUnit = nullptr;
	double closestDist = 10000;

	for (auto & unit : BWAPI::Broodwar->enemy()->getUnits())
	{
		if (unit->isFlying())
		{
			continue;
		}
		if (unit->isMoving() && (unit->getType() != BWAPI::UnitTypes::Protoss_Scarab || unit->getType() != BWAPI::UnitTypes::Protoss_Reaver))
		{
			continue;
		}
		double dist = unit->getDistance(worker);
		double minAttackDistance = range;
		if ((dist < minAttackDistance) && (!closestUnit || (dist < closestDist)))
		{
			closestUnit = unit;
			closestDist = dist;
		}
	}

	return closestUnit;
}

void WorkerManager::finishedWithCombatWorkers()
{
	for (auto & worker : workerData.getWorkers())
	{
        UAB_ASSERT(worker != nullptr, "Worker was null");

		if (workerData.getWorkerJob(worker) == WorkerData::Combat)
		{
			setMineralWorker(worker);
		}
	}
}

BWAPI::Unit WorkerManager::getClosestMineralWorkerTo(BWAPI::Unit enemyUnit)
{
    UAB_ASSERT(enemyUnit != nullptr, "enemyUnit was null");

    BWAPI::Unit closestMineralWorker = nullptr;
    double closestDist = 100000;

    if (previousClosestWorker)
    {
        if (previousClosestWorker->getHitPoints() > 0)
        {
            return previousClosestWorker;
        }
    }

    // for each of our workers
	for (auto & worker : workerData.getWorkers())
	{
        UAB_ASSERT(worker != nullptr, "Worker was null");
		if (!worker)
		{
			continue;
		}
		// if it is a move worker
        if (workerData.getWorkerJob(worker) == WorkerData::Minerals) 
		{
			double dist = worker->getDistance(enemyUnit);

            if (!closestMineralWorker || dist < closestDist)
            {
                closestMineralWorker = worker;
                dist = closestDist;
            }
		}
	}

    previousClosestWorker = closestMineralWorker;
    return closestMineralWorker;
}

BWAPI::Unit WorkerManager::getWorkerScout()
{
    // for each of our workers
	for (auto & worker : workerData.getWorkers())
	{
        UAB_ASSERT(worker != nullptr, "Worker was null");
		if (!worker)
		{
			continue;
		}
		// if it is a move worker
        if (workerData.getWorkerJob(worker) == WorkerData::Scout) 
		{
			return worker;
		}
	}

    return nullptr;
}

void WorkerManager::handleMoveWorkers() 
{
	// for each of our workers
	for (auto & worker : workerData.getWorkers())
	{
        UAB_ASSERT(worker != nullptr, "Worker was null");

		// if it is a move worker
		if (workerData.getWorkerJob(worker) == WorkerData::Move) 
		{
			WorkerMoveData data = workerData.getWorkerMoveData(worker);
			
			Micro::SmartMove(worker, data.position);
			return;
		}
	}
}

// set a worker to mine minerals
void WorkerManager::setMineralWorker(BWAPI::Unit unit)
{
    UAB_ASSERT(unit != nullptr, "Unit was null");

	// check if there is a mineral available to send the worker to
	BWAPI::Unit depot = getLeastSaturatedDepot(unit, true);

	// if there is a valid mineral
	if (depot)
	{
		// update workerData with the new job
		workerData.setWorkerJob(unit, WorkerData::Minerals, depot);
	}
	else
	{
		// BWAPI::Broodwar->printf("No valid depot for mineral worker");
	}
}

BWAPI::Unit WorkerManager::getClosestDepot(BWAPI::Unit worker)
{
	UAB_ASSERT(worker != nullptr, "Worker was null");

	BWAPI::Unit closestDepot = nullptr;
	double closestDistance = 0;

	for (auto & unit : BWAPI::Broodwar->self()->getUnits())
	{
        UAB_ASSERT(unit != nullptr, "Unit was null");

		if (unit->getType().isResourceDepot() && (unit->isCompleted() || unit->getType() == BWAPI::UnitTypes::Zerg_Lair || unit->getType() == BWAPI::UnitTypes::Zerg_Hive) && !workerData.depotIsFull(unit))
		{
			double distance = unit->getDistance(worker);
			if (!closestDepot || distance < closestDistance)
			{
				closestDepot = unit;
				closestDistance = distance;
			}
		}
	}
	if (closestDepot == nullptr)
	{
		for (auto & unit : BWAPI::Broodwar->self()->getUnits())
		{
			UAB_ASSERT(unit != nullptr, "Unit was null");

			if (unit->getType().isResourceDepot() && (unit->isCompleted() || unit->getType() == BWAPI::UnitTypes::Zerg_Lair || unit->getType() == BWAPI::UnitTypes::Zerg_Hive))
			{
				double distance = unit->getDistance(worker);
				if (!closestDepot || distance < closestDistance)
				{
					closestDepot = unit;
					closestDistance = distance;
				}
			}
		}
	}

	return closestDepot;
}

BWAPI::Unit WorkerManager::getLeastSaturatedDepot(BWAPI::Unit worker, bool considerEnemies)
{
	BWAPI::Unit closestDepot = nullptr;
	BWAPI::Unit lowestSaturatedDepot = nullptr;
	double lowestSaturation = DBL_MAX;
	double cloesestSaturation = 0;
	double lowestDistance = DBL_MAX;

	for (auto & unit : BWAPI::Broodwar->self()->getUnits())
	{
		UAB_ASSERT(unit != nullptr, "Unit was null");

		if (unit->getType().isResourceDepot() && (unit->isCompleted() || unit->getType() == BWAPI::UnitTypes::Zerg_Lair || unit->getType() == BWAPI::UnitTypes::Zerg_Hive))
		{
			if (BuildingPlacer::Instance().tileOverlapsBaseLocation(unit->getTilePosition(), BWAPI::UnitTypes::Zerg_Queens_Nest))
			{
				double saturation = 0;
				double distance = 0;
				if (workerData.getMineralsNearDepot(unit) == 0)
				{
					continue;
				}
				if (workerData.getMineralsNearDepot(unit) > 0)
				{
					int oversat = workerData.getWorkersNearDepot(unit) - workerData.getMineralsNearDepot(unit);
					if (worker == nullptr)
					{
						oversat = workerData.getWorkersNearDepot(unit);
					}
					if (oversat > 0)
					{
						saturation += BWAPI::Broodwar->self()->getRace().getWorker().mineralPrice() * oversat;
					}
				}
				double enemies = 0;
				if (considerEnemies)
				{
					saturation += (workerData.getEnemyStrenghtNearDepot(unit));
				}
				//BWAPI::Broodwar->drawTextMap(unit->getPosition(), "%d", (int)saturation);

				if (worker)
				{
					distance = unit->getPosition().getDistance(worker->getPosition());
				}

				if (saturation <= lowestSaturation)
				{
					lowestSaturatedDepot = unit;
					lowestSaturation = saturation;
				}
				if (distance <= lowestDistance)
				{
					closestDepot = unit;
					lowestDistance = distance;
					cloesestSaturation = saturation;
				}
			}
		}
	}
	if ((!StrategyManager::Instance().strategy.underSiege || worker == nullptr) && cloesestSaturation > lowestSaturation + 150)
	{
		closestDepot = lowestSaturatedDepot;
	}

	//if (closestDepot)
	//{
	//	if (worker == nullptr)
	//	{
	//		BWAPI::Broodwar->drawTextMap(closestDepot->getPosition(), "%d => <=", (int)lowestSaturation);
	//	}
	//	else
	//	{
	//		BWAPI::Broodwar->drawLineMap(closestDepot->getPosition(), worker->getPosition(), BWAPI::Colors::White);
	//	}
	//}
	UAB_ASSERT(closestDepot != nullptr, "No depot to go");
	return closestDepot;
}

// other managers that need workers call this when they're done with a unit
void WorkerManager::finishedWithWorker(BWAPI::Unit unit) 
{
	UAB_ASSERT(unit != nullptr, "Unit was null");

	//BWAPI::Broodwar->printf("BuildingManager finished with worker %d", unit->getID());
	workerData.setWorkerJob(unit, WorkerData::Idle, nullptr);
}

BWAPI::Unit WorkerManager::getGasWorker(BWAPI::Unit refinery)
{
	UAB_ASSERT(refinery != nullptr, "Refinery was null");

	BWAPI::Unit closestWorker = nullptr;
	double closestDistance = 0;

	for (auto & unit : workerData.getWorkers())
	{
        UAB_ASSERT(unit != nullptr, "Unit was null");

		if (workerData.getWorkerJob(unit) == WorkerData::Minerals && !unit->isCarryingMinerals())
		{
			double distance = unit->getDistance(refinery);
			if (!closestWorker || distance < closestDistance)
			{
				closestWorker = unit;
				closestDistance = distance;
			}
		}
		//if (!closestWorker)
		//{
		//	if (workerData.getWorkerJob(unit) == WorkerData::Minerals)
		//	{
		//		double distance = unit->getDistance(refinery);
		//		if (!closestWorker || distance < closestDistance)
		//		{
		//			closestWorker = unit;
		//			closestDistance = distance;
		//		}
		//	}
		//}
	}

	return closestWorker;
}

 void WorkerManager::setBuildingWorker(BWAPI::Unit worker, Building & b)
 {
     UAB_ASSERT(worker != nullptr, "Worker was null");

     workerData.setWorkerJob(worker, WorkerData::Build, b.type);
 }

// gets a builder for BuildingManager to use
// if setJobAsBuilder is true (default), it will be flagged as a builder unit
// set 'setJobAsBuilder' to false if we just want to see which worker will build a building
BWAPI::Unit WorkerManager::getBuilder(Building & b, bool setJobAsBuilder)
{
	// variables to hold the closest worker of each type to the building
	BWAPI::Unit closestMovingWorker = nullptr;
	BWAPI::Unit closestMiningWorker = nullptr;
	double closestMovingWorkerDistance = 0;
	double closestMiningWorkerDistance = 0;

	// look through each worker that had moved there first
	for (auto & unit : workerData.getWorkers())
	{
        UAB_ASSERT(unit != nullptr, "Unit was null");

        // gas steal building uses scout worker
        if (b.isGasSteal && (workerData.getWorkerJob(unit) == WorkerData::Scout))
        {
            if (setJobAsBuilder)
            {
                workerData.setWorkerJob(unit, WorkerData::Build, b.type);
            }
            return unit;
        }

		// mining worker check
		if (unit->isCompleted() && workerData.getWorkerJob(unit) == WorkerData::Minerals && !unit->isCarryingMinerals() && !unit->isCarryingGas())
		{
			// if it is a new closest distance, set the pointer
			double distance = unit->getDistance(BWAPI::Position(b.finalPosition));
			if (!closestMiningWorker || distance < closestMiningWorkerDistance)
			{
				closestMiningWorker = unit;
				closestMiningWorkerDistance = distance;
			}
		}
		//if (!closestMiningWorker)
		//{
		//	if (unit->isCompleted() && (workerData.getWorkerJob(unit) == WorkerData::Minerals))
		//	{
		//		// if it is a new closest distance, set the pointer
		//		double distance = unit->getDistance(BWAPI::Position(b.finalPosition));
		//		if (!closestMiningWorker || distance < closestMiningWorkerDistance)
		//		{
		//			closestMiningWorker = unit;
		//			closestMiningWorkerDistance = distance;
		//		}
		//	}
		//}

		// moving worker check
		if (unit->isCompleted() && workerData.getWorkerJob(unit) == WorkerData::Move)
		{
			// if it is a new closest distance, set the pointer
			double distance = unit->getDistance(BWAPI::Position(b.finalPosition));
			if (!closestMovingWorker || distance < closestMovingWorkerDistance)
			{
				closestMovingWorker = unit;
				closestMovingWorkerDistance = distance;
			}
		}
	}

	// if we found a moving worker, use it, otherwise using a mining worker
	BWAPI::Unit chosenWorker = closestMovingWorker ? closestMovingWorker : closestMiningWorker;

	// if the worker exists (one may not have been found in rare cases)
	if (chosenWorker && setJobAsBuilder)
	{
		workerData.setWorkerJob(chosenWorker, WorkerData::Build, b.type);
	}

	// return the worker
	return chosenWorker;
}

// sets a worker as a scout
void WorkerManager::setScoutWorker(BWAPI::Unit worker)
{
	UAB_ASSERT(worker != nullptr, "Worker was null");

	workerData.setWorkerJob(worker, WorkerData::Scout, nullptr);
}

// gets a worker which will move to a current location
BWAPI::Unit WorkerManager::getMoveWorker(BWAPI::Position p)
{
	// set up the pointer
	BWAPI::Unit closestWorker = nullptr;
	double closestDistance = 0;

	// for each worker we currently have
	for (auto & unit : workerData.getWorkers())
	{
        UAB_ASSERT(unit != nullptr, "Unit was null");

		// only consider it if it's a mineral worker that is not currently carrying minerals
		if (unit->isCompleted() && workerData.getWorkerJob(unit) == WorkerData::Minerals && !unit->isCarryingMinerals())
		{
			// if it is a new closest distance, set the pointer
			double distance = unit->getDistance(p);
			if (!closestWorker || distance < closestDistance)
			{
				closestWorker = unit;
				closestDistance = distance;
			}
		}
		//we can try again next frame...
		//if (!closestWorker)
		//{
		//	if (unit->isCompleted() && workerData.getWorkerJob(unit) == WorkerData::Minerals)
		//	{
		//		// if it is a new closest distance, set the pointer
		//		double distance = unit->getDistance(p);
		//		if (!closestWorker || distance < closestDistance)
		//		{
		//			closestWorker = unit;
		//			closestDistance = distance;
		//		}
		//	}
		//}
	}

	// return the worker
	return closestWorker;
}

// sets a worker to move to a given location
void WorkerManager::setMoveWorker(int mineralsNeeded, int gasNeeded, BWAPI::Position p)
{
	// set up the pointer
	BWAPI::Unit closestWorker = nullptr;
	double closestDistance = 0;

	// for each worker we currently have
	for (auto & unit : workerData.getWorkers())
	{
        UAB_ASSERT(unit != nullptr, "Unit was null");

		// only consider it if it's a mineral worker
		if (unit->isCompleted() && workerData.getWorkerJob(unit) == WorkerData::Minerals || !unit->isCarryingMinerals())
		{
			// if it is a new closest distance, set the pointer
			double distance = unit->getDistance(p);
			if (!closestWorker || distance < closestDistance)
			{
				closestWorker = unit;
				closestDistance = distance;
			}
		}
		//if (!closestWorker)
		//{
		//	if (unit->isCompleted() && workerData.getWorkerJob(unit) == WorkerData::Minerals)
		//	{
		//		// if it is a new closest distance, set the pointer
		//		double distance = unit->getDistance(p);
		//		if (!closestWorker || distance < closestDistance)
		//		{
		//			closestWorker = unit;
		//			closestDistance = distance;
		//		}
		//	}
		//}
	}

	if (closestWorker)
	{
		//BWAPI::Broodwar->printf("Setting worker job Move for worker %d", closestWorker->getID());
		workerData.setWorkerJob(closestWorker, WorkerData::Move, WorkerMoveData(mineralsNeeded, gasNeeded, p));
	}
	else
	{
		//BWAPI::Broodwar->printf("Error, no worker found");
	}
}

// will we have the required resources by the time a worker can travel a certain distance
bool WorkerManager::willHaveResources(int mineralsRequired, int gasRequired, double distance)
{
	// if we don't require anything, we will have it
	if (mineralsRequired <= 0 && gasRequired <= 0)
	{
		return true;
	}

	// the speed of the worker unit
	double speed = BWAPI::Broodwar->self()->getRace().getWorker().topSpeed();

    UAB_ASSERT(speed > 0, "Speed is negative");

	// how many frames it will take us to move to the building location
	// add a second to account for worker getting stuck. better early than late
	double framesToMove = (distance / speed) + 50;

	// magic numbers to predict income rates
	double mineralRate = getNumMineralWorkers() * 0.045;
	double gasRate     = getNumGasWorkers() * 0.07;

	// calculate if we will have enough by the time the worker gets there
	if (mineralRate * framesToMove >= mineralsRequired &&
		gasRate * framesToMove >= gasRequired)
	{
		return true;
	}
	else
	{
		return false;
	}
}

void WorkerManager::setCombatWorker(BWAPI::Unit worker)
{
	UAB_ASSERT(worker != nullptr, "Worker was null");

	workerData.setWorkerJob(worker, WorkerData::Combat, nullptr);
}

void WorkerManager::onUnitMorph(BWAPI::Unit unit)
{
	UAB_ASSERT(unit != nullptr, "Unit was null");

	// if something morphs into a worker, add it
	if (unit->getType().isWorker() && unit->getPlayer() == BWAPI::Broodwar->self() && unit->getHitPoints() >= 0)
	{
		workerData.addWorker(unit);
	}

	// if something morphs into a building, it was a worker?
	if (unit->getType().isBuilding() && unit->getPlayer() == BWAPI::Broodwar->self() && unit->getPlayer()->getRace() == BWAPI::Races::Zerg)
	{
		//BWAPI::Broodwar->printf("A Drone started building");
		workerData.workerDestroyed(unit);
	}
}

void WorkerManager::onUnitShow(BWAPI::Unit unit)
{
	UAB_ASSERT(unit != nullptr, "Unit was null");

	// add the depot if it exists
	if (unit->getType().isResourceDepot() && unit->getPlayer() == BWAPI::Broodwar->self() && BuildingPlacer::Instance().tileOverlapsBaseLocation(unit->getTilePosition(), BWAPI::UnitTypes::Zerg_Queens_Nest))
	{
		workerData.addDepot(unit);
	}

	// if something morphs into a worker, add it
	if (unit->getType().isWorker() && unit->getPlayer() == BWAPI::Broodwar->self() && unit->getHitPoints() >= 0)
	{
		//BWAPI::Broodwar->printf("A worker was shown %d", unit->getID());
		workerData.addWorker(unit);
	}
}


void WorkerManager::rebalanceWorkers(int idleCount)
{
	// for each worker
	int workersmadeIdle = 0;
	for (auto & worker : workerData.getWorkers())
	{
        UAB_ASSERT(worker != nullptr, "Worker was null");

		if (!workerData.getWorkerJob(worker) == WorkerData::Minerals)
		{
			continue;
		}

		BWAPI::Unit depot = workerData.getWorkerDepot(worker);

		if (depot && workerData.depotIsFull(depot))
		{
			workerData.setWorkerJob(worker, WorkerData::Idle, nullptr);
			workersmadeIdle++;
		}
		else if (!depot)
		{
			workerData.setWorkerJob(worker, WorkerData::Idle, nullptr);
			workersmadeIdle++;
		}
		if (workersmadeIdle >= idleCount) //we don't want to move all workers, just a few!
		{
			break;
		}
	}
}

void WorkerManager::onUnitDestroy(BWAPI::Unit unit) 
{
	UAB_ASSERT(unit != nullptr, "Unit was null");

	if (unit->getType().isResourceDepot() && unit->getPlayer() == BWAPI::Broodwar->self())
	{
		workerData.removeDepot(unit);
	}

	if (unit->getType().isWorker() && unit->getPlayer() == BWAPI::Broodwar->self()) 
	{
		workerData.workerDestroyed(unit);
	}

	if (unit->getType() == BWAPI::UnitTypes::Resource_Mineral_Field)
	{
		rebalanceWorkers(3);
	}
}

void WorkerManager::drawResourceDebugInfo() 
{
    if (!Config::Debug::DrawResourceInfo)
    {
        return;
    }

	for (auto & worker : workerData.getWorkers()) 
    {
        UAB_ASSERT(worker != nullptr, "Worker was null");

		char job = workerData.getJobCode(worker);

		BWAPI::Position pos = worker->getTargetPosition();

		BWAPI::Broodwar->drawTextMap(worker->getPosition().x, worker->getPosition().y - 5, "\x07%c", job);

		BWAPI::Broodwar->drawLineMap(worker->getPosition().x, worker->getPosition().y, pos.x, pos.y, BWAPI::Colors::Cyan);

		BWAPI::Unit depot = workerData.getWorkerDepot(worker);
		if (depot)
		{
			BWAPI::Broodwar->drawLineMap(worker->getPosition().x, worker->getPosition().y, depot->getPosition().x, depot->getPosition().y, BWAPI::Colors::Orange);
		}
	}
}

void WorkerManager::drawWorkerInformation(int x, int y) 
{
    if (!Config::Debug::DrawWorkerInfo)
    {
        return;
    }

	BWAPI::Broodwar->drawTextScreen(x, y, "\x04 Workers %d", workerData.getNumMineralWorkers());
	BWAPI::Broodwar->drawTextScreen(x, y+20, "\x04 UnitID");
	BWAPI::Broodwar->drawTextScreen(x+50, y+20, "\x04 State");

	int yspace = 0;

	for (auto & unit : workerData.getWorkers())
	{
        UAB_ASSERT(unit != nullptr, "Worker was null");

		BWAPI::Broodwar->drawTextScreen(x, y+40+((yspace)*10), "\x03 %d", unit->getID());
		BWAPI::Broodwar->drawTextScreen(x+50, y+40+((yspace++)*10), "\x03 %c", workerData.getJobCode(unit));
	}
}

bool WorkerManager::isFree(BWAPI::Unit worker)
{
    UAB_ASSERT(worker != nullptr, "Worker was null");

	return workerData.getWorkerJob(worker) == WorkerData::Minerals || workerData.getWorkerJob(worker) == WorkerData::Idle;
}

bool WorkerManager::isWorkerScout(BWAPI::Unit worker)
{
    UAB_ASSERT(worker != nullptr, "Worker was null");

	return (workerData.getWorkerJob(worker) == WorkerData::Scout);
}

bool WorkerManager::isBuilder(BWAPI::Unit worker)
{
    UAB_ASSERT(worker != nullptr, "Worker was null");

	return (workerData.getWorkerJob(worker) == WorkerData::Build);
}

int WorkerManager::getNumMineralWorkers() 
{
	return workerData.getNumMineralWorkers();	
}

int WorkerManager::getNumIdleWorkers() 
{
	return workerData.getNumIdleWorkers();	
}

int WorkerManager::getNumGasWorkers() 
{
	return workerData.getNumGasWorkers();
}

bool WorkerManager::isDepot(BWAPI::Unit depot, bool mustBeCompleted)
{
	bool isDepot = false;
	if (depot->getType().isResourceDepot() && (depot->isCompleted() || depot->getType() == BWAPI::UnitTypes::Zerg_Lair || depot->getType() == BWAPI::UnitTypes::Zerg_Hive || mustBeCompleted == false))
	{
		if (BuildingPlacer::Instance().tileOverlapsBaseLocation(depot->getTilePosition(), BWAPI::UnitTypes::Zerg_Queens_Nest))
		{
			isDepot = true;
		}
	}
	return isDepot;
}