#include "ScoutManager.h"
#include "ProductionManager.h"

using namespace UAlbertaBot;

ScoutManager::ScoutManager() 
    : _workerScout(nullptr)
    , _numWorkerScouts(0)
    , _scoutUnderAttack(false)
    , _gasStealStatus("None")
    , _scoutStatus("None")
    , _didGasSteal(false)
    , _gasStealFinished(false)
    , _currentRegionVertexIndex(-1)
    , _previousScoutHP(0)
{
}

ScoutManager & ScoutManager::Instance() 
{
	static ScoutManager instance;
	return instance;
}

void ScoutManager::update()
{
    if (!Config::Modules::UsingScoutManager)
    {
        return;
    }
	moveScouts();
    drawScoutInformation(200, 320);
}

void ScoutManager::setWorkerScout(BWAPI::Unit unit)
{
    // if we have a previous worker scout, release it back to the worker manager
    //if (_workerScout)
    //{
    //    WorkerManager::Instance().finishedWithWorker(_workerScout);
    //}

    _workerScout = unit;
    WorkerManager::Instance().setScoutWorker(_workerScout);
}

void ScoutManager::drawScoutInformation(int x, int y)
{
    if (!Config::Debug::DrawScoutInfo)
    {
        return;
    }

    BWAPI::Broodwar->drawTextScreen(x, y, "ScoutInfo: %s", _scoutStatus.c_str());
    BWAPI::Broodwar->drawTextScreen(x, y+10, "GasSteal: %s", _gasStealStatus.c_str());
    for (size_t i(0); i < _enemyRegionVertices.size(); ++i)
    {
        BWAPI::Broodwar->drawCircleMap(_enemyRegionVertices[i], 4, BWAPI::Colors::Green, false);
        BWAPI::Broodwar->drawTextMap(_enemyRegionVertices[i], "%d", i);
    }
}

void ScoutManager::moveScouts()
{
	if (!_workerScout || !_workerScout->exists() || !(_workerScout->getHitPoints() > 0))
	{
		return;
	}

    int scoutHP = _workerScout->getHitPoints() + _workerScout->getShields();
    
	// get the enemy base location, if we have one
	BWAPI::TilePosition enemyBaseLocation = InformationManager::Instance().getMainBaseLocation(BWAPI::Broodwar->enemy());

    int scoutDistanceThreshold = 30;

    if (_workerScout->isCarryingGas())
    {
        BWAPI::Broodwar->drawCircleMap(_workerScout->getPosition(), 10, BWAPI::Colors::Purple, true);
    }

    // if we initiated a gas steal and the worker isn't idle, 
    bool finishedConstructingGasSteal = _workerScout->isIdle() || _workerScout->isCarryingGas();
    if (!_gasStealFinished && _didGasSteal && !finishedConstructingGasSteal)
    {
        return;
    }
    // check to see if the gas steal is completed
    else if (_didGasSteal && finishedConstructingGasSteal)
    {
        _gasStealFinished = true;
    }
    
	if (_workerScout && enemyBaseLocation != BWAPI::TilePositions::None)
	{
		//BWAPI::Broodwar->drawLineMap(_workerScout->getPosition(), BWAPI::Position(enemyBaseLocation), BWAPI::Colors::Red);
		//BWAPI::Broodwar->drawTextMap(_workerScout->getPosition(), "%d/%d", scoutHP, _workerScout->getType().maxHitPoints());
		if (scoutHP < _workerScout->getType().maxHitPoints())
		{
			Micro::SmartMove(_workerScout, BWAPI::Position(InformationManager::Instance().getMainBaseLocation(BWAPI::Broodwar->self())));
			//BWAPI::Broodwar->drawLineMap(_workerScout->getPosition(), BWAPI::Position(InformationManager::Instance().getMainBaseLocation(BWAPI::Broodwar->self())), BWAPI::Colors::Yellow);
			BWAPI::Unit closestHatchery = _workerScout->getClosestUnit(!BWAPI::Filter::IsEnemy&&BWAPI::Filter::IsResourceDepot);
			if (closestHatchery && closestHatchery->getDistance(_workerScout->getPosition()) < 300)
			{
				WorkerManager::Instance().finishedWithWorker(_workerScout);
				_workerScout = nullptr;
				return;
			}
			return;
		}
		else
		{
			//int scoutDistanceToEnemy = MapTools::Instance().getGroundDistance(_workerScout->getPosition(), BWAPI::Position(*enemyBaseLocation));
			int scoutDistanceToEnemy = _workerScout->getPosition().getDistance(BWAPI::Position(enemyBaseLocation));
			bool scoutInRangeOfenemy = scoutDistanceToEnemy <= scoutDistanceThreshold;

			// we only care if the scout is under attack within the enemy region
			// this ignores if their scout worker attacks it on the way to their base
			if (scoutHP < _previousScoutHP)
			{
				_scoutUnderAttack = true;
			}

			// if the scout is in the enemy region
			if (scoutInRangeOfenemy)
			{
				// get the closest enemy worker
				BWAPI::Unit closestWorker = closestEnemyWorker();

				// if the worker scout is not under attack
				if (!_scoutUnderAttack)
				{
					// if there is a worker nearby, harass it
					if (Config::Strategy::ScoutHarassEnemy && (!Config::Strategy::GasStealWithScout || _gasStealFinished) && closestWorker && (_workerScout->getDistance(closestWorker) < 800))
					{
						_scoutStatus = "Harass enemy worker";
						_currentRegionVertexIndex = -1;
						Micro::SmartAttackUnit(_workerScout, closestWorker);
					}
					// otherwise keep moving to the enemy region
					else
					{
						_scoutStatus = "Following perimeter";
						followPerimeter();
					}

				}
				// if the worker scout is under attack
				else
				{
					_scoutStatus = "Under attack inside, fleeing";
					followPerimeter();
				}
			}
			// if the scout is not in the enemy region
			else if (_scoutUnderAttack)
			{
				_scoutStatus = "Under attack inside, fleeing";

				followPerimeter();
			}
			else
			{
				_scoutStatus = "Enemy region known, going there";

				// move to the enemy region
				followPerimeter();
			}
		}
	}

	// for each start location in the level
	if (!enemyBaseLocation)
	{
        _scoutStatus = "Enemy base unknown, exploring";

		for (BWAPI::TilePosition startLocation : BWEM::Map::Instance().StartingLocations()) 
		{
			// if we haven't explored it yet
			if (!BWAPI::Broodwar->isExplored(startLocation)) 
			{
				// assign a zergling to go scout it
				Micro::SmartMove(_workerScout, BWAPI::Position(startLocation));			
				return;
			}
		}
	}

    _previousScoutHP = scoutHP;
}

void ScoutManager::followPerimeter()
{
	Micro::SmartOviScout(_workerScout, BWAPI::Position(InformationManager::Instance().getMainBaseLocation(BWAPI::Broodwar->enemy())), 200, true);
	//Micro::SmartMove(_workerScout, fleeTo);
}

BWAPI::Unit ScoutManager::closestEnemyWorker()
{
	BWAPI::Unit enemyWorker = nullptr;
	double maxDist = 0;

	for (auto & unit : BWAPI::Broodwar->enemy()->getUnits())
	{
		if (unit->getType().isWorker() && unit->isConstructing())
		{
			return unit;
		}
	}
	return enemyWorker;
}

bool ScoutManager::enemyWorkerInRadius()
{
	for (auto & unit : BWAPI::Broodwar->enemy()->getUnits())
	{
		if (unit->getType().isWorker() && (unit->getDistance(_workerScout) < 300))
		{
			return true;
		}
	}

	return false;
}

bool ScoutManager::immediateThreat()
{
	BWAPI::Unitset enemyAttackingWorkers;
	for (auto & unit : BWAPI::Broodwar->enemy()->getUnits())
	{
		if (unit->getType().isWorker() && unit->isAttacking())
		{
			enemyAttackingWorkers.insert(unit);
		}
	}
	
	if (_workerScout->isUnderAttack())
	{
		return true;
	}

	for (auto & unit : BWAPI::Broodwar->enemy()->getUnits())
	{
		double dist = unit->getDistance(_workerScout);
		double range = unit->getType().groundWeapon().maxRange();

		if (unit->getType().canAttack() && !unit->getType().isWorker() && (dist <= range + 32))
		{
			return true;
		}
	}

	return false;
}

int ScoutManager::getClosestVertexIndex(BWAPI::Unit unit)
{
    int closestIndex = -1;
    double closestDistance = 10000000;

    for (size_t i(0); i < _enemyRegionVertices.size(); ++i)
    {
        double dist = unit->getDistance(_enemyRegionVertices[i]);
        if (dist < closestDistance)
        {
            closestDistance = dist;
            closestIndex = i;
        }
    }

    return closestIndex;
}

BWAPI::Position ScoutManager::getFleePosition()
{
    UAB_ASSERT_WARNING(!_enemyRegionVertices.empty(), "We should have an enemy region vertices if we are fleeing");
    
    BWAPI::TilePosition enemyBaseLocation = InformationManager::Instance().getMainBaseLocation(BWAPI::Broodwar->enemy());

    // if this is the first flee, we will not have a previous perimeter index
    if (_currentRegionVertexIndex == -1)
    {
        // so return the closest position in the polygon
        int closestPolygonIndex = getClosestVertexIndex(_workerScout);

        UAB_ASSERT_WARNING(closestPolygonIndex != -1, "Couldn't find a closest vertex");

        if (closestPolygonIndex == -1)
        {
            return BWAPI::Position(BWAPI::Broodwar->self()->getStartLocation());
        }
        else
        {
            // set the current index so we know how to iterate if we are still fleeing later
            _currentRegionVertexIndex = closestPolygonIndex;
            return _enemyRegionVertices[closestPolygonIndex];
        }
    }
    // if we are still fleeing from the previous frame, get the next location if we are close enough
    else
    {
        double distanceFromCurrentVertex = _enemyRegionVertices[_currentRegionVertexIndex].getDistance(_workerScout->getPosition());

        // keep going to the next vertex in the perimeter until we get to one we're far enough from to issue another move command
        while (distanceFromCurrentVertex < 128)
        {
            _currentRegionVertexIndex = (_currentRegionVertexIndex + 1) % _enemyRegionVertices.size();

            distanceFromCurrentVertex = _enemyRegionVertices[_currentRegionVertexIndex].getDistance(_workerScout->getPosition());
        }

        return _enemyRegionVertices[_currentRegionVertexIndex];
    }

}
