#include "MapTools.h"
#include "InformationManager.h"
#include "BuildingManager.h"
#include "UnitUtil.h"
#include "MapGrid.h"

using namespace UAlbertaBot;

MapTools & MapTools::Instance()
{
    static MapTools instance;
    return instance;
}

// constructor for MapTools
MapTools::MapTools()
    : _rows(BWAPI::Broodwar->mapHeight())
    , _cols(BWAPI::Broodwar->mapWidth())
{
    _map    = std::vector<bool>(_rows*_cols,false);
    _units  = std::vector<bool>(_rows*_cols,false);
    _fringe = std::vector<int>(_rows*_cols,0);

    setBWAPIMapData();
}

// return the index of the 1D array from (row,col)
inline int MapTools::getIndex(int row,int col)
{
    return row * _cols + col;
}

bool MapTools::unexplored(DistanceMap & dmap,const int index) const
{
    return (index != -1) && dmap[index] == -1 && _map[index];
}

// resets the distance and fringe vectors, call before each search
void MapTools::reset()
{
    std::fill(_fringe.begin(),_fringe.end(),0);
}

// reads in the map data from bwapi and stores it in our map format
void MapTools::setBWAPIMapData()
{
    // for each row and column
    for (int r(0); r < _rows; ++r)
    {
        for (int c(0); c < _cols; ++c)
        {
            bool clear = true;

            // check each walk tile within this TilePosition
            for (int i=0; i<4; ++i)
            {
                for (int j=0; j<4; ++j)
                {
                    if (!BWAPI::Broodwar->isWalkable(c*4 + i,r*4 + j))
                    {
                        clear = false;
                        break;
                    }

                    if (clear)
                    {
                        break;
                    }
                }
            }

            // set the map as binary clear or not
            _map[getIndex(r,c)] = clear;
        }
    }
}

void MapTools::resetFringe()
{
    std::fill(_fringe.begin(),_fringe.end(),0);
}

int MapTools::getGroundDistance(BWAPI::Position origin,BWAPI::Position destination)
{
	// if we have too many maps, reset our stored maps in case we run out of memory
	if (_allMaps.size() > 20)
	{
		_allMaps.clear();

		BWAPI::Broodwar->printf("Cleared stored distance map cache");
	}

	// if we haven't yet computed the distance map to the destination
	if (_allMaps.find(destination) == _allMaps.end())
	{
		// if we have computed the opposite direction, we can use that too
		if (_allMaps.find(origin) != _allMaps.end())
		{
			return _allMaps[origin][destination];
		}

		// add the map and compute it
		_allMaps.insert(std::pair<BWAPI::Position, DistanceMap>(destination, DistanceMap()));
		computeDistance(_allMaps[destination], destination);
	}

	double totalLength = 0;
	int iCurr = 0;
	const BWEM::ChokePoint* Previous = nullptr;
	for (auto choke : BWEM::Map::Instance().GetPath(origin, destination))
	{
		if (iCurr == 0)
		{
			totalLength += origin.getDistance(BWAPI::Position(choke->Center()));
		}
		else
		{
			totalLength += BWAPI::Position(Previous->Center()).getDistance(BWAPI::Position(choke->Center()));
		}
		Previous = choke;
		++iCurr;
	}
	if (iCurr == 0)
	{
		totalLength += origin.getDistance(destination);
	}
	if (Previous)
	{
		totalLength += destination.getDistance(BWAPI::Position(Previous->Center()));
	}
	return std::round(totalLength);
}

// computes walk distance from Position P to all other points on the map
void MapTools::computeDistance(DistanceMap & dmap,const BWAPI::Position p)
{
    search(dmap,p.y / 32,p.x / 32);
}

// does the dynamic programming search
void MapTools::search(DistanceMap & dmap,const int sR,const int sC)
{
    // reset the internal variables
    resetFringe();

    // set the starting position for this search
    dmap.setStartPosition(sR,sC);

    // set the distance of the start cell to zero
    dmap[getIndex(sR,sC)] = 0;

    // set the fringe variables accordingly
    int fringeSize(1);
    int fringeIndex(0);
    _fringe[0] = getIndex(sR,sC);
    dmap.addSorted(getTilePosition(_fringe[0]));

    // temporary variables used in search loop
    int currentIndex,nextIndex;
    int newDist;

    // the size of the map
    int size = _rows*_cols;

    // while we still have things left to expand
    while (fringeIndex < fringeSize)
    {
        // grab the current index to expand from the fringe
        currentIndex = _fringe[fringeIndex++];
        newDist = dmap[currentIndex] + 1;

        // search up
        nextIndex = (currentIndex > _cols) ? (currentIndex - _cols) : -1;
        if (unexplored(dmap,nextIndex))
        {
            // set the distance based on distance to current cell
            dmap.setDistance(nextIndex,newDist);
            dmap.setMoveTo(nextIndex,'D');
            dmap.addSorted(getTilePosition(nextIndex));

            // put it in the fringe
            _fringe[fringeSize++] = nextIndex;
        }

        // search down
        nextIndex = (currentIndex + _cols < size) ? (currentIndex + _cols) : -1;
        if (unexplored(dmap,nextIndex))
        {
            // set the distance based on distance to current cell
            dmap.setDistance(nextIndex,newDist);
            dmap.setMoveTo(nextIndex,'U');
            dmap.addSorted(getTilePosition(nextIndex));

            // put it in the fringe
            _fringe[fringeSize++] = nextIndex;
        }

        // search left
        nextIndex = (currentIndex % _cols > 0) ? (currentIndex - 1) : -1;
        if (unexplored(dmap,nextIndex))
        {
            // set the distance based on distance to current cell
            dmap.setDistance(nextIndex,newDist);
            dmap.setMoveTo(nextIndex,'R');
            dmap.addSorted(getTilePosition(nextIndex));

            // put it in the fringe
            _fringe[fringeSize++] = nextIndex;
        }

        // search right
        nextIndex = (currentIndex % _cols < _cols - 1) ? (currentIndex + 1) : -1;
        if (unexplored(dmap,nextIndex))
        {
            // set the distance based on distance to current cell
            dmap.setDistance(nextIndex,newDist);
            dmap.setMoveTo(nextIndex,'L');
            dmap.addSorted(getTilePosition(nextIndex));

            // put it in the fringe
            _fringe[fringeSize++] = nextIndex;
        }
    }
}

const std::vector<BWAPI::TilePosition> & MapTools::getClosestTilesTo(BWAPI::Position pos)
{
    // make sure the distance map is calculated with pos as a destination
	int a = getGroundDistance(pos, pos);
	//int a = getGroundDistance(BWAPI::Position(BWAPI::Broodwar->self()->getStartLocation()),pos);

    return _allMaps[pos].getSortedTiles();
}

BWAPI::TilePosition MapTools::getTilePosition(int index)
{
    return BWAPI::TilePosition(index % _cols,index / _cols);
}

BWAPI::TilePosition MapTools::getNextExpansion()
{
    return getNextExpansion(BWAPI::Broodwar->self());
}

void MapTools::drawHomeDistanceMap()
{
    BWAPI::Position homePosition = BWAPI::Position(BWAPI::Broodwar->self()->getStartLocation());
    for (int x = 0; x < BWAPI::Broodwar->mapWidth(); ++x)
    {
        for (int y = 0; y < BWAPI::Broodwar->mapHeight(); ++y)
        {
            BWAPI::Position pos(x*32, y*32);

            int dist = getGroundDistance(pos, homePosition);

            BWAPI::Broodwar->drawTextMap(pos + BWAPI::Position(16,16), "%d", dist);
        }
    }
}

BWAPI::TilePosition MapTools::getNextExpansion(BWAPI::Player player)
{
    BWAPI::TilePosition closest = BWAPI::TilePositions::None;
    double minDistance = 100000;

    BWAPI::TilePosition homeTile = player->getStartLocation();

    // for each base location
	for(BWEM::Area area : BWEM::Map::Instance().Areas())
	{
		for (BWEM::Base base : area.Bases())
		{
			// if the base has gas
			if(!(base.Location() == player->getStartLocation()))
			{
				// get the tile position of the base
				BWAPI::TilePosition tile = base.Location();
				bool buildingInTheWay = false;
				bool enemyNearby = false;

				for (int x = 0; x < BWAPI::Broodwar->self()->getRace().getCenter().tileWidth(); ++x)
				{
					for (int y = 0; y < BWAPI::Broodwar->self()->getRace().getCenter().tileHeight(); ++y)
					{
						BWAPI::TilePosition tp(tile.x + x, tile.y + y);

						for (auto & unit : BWAPI::Broodwar->getUnitsOnTile(tp))
						{
							if (unit->getType().isBuilding() && !unit->isFlying())
							{
								buildingInTheWay = true;
								break;
							}
						}
					}
				}
				for (auto unit : InformationManager::Instance().getUnitData(BWAPI::Broodwar->enemy()).getUnits())
				{
					if (unit.second.lastPosition.getDistance(base.Center()) < 400 && !unit.second.type.isWorker())
					{
						enemyNearby = true;
						break;
					}
				}
				if (buildingInTheWay || enemyNearby)
				{
					continue;
				}

				// the base's distance from our main nexus
				BWAPI::Position myBasePosition(player->getStartLocation());
				BWAPI::Position thisTile = BWAPI::Position(tile);
				double distanceFromHome = MapTools::Instance().getGroundDistance(thisTile, myBasePosition);
				if (base.Geysers().empty()) //Min-Only less viable
				{
					distanceFromHome *= 2;
				}
				distanceFromHome /= (base.Minerals().size() + 1); //when there are few or no minerals the base should also be considered less valuable e.g. on Challenger
				if (WorkerManager::Instance().workerData.getNumDepots() >= 2)
				{
					distanceFromHome /= thisTile.getDistance(getEnemyCenter());
				}
				//BWAPI::Broodwar->drawCircleMap(base.Center(), 50, BWAPI::Colors::White);
				//BWAPI::Broodwar->drawTextMap(base.Center(), "%d / %d", (int)distanceFromHome, (int)thisTile.getDistance(getEnemyCenter()));

				// if it is not connected, continue
				if (BWEM::Map::Instance().GetPath(BWAPI::Position(homeTile), thisTile).empty() || distanceFromHome < 0)
				{
					continue;
				}

				if (closest == BWAPI::TilePositions::None || distanceFromHome < minDistance)
				{
					closest = BWAPI::TilePosition(base.Location());
					minDistance = distanceFromHome;
				}
			}
		}
    }
	return closest;
}

std::vector<MapTools::ScoutLocation> MapTools::getScoutLocations()
{
	std::vector<ScoutLocation> scoutPositions;

	//BWAPI::Position center = BWAPI::Position(BWAPI::Broodwar->mapWidth() * 16, BWAPI::Broodwar->mapHeight() * 16);
	//scoutPositions.push_back(center);
	//BWAPI::Position forwardScout = (center + getBaseCenter()) / 2;
	//scoutPositions.push_back(forwardScout);
	int OverlordsRemainingForUnits = std::ceil(UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Overlord, true) / 10.0);
	for each (auto ui in InformationManager::Instance().getUnitInfo(BWAPI::Broodwar->self()))
	{
		if (OverlordsRemainingForUnits <= 0)
		{
			break;
		}
		if (ui.second.unit->canAttack() && !ui.second.type.isWorker())
		{
			ScoutLocation sl;
			sl.pos = ui.second.lastPosition;
			sl.forceLowAvoidDistance = true;
			sl.onlyAvoidAA = true;
			scoutPositions.push_back(sl);
			--OverlordsRemainingForUnits;
		}
	}
	for each (auto ui in InformationManager::Instance().getUnitInfo(BWAPI::Broodwar->enemy()))
	{
		if (ui.second.unit->isBurrowed() || ui.second.unit->isCloaked())
		{
			ScoutLocation sl;
			sl.pos = ui.second.lastPosition;
			sl.forceLowAvoidDistance = false;
			sl.onlyAvoidAA = true;
			scoutPositions.push_back(sl);
		}
	}
	if (BWAPI::Broodwar->enemy()->getStartLocation().isValid())
	{
		ScoutLocation sl;
		sl.pos = BWAPI::Position(BWAPI::Broodwar->enemy()->getStartLocation());
		sl.forceLowAvoidDistance = false;
		scoutPositions.push_back(sl);
	}
	for each(auto base in BWEM::Map::Instance().StartingLocations())
	{
		bool insert = true;
		for each(auto element in scoutPositions)
		{
			if (element.pos.getDistance(BWAPI::Position(base)) < 180)
			{
				insert = false;
			}
		}
		if(insert && base != BWAPI::Broodwar->self()->getStartLocation())
		{
			ScoutLocation sl;
			sl.pos = BWAPI::Position(base);
			sl.forceLowAvoidDistance = false;
			sl.onlyAvoidAA = false;
			scoutPositions.push_back(sl);
		}
	}
	for (BWEM::Area area : BWEM::Map::Instance().Areas())
	{
		for (BWEM::Base base : area.Bases())
		{
			bool insert = true;
			for each(auto element in scoutPositions)
			{
				if (element.pos.getDistance(base.Center()) < 180)
				{
					insert = false;
				}
			}
			if (insert)
			{
				ScoutLocation sl;
				sl.pos = base.Center();
				sl.forceLowAvoidDistance = false;
				sl.onlyAvoidAA = false;
				scoutPositions.push_back(sl);
			}
		}
	}
	while (scoutPositions.size() < UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Overlord, true))
	{
		for (auto & detectorUnit : BWAPI::Broodwar->self()->getUnits())
		{
			if (detectorUnit->getType().isFlyer() && detectorUnit->getType().isDetector())
			{
				bool insert = true;
				ScoutLocation sl;
				sl.pos = MapGrid::Instance().getClosestUnExplored(detectorUnit->getPosition());
				sl.forceLowAvoidDistance = false;
				sl.onlyAvoidAA = false;
				scoutPositions.push_back(sl);
			}
			if (scoutPositions.size() >= UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Overlord, true))
			{
				break;
			}
		}
	}
	return scoutPositions;
}

void MapTools::parseMap()
{
    BWAPI::Broodwar->printf("Parsing Map Information");
    std::ofstream mapFile;
    std::string file = "c:\\scmaps\\" + BWAPI::Broodwar->mapName() + ".txt";
    mapFile.open(file.c_str());

    mapFile << BWAPI::Broodwar->mapWidth()*4 << "\n";
    mapFile << BWAPI::Broodwar->mapHeight()*4 << "\n";

    for (int j=0; j<BWAPI::Broodwar->mapHeight()*4; j++) 
    {
        for (int i=0; i<BWAPI::Broodwar->mapWidth()*4; i++) 
        {
            if (BWAPI::Broodwar->isWalkable(i,j)) 
            {
                mapFile << "0";
            }
            else 
            {
                mapFile << "1";
            }
        }

        mapFile << "\n";
    }

    BWAPI::Broodwar->printf(file.c_str());

    mapFile.close();
}

std::map<BWAPI::TilePosition, int> MapTools::getConcaveTileMap(BWAPI::Position concaveCenter, BWAPI::Position concaveEdge, int ringWidth)
{
	std::map<BWAPI::TilePosition, int> concaveTiles;
	int tileDistance = int(std::sqrt(std::pow(BWAPI::TilePosition(concaveCenter).x - BWAPI::TilePosition(concaveEdge).x, 2) + std::pow(BWAPI::TilePosition(concaveCenter).y - BWAPI::TilePosition(concaveEdge).y, 2)));
	for (int x = 0; x < BWAPI::Broodwar->mapWidth(); x++)
	{
		for (int y = 0; y < BWAPI::Broodwar->mapHeight(); y++)
		{
			int dist = int(std::sqrt(std::pow(BWAPI::TilePosition(concaveCenter).x - x, 2) + std::pow(BWAPI::TilePosition(concaveCenter).y - y, 2)));
			if (dist == tileDistance)
			{
				BWAPI::TilePosition tileToAdd = BWAPI::TilePosition(x, y);
				concaveTiles[tileToAdd] = 0;
			}
		}
	}
	return concaveTiles;
}

BWAPI::Position MapTools::getBaseCenter()
{
	int spotsWithBuildings = 0;
	int sumX = 0;
	int sumY = 0;
	for (BWEM::Area area : BWEM::Map::Instance().Areas())
	{
		for (BWEM::Base base : area.Bases())
		{
			for (auto & unit : BWAPI::Broodwar->self()->getUnits())
			{
				if (unit->getType().isBuilding() && unit->getDistance(base.Center()) < 200)
				{
					sumX += base.Center().x;
					sumY += base.Center().y;
					spotsWithBuildings++;
					break;
				}
			}
		}
	}
	BWAPI::Position pos = BWAPI::Position(0, 0);
	if (spotsWithBuildings != 0)
	{
		pos = BWAPI::Position(sumX / spotsWithBuildings, sumY / spotsWithBuildings);
	}
	return pos;
}

BWAPI::Position MapTools::getEnemyCenter()
{
	int spotsWithEnemies = 0;
	int sumX = 0;
	int sumY = 0;
	for each (auto ui in InformationManager::Instance().getUnitInfo(BWAPI::Broodwar->enemy()))
	{
		double value = (ui.second.type.mineralPrice() + ui.second.type.gasPrice()) / (ui.second.type.isTwoUnitsInOneEgg() ? 2 : 1);
		sumX += ui.second.lastPosition.x * value;
		sumY += ui.second.lastPosition.y * value;
		spotsWithEnemies += value;
	}
	BWAPI::Position pos = BWAPI::Position(0, 0);
	if (spotsWithEnemies != 0)
	{
		pos = BWAPI::Position(sumX / spotsWithEnemies, sumY / spotsWithEnemies);
	}
	else
	{
		pos.x = BWAPI::Broodwar->mapWidth() * 32;
		pos.y = BWAPI::Broodwar->mapHeight() * 32;
	}
	return pos;
}

BWAPI::Position MapTools::getConcaveChoke(BWAPI::Position nearByTo, BWAPI::Position closerTo, BWAPI::Position thanThis)
{
	int closestChokeDistance = 10000;
	BWAPI::Position pos = nearByTo;
	for (BWEM::Area area : BWEM::Map::Instance().Areas())
	{
		for (const BWEM::ChokePoint* choke : area.ChokePoints())
		{
			if (BWAPI::Position(choke->Center()).getDistance(closerTo) < BWAPI::Position(choke->Center()).getDistance(thanThis))
			{
				if (BWAPI::Position(choke->Center()).getDistance(nearByTo) < closestChokeDistance)
				{
					closestChokeDistance = BWAPI::Position(choke->Center()).getDistance(nearByTo);
					pos = BWAPI::Position(choke->Center());
				}
			}
		}
	}
	return pos;
}

BWAPI::Position MapTools::getClosestSunkenPosition()
{
	BWAPI::Position sunkenLocation = MapTools::Instance().getBaseCenter();

	int closestSunkenDistance = 10000;
	BWAPI::Unit closestSunken = nullptr;
	for each (auto sunken in BWAPI::Broodwar->self()->getUnits())
	{
		if (sunken->getType() != BWAPI::UnitTypes::Zerg_Sunken_Colony)
		{
			continue;
		}
		int distance = sunkenLocation.getDistance(sunken->getPosition());
		{
			if (distance < closestSunkenDistance)
			{
				closestSunken = sunken;
				closestSunkenDistance = distance;
			}
		}
		if (sunken->isAttacking()) //If a sunken is already attacking, then it doesn't matter how close it is, it is the one we want to support and near which we attack
		{
			closestSunken = sunken;
			closestSunkenDistance = distance;
			break;
		}
	}
	if (closestSunken != nullptr)
	{
		sunkenLocation = closestSunken->getPosition();
	}
	return sunkenLocation;
}