#include "Common.h"
#include "GameCommander.h"
#include "UnitUtil.h"
#include <random>

using namespace UAlbertaBot;

GameCommander::GameCommander()
	: _initialScoutSet(false)
{
	srand(time(NULL));
	strategyMap[0] = "Freestyle";
	strategyMap[1] = "MutaAllIn";
	strategyMap[2] = "StoicLingMuta";
	strategyMap[3] = "HydraAllIn";
	strategyMap[4] = "StoicLingHydra";
	strategyMap[5] = "StoicLingLurker";
	strategyMap[6] = "StoicHydraLurker";
	strategyMap[7] = "StoicLingHydraLurker";
	strategyMap[8] = "StoicLingHydraUltra";
	strategyMap[9] = "StoicHydra";
	strategyMap[10] = "StoicLurker";
	strategyMap[11] = "StoicMuta";
	strategyMap[12] = "StoicLingMutaGuardian";
	strategyMap[13] = "StoicLingHydraGuardian";
	strategyMap[14] = "StoicHydraGuardian";
	strategyMap[15] = "Adaptive";
	strategyMap[16] = "Randomized";
	strategyMap[17] = "SmartUnlock";
}

void GameCommander::update()
{
	_timerManager.startTimer(TimerManager::All);
	scorePerFrame = (double)StrategyManager::Instance().getScore(BWAPI::Broodwar->self()) * 24 * 60 / (double)(BWAPI::Broodwar->getFrameCount() + 600 * 60);
	StrategyManager::Instance().strategy = strategy;
	if (scorePerFrame > highestScorePerFrame)
	{
		highestScorePerFrame = scorePerFrame;
		highestScoreFrame = BWAPI::Broodwar->getFrameCount();
	}

	if (BWAPI::Broodwar->self()->supplyUsed() < 1 && BWAPI::Broodwar->self()->minerals() < 50
		|| scorePerFrame < highestScorePerFrame / 2)
	{
		BWAPI::Broodwar->sendText("Good Game, well played, %s", BWAPI::Broodwar->enemy()->getName().c_str());
		BWAPI::Broodwar->leaveGame();
		return;
	}

	// populate the unit vectors we will pass into various managers
	handleUnitAssignments();

	// utility managers
	_timerManager.startTimer(TimerManager::InformationManager);
	InformationManager::Instance().update();
	_timerManager.stopTimer(TimerManager::InformationManager);

	enemyUnitCost = 0;
	enemyAirCost = 0;
	double enemyGroundCost = 0;
	enemyAntiAirCost = 0;
	bool lurkersStillGood = true;
	bool allowMutaInMutaLessBuild = false;
	double enemyMobileDetectorCost = 0;
	double enemyAntiGroundCost = 0;
	double enemyDefenseBuildingCost = 0;
	double enemyBuildingCost = 0;
	double enemyFlyingAntiAir = 0;
	double enemyFlyingBuildingCost = 0;
	double enemySmallCost = 0;
	double enemyMediumCost = 0;
	double enemyLargeCost = 0;
	double enemyGroundAoE = 0;
	double enemyAirAoE = 0;
	double enemyAntiMeleeAoE = 0;
	double enemyMarineGoonCost = 0;
	bool hydraUnlock = false;
	bool lurkerUnlock = false;
	bool mutaUnlock = false;
	bool ultraUnlock = false;
	bool guardianUnlock = false;

	double myTotalCost = 0;
	double myHydraCost = 0;
	double myLurkerCost = 0;
	double myMutaCost = 0;
	double myLingCost = 0;
	double myUltraCost = 0;
	double myGuardCost = 0;
	double myUnitCost = 0;
	int baseCount = 0;
	for (auto enemy : InformationManager::Instance().getUnitData(BWAPI::Broodwar->enemy()).getUnits())
	{
		if (enemy.second.type == BWAPI::UnitTypes::Terran_Marine
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Dragoon)
		{
			enemyMarineGoonCost += enemy.second.type.mineralPrice() + enemy.second.type.gasPrice();
		}
		if (enemy.second.type.size() == BWAPI::UnitSizeTypes::Small)
		{
			enemySmallCost += enemy.second.type.mineralPrice() + enemy.second.type.gasPrice();
		}
		if (enemy.second.type.size() == BWAPI::UnitSizeTypes::Medium)
		{
			enemyMediumCost += enemy.second.type.mineralPrice() + enemy.second.type.gasPrice();
		}
		if (enemy.second.type.size() == BWAPI::UnitSizeTypes::Medium)
		{
			enemyLargeCost += enemy.second.type.mineralPrice() + enemy.second.type.gasPrice();
		}
		if (enemy.second.type == BWAPI::UnitTypes::Protoss_High_Templar
			|| enemy.second.type == BWAPI::UnitTypes::Zerg_Defiler)
		{
			enemyGroundAoE += enemy.second.type.mineralPrice() + enemy.second.type.gasPrice();
			enemyAirAoE += enemy.second.type.mineralPrice() + enemy.second.type.gasPrice();
		}
		if (enemy.second.type == BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Vulture
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Vulture_Spider_Mine
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Reaver)
		{
			enemyGroundAoE += enemy.second.type.mineralPrice() + enemy.second.type.gasPrice();
			lurkersStillGood = false;
		}
		if (enemy.second.type == BWAPI::UnitTypes::Terran_Firebat
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Archon
			|| enemy.second.type == BWAPI::UnitTypes::Zerg_Lurker
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Vulture)
		{
			enemyAntiMeleeAoE += enemy.second.type.mineralPrice() + enemy.second.type.gasPrice();
		}
		if (enemy.second.type == BWAPI::UnitTypes::Protoss_Corsair
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Valkyrie
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Archon)
		{
			enemyAirAoE += enemy.second.type.mineralPrice() + enemy.second.type.gasPrice();
		}
		if (enemy.second.type.isFlyer() && enemy.second.type != BWAPI::UnitTypes::Zerg_Overlord)
		{
			enemyAirCost += enemy.second.type.mineralPrice() + enemy.second.type.gasPrice();
		}
		if (enemy.second.type.airWeapon() != BWAPI::WeaponTypes::Enum::None)
		{
			enemyAntiAirCost += enemy.second.type.mineralPrice() + enemy.second.type.gasPrice();
			if (enemy.second.type.isFlyer())
			{
				enemyFlyingAntiAir += enemy.second.type.mineralPrice() + enemy.second.type.gasPrice();
			}
		}
		if (enemy.second.type.groundWeapon() != BWAPI::WeaponTypes::Enum::None)
		{
			if (enemy.second.type == BWAPI::UnitTypes::Protoss_Scout
				|| enemy.second.type == BWAPI::UnitTypes::Terran_Wraith
				|| enemy.second.type == BWAPI::UnitTypes::Terran_Goliath)
			{
				enemyAntiGroundCost += (enemy.second.type.mineralPrice() + enemy.second.type.gasPrice()) / 2;
			}
			else
			{
				enemyAntiGroundCost += enemy.second.type.mineralPrice() + enemy.second.type.gasPrice();
			}
		}
		if (!enemy.second.type.isWorker() && !enemy.second.type.isBuilding() && enemy.second.type != BWAPI::UnitTypes::Zerg_Overlord)
		{
			enemyUnitCost += enemy.second.type.mineralPrice() + enemy.second.type.gasPrice();
			if (!enemy.second.type.isFlyer())
			{
				enemyGroundCost += enemy.second.type.mineralPrice() + enemy.second.type.gasPrice();
			}
		}
		if (enemy.second.type.isDetector())
		{
			lurkersStillGood = false;
			if (!enemy.second.type.isBuilding())
			{
				enemyMobileDetectorCost += enemy.second.type.mineralPrice() + enemy.second.type.gasPrice();
			}
		}
		if (enemy.second.type == BWAPI::UnitTypes::Protoss_Reaver
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Shuttle
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Dropship
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode)
		{
			allowMutaInMutaLessBuild = true;
		}
		if (enemy.second.type.isBuilding())
		{
			enemyBuildingCost += enemy.second.type.mineralPrice() + enemy.second.type.gasPrice();
			if (enemy.second.type == BWAPI::UnitTypes::Terran_Barracks
				|| enemy.second.type == BWAPI::UnitTypes::Terran_Factory
				|| enemy.second.type == BWAPI::UnitTypes::Protoss_Gateway
				|| enemy.second.type == BWAPI::UnitTypes::Protoss_Robotics_Facility
				|| enemy.second.type == BWAPI::UnitTypes::Zerg_Spawning_Pool
				|| enemy.second.type == BWAPI::UnitTypes::Zerg_Hydralisk_Den
				|| enemy.second.type == BWAPI::UnitTypes::Zerg_Ultralisk_Cavern
				|| enemy.second.type == BWAPI::UnitTypes::Zerg_Defiler_Mound)
			{
				enemyGroundCost += enemy.second.type.mineralPrice() + enemy.second.type.gasPrice();
			}
			if (enemy.second.type == BWAPI::UnitTypes::Terran_Starport
				|| enemy.second.type == BWAPI::UnitTypes::Protoss_Stargate
				|| enemy.second.type == BWAPI::UnitTypes::Zerg_Spire
				|| enemy.second.type == BWAPI::UnitTypes::Zerg_Greater_Spire)
			{
				enemyAirCost += enemy.second.type.mineralPrice() + enemy.second.type.gasPrice();
			}
			if (enemy.second.type.canAttack())
			{
				enemyDefenseBuildingCost += enemy.second.type.mineralPrice() + enemy.second.type.gasPrice();
				if (enemy.second.type == BWAPI::UnitTypes::Zerg_Sunken_Colony || enemy.second.type == BWAPI::UnitTypes::Zerg_Spore_Colony)
				{
					enemyDefenseBuildingCost += BWAPI::UnitTypes::Zerg_Creep_Colony.mineralPrice() + BWAPI::UnitTypes::Zerg_Sunken_Colony.gasPrice();
					enemyDefenseBuildingCost += BWAPI::UnitTypes::Zerg_Drone.mineralPrice() + BWAPI::UnitTypes::Zerg_Drone.gasPrice();
				}
			}
			if (enemy.second.type == BWAPI::UnitTypes::Terran_Bunker)
			{
				enemyDefenseBuildingCost += enemy.second.type.mineralPrice() + enemy.second.type.gasPrice();
				enemyUnitCost += 4 * BWAPI::UnitTypes::Terran_Marine.mineralPrice();
			}
			if (enemy.second.type.isFlyingBuilding() && enemy.second.unit->isFlying())
			{
				enemyFlyingBuildingCost += enemy.second.type.mineralPrice() + enemy.second.type.gasPrice();
			}
		}
		//hydraUnlock against Terran:
		if (enemy.second.type == BWAPI::UnitTypes::Terran_Vulture
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Vulture_Spider_Mine
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Goliath
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Missile_Turret
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Factory
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Armory
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Starport
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Bunker
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Wraith
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Valkyrie
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Battlecruiser
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Dropship
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Science_Vessel)
		{
			hydraUnlock = true;
		}
		if (enemy.second.type == BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Factory
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Armory
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Starport
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Wraith
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Valkyrie
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Battlecruiser
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Dropship
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Science_Vessel)
		{
			mutaUnlock = true;
		}
		//hydraUnlock against Protoss
		if (enemy.second.type == BWAPI::UnitTypes::Protoss_Dragoon
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Archon
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Cybernetics_Core
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Templar_Archives
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Stargate
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Fleet_Beacon
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Arbiter_Tribunal
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Corsair
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Scout
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Carrier
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Arbiter
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Photon_Cannon
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Forge)
		{
			hydraUnlock = true;
		}
		//hydraUnlock against Zerg
		if (enemy.second.type == BWAPI::UnitTypes::Zerg_Devourer)
		{
			hydraUnlock = true;
		}
		//lurkerUnlock against Terran
		if (enemy.second.type == BWAPI::UnitTypes::Terran_Firebat
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Medic
			|| enemy.second.type == BWAPI::UnitTypes::Terran_Academy)
		{
			lurkerUnlock = true;
		}
		//mutaUnlock against Protoss
		if (enemy.second.type == BWAPI::UnitTypes::Protoss_Stargate
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Fleet_Beacon
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Arbiter_Tribunal
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Corsair
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Scout
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Carrier
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Arbiter
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Shuttle
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Reaver
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Observer
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Robotics_Facility
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Robotics_Support_Bay)
		{
			mutaUnlock = true;
		}
		//guardianUnlock against Protoss
		if (enemy.second.type == BWAPI::UnitTypes::Protoss_Archon
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Dark_Templar
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Templar_Archives
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_High_Templar)
		{
			guardianUnlock = true;
			ultraUnlock = true;
		}
		if (enemy.second.type == BWAPI::UnitTypes::Terran_Command_Center
			|| enemy.second.type == BWAPI::UnitTypes::Protoss_Nexus)
		{
			baseCount++;
		}
		//mutaUnlock against Zerg
		if (enemy.second.type == BWAPI::UnitTypes::Zerg_Sunken_Colony
			|| enemy.second.type == BWAPI::UnitTypes::Zerg_Hydralisk
			|| enemy.second.type == BWAPI::UnitTypes::Zerg_Lair
			|| enemy.second.type == BWAPI::UnitTypes::Zerg_Spire
			|| enemy.second.type == BWAPI::UnitTypes::Zerg_Mutalisk
			|| enemy.second.type == BWAPI::UnitTypes::Zerg_Scourge
			|| enemy.second.type == BWAPI::UnitTypes::Zerg_Hydralisk_Den)
		{
			mutaUnlock = true;
		}
	}
	if (BWAPI::Broodwar->enemy()->getRace() == BWAPI::Races::Protoss
		|| BWAPI::Broodwar->enemy()->getRace() == BWAPI::Races::Terran)
	{
		if (baseCount >= 2)
		{
			hydraUnlock = true;
			mutaUnlock = true;
		}
	}
	for (auto myUnits : InformationManager::Instance().getUnitData(BWAPI::Broodwar->self()).getUnits())
	{
		if (myUnits.second.type == BWAPI::UnitTypes::Zerg_Sunken_Colony && BWAPI::Broodwar->enemy()->getRace() == BWAPI::Races::Zerg)
		{
			mutaUnlock = true;
		}
		if (!myUnits.second.type.isWorker() && !myUnits.second.type.isBuilding() && myUnits.second.type != BWAPI::UnitTypes::Zerg_Overlord)
		{
			if (myUnits.second.type == BWAPI::UnitTypes::Zerg_Zergling)
			{
				myLingCost += (myUnits.second.type.mineralPrice() + myUnits.second.type.gasPrice()) / 2;
				myTotalCost += (myUnits.second.type.mineralPrice() + myUnits.second.type.gasPrice()) / 2;
				myUnitCost += (myUnits.second.type.mineralPrice() + myUnits.second.type.gasPrice()) / 2;
			}
			myUnitCost += myUnits.second.type.mineralPrice() + myUnits.second.type.gasPrice();
		}
		else
		{
			myTotalCost += myUnits.second.type.mineralPrice() + myUnits.second.type.gasPrice();
		}
		if (myUnits.second.type == BWAPI::UnitTypes::Zerg_Hydralisk)
		{
			myHydraCost += myUnits.second.type.mineralPrice() + myUnits.second.type.gasPrice();
		}
		if (myUnits.second.type == BWAPI::UnitTypes::Zerg_Lurker)
		{
			myLurkerCost += myUnits.second.type.mineralPrice() + myUnits.second.type.gasPrice();
			myLurkerCost += BWAPI::UnitTypes::Zerg_Hydralisk.mineralPrice() + BWAPI::UnitTypes::Zerg_Hydralisk.gasPrice();
		}
		if (myUnits.second.type == BWAPI::UnitTypes::Zerg_Mutalisk)
		{
			myMutaCost += myUnits.second.type.mineralPrice() + myUnits.second.type.gasPrice();
		}
		if (myUnits.second.type == BWAPI::UnitTypes::Zerg_Ultralisk)
		{
			myUltraCost += myUnits.second.type.mineralPrice() + myUnits.second.type.gasPrice();
		}
		if (myUnits.second.type == BWAPI::UnitTypes::Zerg_Guardian)
		{
			myGuardCost += myUnits.second.type.mineralPrice() + myUnits.second.type.gasPrice();
		}
	}
	double remainingLingTechCost = BWAPI::UnitTypes::Zerg_Hatchery.mineralPrice() + BWAPI::UnitTypes::Zerg_Hatchery.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Spawning_Pool.mineralPrice() + BWAPI::UnitTypes::Zerg_Spawning_Pool.gasPrice();
	double remainingHydraTechCost = BWAPI::UnitTypes::Zerg_Hatchery.mineralPrice() + BWAPI::UnitTypes::Zerg_Hatchery.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Spawning_Pool.mineralPrice() + BWAPI::UnitTypes::Zerg_Spawning_Pool.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Extractor.mineralPrice() + BWAPI::UnitTypes::Zerg_Extractor.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Hydralisk_Den.mineralPrice() + BWAPI::UnitTypes::Zerg_Hydralisk_Den.gasPrice();
	double remainingLurkerTechCost = BWAPI::UnitTypes::Zerg_Hatchery.mineralPrice() + BWAPI::UnitTypes::Zerg_Hatchery.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Spawning_Pool.mineralPrice() + BWAPI::UnitTypes::Zerg_Spawning_Pool.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Extractor.mineralPrice() + BWAPI::UnitTypes::Zerg_Extractor.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Hydralisk_Den.mineralPrice() + BWAPI::UnitTypes::Zerg_Hydralisk_Den.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Lair.mineralPrice() + BWAPI::UnitTypes::Zerg_Lair.gasPrice()
		+ BWAPI::TechTypes::Lurker_Aspect.mineralPrice() + BWAPI::TechTypes::Lurker_Aspect.gasPrice();
	double remainingMutaTechCost = BWAPI::UnitTypes::Zerg_Hatchery.mineralPrice() + BWAPI::UnitTypes::Zerg_Hatchery.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Spawning_Pool.mineralPrice() + BWAPI::UnitTypes::Zerg_Spawning_Pool.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Extractor.mineralPrice() + BWAPI::UnitTypes::Zerg_Extractor.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Lair.mineralPrice() + BWAPI::UnitTypes::Zerg_Lair.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Spire.mineralPrice() + BWAPI::UnitTypes::Zerg_Spire.gasPrice();
	double remainingUltraTechCost = BWAPI::UnitTypes::Zerg_Hatchery.mineralPrice() + BWAPI::UnitTypes::Zerg_Hatchery.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Spawning_Pool.mineralPrice() + BWAPI::UnitTypes::Zerg_Spawning_Pool.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Extractor.mineralPrice() + BWAPI::UnitTypes::Zerg_Extractor.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Lair.mineralPrice() + BWAPI::UnitTypes::Zerg_Lair.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Queens_Nest.mineralPrice() + BWAPI::UnitTypes::Zerg_Queens_Nest.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Hive.mineralPrice() + BWAPI::UnitTypes::Zerg_Hive.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Ultralisk_Cavern.mineralPrice() + BWAPI::UnitTypes::Zerg_Ultralisk_Cavern.gasPrice();
	double remainingGuardTechCost = BWAPI::UnitTypes::Zerg_Hatchery.mineralPrice() + BWAPI::UnitTypes::Zerg_Hatchery.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Spawning_Pool.mineralPrice() + BWAPI::UnitTypes::Zerg_Spawning_Pool.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Extractor.mineralPrice() + BWAPI::UnitTypes::Zerg_Extractor.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Lair.mineralPrice() + BWAPI::UnitTypes::Zerg_Lair.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Spire.mineralPrice() + BWAPI::UnitTypes::Zerg_Spire.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Queens_Nest.mineralPrice() + BWAPI::UnitTypes::Zerg_Queens_Nest.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Hive.mineralPrice() + BWAPI::UnitTypes::Zerg_Hive.gasPrice()
		+ BWAPI::UnitTypes::Zerg_Spire.mineralPrice() + BWAPI::UnitTypes::Zerg_Greater_Spire.gasPrice();

	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Hatchery) > 0 || UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Lair) > 0 || UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Hive) > 0)
	{
		remainingLingTechCost -= BWAPI::UnitTypes::Zerg_Hatchery.mineralPrice() + BWAPI::UnitTypes::Zerg_Hatchery.gasPrice();
		remainingHydraTechCost -= BWAPI::UnitTypes::Zerg_Hatchery.mineralPrice() + BWAPI::UnitTypes::Zerg_Hatchery.gasPrice();
		remainingLurkerTechCost -= BWAPI::UnitTypes::Zerg_Hatchery.mineralPrice() + BWAPI::UnitTypes::Zerg_Hatchery.gasPrice();
		remainingMutaTechCost -= BWAPI::UnitTypes::Zerg_Hatchery.mineralPrice() + BWAPI::UnitTypes::Zerg_Hatchery.gasPrice();
		remainingUltraTechCost -= BWAPI::UnitTypes::Zerg_Hatchery.mineralPrice() + BWAPI::UnitTypes::Zerg_Hatchery.gasPrice();
		remainingGuardTechCost -= BWAPI::UnitTypes::Zerg_Hatchery.mineralPrice() + BWAPI::UnitTypes::Zerg_Hatchery.gasPrice();
	}
	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Spawning_Pool) > 0)
	{
		remainingLingTechCost -= BWAPI::UnitTypes::Zerg_Spawning_Pool.mineralPrice() + BWAPI::UnitTypes::Zerg_Spawning_Pool.gasPrice();
		remainingHydraTechCost -= BWAPI::UnitTypes::Zerg_Spawning_Pool.mineralPrice() + BWAPI::UnitTypes::Zerg_Spawning_Pool.gasPrice();
		remainingLurkerTechCost -= BWAPI::UnitTypes::Zerg_Spawning_Pool.mineralPrice() + BWAPI::UnitTypes::Zerg_Spawning_Pool.gasPrice();
		remainingMutaTechCost -= BWAPI::UnitTypes::Zerg_Spawning_Pool.mineralPrice() + BWAPI::UnitTypes::Zerg_Spawning_Pool.gasPrice();
		remainingUltraTechCost -= BWAPI::UnitTypes::Zerg_Spawning_Pool.mineralPrice() + BWAPI::UnitTypes::Zerg_Spawning_Pool.gasPrice();
		remainingGuardTechCost -= BWAPI::UnitTypes::Zerg_Spawning_Pool.mineralPrice() + BWAPI::UnitTypes::Zerg_Spawning_Pool.gasPrice();
	}
	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Hydralisk_Den) > 0)
	{
		remainingHydraTechCost -= BWAPI::UnitTypes::Zerg_Hydralisk_Den.mineralPrice() + BWAPI::UnitTypes::Zerg_Hydralisk_Den.gasPrice();
		remainingLurkerTechCost -= BWAPI::UnitTypes::Zerg_Hydralisk_Den.mineralPrice() + BWAPI::UnitTypes::Zerg_Hydralisk_Den.gasPrice();
	}
	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Lair) > 0 || UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Hive) > 0)
	{
		remainingLurkerTechCost -= BWAPI::UnitTypes::Zerg_Lair.mineralPrice() + BWAPI::UnitTypes::Zerg_Lair.gasPrice();
		remainingMutaTechCost -= BWAPI::UnitTypes::Zerg_Lair.mineralPrice() + BWAPI::UnitTypes::Zerg_Lair.gasPrice();
		remainingUltraTechCost -= BWAPI::UnitTypes::Zerg_Lair.mineralPrice() + BWAPI::UnitTypes::Zerg_Lair.gasPrice();
		remainingGuardTechCost -= BWAPI::UnitTypes::Zerg_Lair.mineralPrice() + BWAPI::UnitTypes::Zerg_Lair.gasPrice();
	}
	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Spire) > 0)
	{
		remainingMutaTechCost -= BWAPI::UnitTypes::Zerg_Spire.mineralPrice() + BWAPI::UnitTypes::Zerg_Spire.gasPrice();
		remainingGuardTechCost -= BWAPI::UnitTypes::Zerg_Spire.mineralPrice() + BWAPI::UnitTypes::Zerg_Spire.gasPrice();
	}
	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Queens_Nest) > 0)
	{
		remainingUltraTechCost -= BWAPI::UnitTypes::Zerg_Queens_Nest.mineralPrice() + BWAPI::UnitTypes::Zerg_Queens_Nest.gasPrice();
		remainingGuardTechCost -= BWAPI::UnitTypes::Zerg_Queens_Nest.mineralPrice() + BWAPI::UnitTypes::Zerg_Queens_Nest.gasPrice();
	}
	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Hive) > 0)
	{
		remainingUltraTechCost -= BWAPI::UnitTypes::Zerg_Hive.mineralPrice() + BWAPI::UnitTypes::Zerg_Hive.gasPrice();
		remainingGuardTechCost -= BWAPI::UnitTypes::Zerg_Hive.mineralPrice() + BWAPI::UnitTypes::Zerg_Hive.gasPrice();
	}
	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Ultralisk_Cavern) > 0)
	{
		remainingUltraTechCost -= BWAPI::UnitTypes::Zerg_Ultralisk_Cavern.mineralPrice() + BWAPI::UnitTypes::Zerg_Ultralisk_Cavern.gasPrice();
	}
	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Greater_Spire) > 0)
	{
		remainingGuardTechCost -= BWAPI::UnitTypes::Zerg_Greater_Spire.mineralPrice() + BWAPI::UnitTypes::Zerg_Greater_Spire.gasPrice();
	}
	if (BWAPI::Broodwar->self()->hasResearched(BWAPI::TechTypes::Lurker_Aspect))
	{
		remainingLurkerTechCost -= BWAPI::TechTypes::Lurker_Aspect.mineralPrice() + BWAPI::TechTypes::Lurker_Aspect.gasPrice();
	}
	if (myHydraCost + myLingCost + myLurkerCost > 2 * remainingUltraTechCost)
	{
		ultraUnlock = true;
	}
	if (enemyDefenseBuildingCost > enemyAirCost + enemyGroundCost)
	{
		guardianUnlock = true;
	}

	double lurkerBonus = 2;
	if (enemyMobileDetectorCost > 0 || !lurkersStillGood)
	{
		lurkerBonus = 1;
	}

	bool workerCut = false;

	if (_strategy != "Freestyle" && UnitUtil::GetAllUnitCount(BWAPI::Broodwar->self()->getRace().getWorker()) * 2 >= strategy.FreeStyleWorkerSupply)
	{
		_strategy = "Freestyle";
		BWAPI::Broodwar->sendText("Freestyle unlocked!");
	}

	if (_strategy == "StoicLingMuta")
	{
		lingScore = myTotalCost - remainingLingTechCost - myLingCost;
		mutaScore = myTotalCost - myMutaCost - remainingMutaTechCost;
	}

	if (_strategy == "StoicHydra")
	{
		lingScore = myTotalCost - remainingLingTechCost - myLingCost * 4;
		hydraScore = myTotalCost - remainingHydraTechCost - myHydraCost;
	}

	if (_strategy == "StoicMuta")
	{
		lingScore = myTotalCost - remainingLingTechCost - myLingCost * 4;
		mutaScore = myTotalCost - myMutaCost - remainingMutaTechCost;
	}

	if (_strategy == "StoicLingHydra")
	{
		lingScore = myTotalCost - remainingLingTechCost - myLingCost;
		hydraScore = myTotalCost - remainingHydraTechCost - myHydraCost;
	}

	if (_strategy == "StoicLingLurker")
	{
		lingScore = myTotalCost - remainingLingTechCost - myLingCost;
		lurkerScore = myTotalCost - remainingLurkerTechCost - myLurkerCost;
	}

	if (_strategy == "StoicLurker")
	{
		lingScore = myTotalCost - remainingLingTechCost - myLingCost * 4;
		lurkerScore = myTotalCost - remainingLurkerTechCost - myLurkerCost;
	}

	if (_strategy == "StoicLingHydraLurker")
	{
		lingScore = myTotalCost - remainingLingTechCost - myLingCost;
		hydraScore = myTotalCost - remainingHydraTechCost - myHydraCost;
		lurkerScore = myTotalCost - remainingLurkerTechCost - myLurkerCost;
	}

	if (_strategy == "StoicHydraLurker")
	{
		lingScore = myTotalCost - remainingLingTechCost - myLingCost * 4;
		hydraScore = myTotalCost - remainingHydraTechCost - myHydraCost;
		lurkerScore = myTotalCost - remainingLurkerTechCost - myLurkerCost;
	}

	if (_strategy == "StoicLingHydraUltra")
	{
		lingScore = myTotalCost - remainingLingTechCost - myLingCost;
		hydraScore = myTotalCost - remainingHydraTechCost - myHydraCost;
		ultraScore = myTotalCost - myUltraCost - remainingUltraTechCost;
	}

	if (_strategy == "StoicLingMutaGuardian")
	{
		lingScore = myTotalCost - remainingLingTechCost - myLingCost;
		mutaScore = myTotalCost - myMutaCost - remainingMutaTechCost;
		guardScore = myTotalCost - myGuardCost - remainingGuardTechCost;
	}

	if (_strategy == "StoicLingHydraGuardian")
	{
		lingScore = myTotalCost - remainingLingTechCost - myLingCost;
		hydraScore = myTotalCost - remainingHydraTechCost - myHydraCost;
		guardScore = myTotalCost - myGuardCost - remainingGuardTechCost;
	}

	if (_strategy == "StoicHydraGuardian")
	{
		lingScore = myTotalCost - remainingLingTechCost - myLingCost * 4;
		hydraScore = myTotalCost - remainingHydraTechCost - myHydraCost;
		guardScore = myTotalCost - myGuardCost - remainingGuardTechCost;
	}

	if (_strategy == "MutaAllIn")
	{
		if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Spire, true) + UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Greater_Spire) == 0)
		{
			lingScore = myTotalCost - remainingLingTechCost - myLingCost * 4 + enemyGroundCost - enemyAntiGroundCost - enemyGroundAoE - enemyAntiMeleeAoE;
		}
		else
		{
			lingScore = 0;
			workerCut = true;
		}
		mutaScore = myTotalCost - myMutaCost - remainingMutaTechCost + enemyGroundCost + enemyAirCost - enemyAntiAirCost;
	}
	if (_strategy == "HydraAllIn")
	{
		if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Hydralisk_Den, true) == 0)
		{
			lingScore = myTotalCost - remainingLingTechCost - myLingCost * 4 + enemyGroundCost - enemyAntiGroundCost - enemyGroundAoE - enemyAntiMeleeAoE;
		}
		else
		{
			lingScore = 0;
			workerCut = true;
		}
		hydraScore = myTotalCost - myHydraCost - remainingHydraTechCost + enemyGroundCost - enemyAntiGroundCost + enemyAirCost - enemySmallCost * 0.5 - enemyMediumCost * 0.25 - enemyGroundAoE + enemyDefenseBuildingCost / 2;
	}
	if (_strategy == "Adaptive")
	{
		lingScore = myTotalCost - remainingLingTechCost + enemyGroundCost - enemyAntiGroundCost - enemyGroundAoE - enemyAntiMeleeAoE;
		hydraScore = myTotalCost - remainingHydraTechCost + enemyGroundCost - enemyAntiGroundCost + enemyAirCost - enemySmallCost * 0.5 - enemyMediumCost * 0.25 - enemyGroundAoE + enemyDefenseBuildingCost / 2;
		mutaScore = myTotalCost - remainingMutaTechCost + enemyGroundCost + enemyAirCost - enemyAntiAirCost - enemyDefenseBuildingCost;
		lurkerScore = myTotalCost - remainingLurkerTechCost + enemyGroundCost * lurkerBonus - enemyAntiGroundCost - enemyDefenseBuildingCost;
		ultraScore = myTotalCost - remainingUltraTechCost + enemyGroundCost - enemyAntiGroundCost;
		guardScore = myTotalCost - remainingGuardTechCost + enemyGroundCost - enemyAntiAirCost + enemyGroundAoE + enemyDefenseBuildingCost;
	}
	if (_strategy == "Randomized")
	{
		if (strategy.enableZergling)
		{
			lingScore = myTotalCost - remainingLingTechCost - myLingCost * strategy.ZerglingDeflation;
		}
		if (strategy.enableHydralisk)
		{
			hydraScore = myTotalCost - remainingHydraTechCost - myHydraCost * strategy.HydraliskDeflation;
		}
		if (strategy.enableLurker)
		{
			lurkerScore = myTotalCost - remainingLurkerTechCost - myLurkerCost * strategy.LurkerDeflation;
		}
		if (strategy.enableMutalisk)
		{
			mutaScore = myTotalCost - remainingMutaTechCost - myMutaCost * strategy.MutaliskDeflation;
		}
		if (strategy.enableUltralisk)
		{
			ultraScore = myTotalCost - remainingUltraTechCost - myUltraCost * strategy.UltraliskDeflation;
		}
		if (strategy.enableGuardian)
		{
			guardScore = myTotalCost - remainingGuardTechCost - myGuardCost * strategy.GuardianDeflation;
		}
	}
	if (_strategy == "Freestyle")
	{
		lingScore = myTotalCost - remainingLingTechCost - myLingCost + enemyGroundCost - enemyAntiGroundCost - enemyGroundAoE - enemyAntiMeleeAoE;
		hydraScore = myTotalCost - myHydraCost - remainingHydraTechCost + enemyGroundCost - enemyAntiGroundCost + enemyAirCost - enemySmallCost * 0.5 - enemyMediumCost * 0.25 - enemyGroundAoE + enemyDefenseBuildingCost / 2 + enemyAntiMeleeAoE;
		mutaScore = myTotalCost - myMutaCost - remainingMutaTechCost + enemyGroundCost + enemyAirCost - enemyAntiAirCost - enemyDefenseBuildingCost;
		lurkerScore = myTotalCost - myLurkerCost - remainingLurkerTechCost + enemyGroundCost * lurkerBonus - enemyAntiGroundCost - enemyDefenseBuildingCost;
		ultraScore = myTotalCost - myUltraCost - remainingUltraTechCost + (myLingCost + myHydraCost + myLurkerCost) / 2.0;
		guardScore = myTotalCost - myGuardCost - remainingGuardTechCost + enemyGroundCost - enemyAntiAirCost + enemyGroundAoE + enemyDefenseBuildingCost + enemyAntiMeleeAoE;
	}
	if (_strategy == "SmartUnlock")
	{
		lingScore = myTotalCost - remainingLingTechCost - myLingCost + enemyGroundCost - enemyAntiGroundCost - enemyGroundAoE - enemyAntiMeleeAoE;
		if (hydraUnlock)
		{
			hydraScore = myTotalCost - myHydraCost - remainingHydraTechCost + enemyGroundCost - enemyAntiGroundCost + enemyAirCost - enemySmallCost * 0.5 - enemyMediumCost * 0.25 - enemyGroundAoE + enemyDefenseBuildingCost / 2 + enemyAntiMeleeAoE;
		}
		if (mutaUnlock)
		{
			mutaScore = myTotalCost - myMutaCost - remainingMutaTechCost + enemyGroundCost + enemyAirCost - enemyAntiAirCost - enemyDefenseBuildingCost;
		}
		if (lurkerUnlock)
		{
			lurkerScore = myTotalCost - myLurkerCost - remainingLurkerTechCost + enemyGroundCost * lurkerBonus - enemyAntiGroundCost - enemyDefenseBuildingCost;
		}
		if (ultraUnlock)
		{
			ultraScore = myTotalCost - myUltraCost - remainingUltraTechCost + (myLingCost + myHydraCost + myLurkerCost) / 2.0;
		}
		if (guardianUnlock)
		{
			guardScore = myTotalCost - myGuardCost - remainingGuardTechCost + enemyGroundCost - enemyAntiAirCost + enemyGroundAoE + enemyDefenseBuildingCost + enemyAntiMeleeAoE;
		}
	}

	if (enemyFlyingAntiAir > 0) //3 Wraiths killed like 10 guardians in one game... don't get Guardians when enemy has stuff like that!
	{
		guardScore = 0;
	}
	strategy.ZerglingDeflation = lingScore;
	strategy.HydraliskDeflation = hydraScore;
	strategy.LurkerDeflation = lurkerScore;
	strategy.MutaliskDeflation = mutaScore;
	strategy.UltraliskDeflation = ultraScore;
	strategy.GuardianDeflation = guardScore;

	if (InformationManager::Instance().supplyIWasAttacked < 400)
	{
		strategy.WorkerSupplyBeforeUnits = InformationManager::Instance().supplyIWasAttacked;
	}

	_timerManager.startTimer(TimerManager::MapGrid);
	MapGrid::Instance().update();
	_timerManager.stopTimer(TimerManager::MapGrid);

	//_timerManager.startTimer(TimerManager::MapTools);
	//MapTools::Instance().update();
	//_timerManager.stopTimer(TimerManager::MapTools);

	//_timerManager.startTimer(TimerManager::Search);
	//BOSSManager::Instance().update(35 - _timerManager.getTotalElapsed());
	//_timerManager.stopTimer(TimerManager::Search);

	// economy and base managers
	_timerManager.startTimer(TimerManager::Worker);
	WorkerManager::Instance().update();
	_timerManager.stopTimer(TimerManager::Worker);

	_timerManager.startTimer(TimerManager::Production);
	ProductionManager::Instance().update(strategy, sunkensPerWorkerSupply, lingScore, hydraScore, lurkerScore, mutaScore, ultraScore, guardScore);
	_timerManager.stopTimer(TimerManager::Production);

	_timerManager.startTimer(TimerManager::Building);
	BuildingManager::Instance().update();
	_timerManager.stopTimer(TimerManager::Building);

	// combat and scouting managers
	_timerManager.startTimer(TimerManager::Combat);
	_combatCommander.update(_combatUnits, myTotalCost, enemyUnitCost);
	_timerManager.stopTimer(TimerManager::Combat);

	_timerManager.startTimer(TimerManager::Scout);
    ScoutManager::Instance().update();
	_timerManager.stopTimer(TimerManager::Scout);
		
	_timerManager.stopTimer(TimerManager::All);

	drawDebugInterface();
}

void GameCommander::drawDebugInterface()
{
	InformationManager::Instance().drawExtendedInterface();
	InformationManager::Instance().drawUnitInformation(425,30);
	BuildingManager::Instance().drawBuildingInformation(200,50);
	BuildingPlacer::Instance().drawReservedTiles();
	ProductionManager::Instance().drawProductionInformation(30, 50);
	BOSSManager::Instance().drawSearchInformation(490, 100);
    BOSSManager::Instance().drawStateInformation(250, 0);
    
    _timerManager.displayTimers(490, 225);
    drawGameInformation(4, 1);

	// draw position of mouse cursor
	if (Config::Debug::DrawMouseCursorInfo)
	{
		int mouseX = BWAPI::Broodwar->getMousePosition().x + BWAPI::Broodwar->getScreenPosition().x;
		int mouseY = BWAPI::Broodwar->getMousePosition().y + BWAPI::Broodwar->getScreenPosition().y;
		BWAPI::Broodwar->drawTextMap(mouseX + 20, mouseY, " %d %d", mouseX, mouseY);
	}
}

void GameCommander::drawGameInformation(int x, int y)
{
    BWAPI::Broodwar->drawTextScreen(x, y, "\x04Players:");
	BWAPI::Broodwar->drawTextScreen(x + 50, y, "%c%s\x04: %d %c%s\x04: %d", BWAPI::Broodwar->self()->getTextColor(), BWAPI::Broodwar->self()->getName().c_str(), (int)InformationManager::Instance().getUnitData(BWAPI::Broodwar->enemy()).totalLost(), BWAPI::Broodwar->enemy()->getTextColor(), BWAPI::Broodwar->enemy()->getName().c_str(), (int)InformationManager::Instance().getUnitData(BWAPI::Broodwar->self()).totalLost());
	y += 12;

	BWAPI::Broodwar->drawTextScreen(x, y, "\x04Strategy:");
	BWAPI::Broodwar->drawTextScreen(x + 50, y, "\x03%s (#%d) with %d Drones before Army", _strategy.c_str(), StratNo, int(strategy.WorkerSupplyBeforeUnits / 2.0) + 4);
	BWAPI::Broodwar->setTextSize();
	y += 12;

	BWAPI::Broodwar->drawTextScreen(x, y, "\x04" "Army:");
	BWAPI::Broodwar->drawTextScreen(x + 50, y, "\x03" "Zergling: %d Hydralisk: %d Lurker: %d Mutalisk: %d Ultralisk: %d Guardian: %d", int(lingScore), int(hydraScore), int(lurkerScore), int(mutaScore), int(ultraScore), int(guardScore));
	BWAPI::Broodwar->setTextSize();
	y += 12;

    BWAPI::Broodwar->drawTextScreen(x, y, "\x04Map:");
	BWAPI::Broodwar->drawTextScreen(x+50, y, "\x03%s", BWAPI::Broodwar->mapFileName().c_str());
	BWAPI::Broodwar->setTextSize();
}

// assigns units to various managers
void GameCommander::handleUnitAssignments()
{
	_validUnits.clear();
    _combatUnits.clear();

	// filter our units for those which are valid and usable
	setValidUnits();

	// set each type of unit
	setScoutUnits();
	setCombatUnits();
}

bool GameCommander::isAssigned(BWAPI::Unit unit) const
{
	return _combatUnits.contains(unit) || _scoutUnits.contains(unit);
}

// validates units as usable for distribution to various managers
void GameCommander::setValidUnits()
{
	// make sure the unit is completed and alive and usable
	for (auto & unit : BWAPI::Broodwar->self()->getUnits())
	{
		if (UnitUtil::IsValidUnit(unit))
		{	
			_validUnits.insert(unit);
		}
	}
}

void GameCommander::setScoutUnits()
{
    // if we haven't set a scout unit, do it
	if (_scoutUnits.empty() && !_initialScoutSet && UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Overlord) >= 2)
    {
		// grab the closest worker to the supply provider to send to scout
		BWAPI::Unit workerScout = getClosestWorkerToTarget(BWAPI::Position(MapTools::Instance().getNextExpansion()));

		// if we find a worker (which we should) add it to the scout units
		if (workerScout)
		{
            ScoutManager::Instance().setWorkerScout(workerScout);
			assignUnit(workerScout, _scoutUnits);
            _initialScoutSet++;
		}
    }
}

// sets combat units to be passed to CombatCommander
void GameCommander::setCombatUnits()
{
	for (auto & unit : _validUnits)
	{
		if (!isAssigned(unit) && UnitUtil::IsCombatUnit(unit) || unit->getType().isWorker())		
		{	
			assignUnit(unit, _combatUnits);
		}
	}
}

BWAPI::Unit GameCommander::getFirstSupplyProvider()
{
	BWAPI::Unit supplyProvider = nullptr;

	if (BWAPI::Broodwar->self()->getRace() == BWAPI::Races::Zerg)
	{
		for (auto & unit : BWAPI::Broodwar->self()->getUnits())
		{
			//if (unit->getType().isResourceDepot() && _initialScoutSet == 0)
			//{
			//	supplyProvider = unit;
			//}
			if (unit->getType() == BWAPI::UnitTypes::Zerg_Spawning_Pool)
			{
				supplyProvider = unit;
			}
		}
	}
	else
	{
		
		for (auto & unit : BWAPI::Broodwar->self()->getUnits())
		{
			if (unit->getType() == BWAPI::Broodwar->self()->getRace().getSupplyProvider())
			{
				supplyProvider = unit;
			}
		}
	}

	return supplyProvider;
}

void GameCommander::onUnitShow(BWAPI::Unit unit)			
{ 
	InformationManager::Instance().onUnitShow(unit); 
	WorkerManager::Instance().onUnitShow(unit);
}

void GameCommander::onUnitHide(BWAPI::Unit unit)			
{ 
	InformationManager::Instance().onUnitHide(unit); 
}

void GameCommander::onUnitCreate(BWAPI::Unit unit)		
{ 
	InformationManager::Instance().onUnitCreate(unit); 
}

void GameCommander::onUnitComplete(BWAPI::Unit unit)
{
	InformationManager::Instance().onUnitComplete(unit);
}

void GameCommander::onUnitRenegade(BWAPI::Unit unit)		
{ 
	InformationManager::Instance().onUnitRenegade(unit); 
}

void GameCommander::onUnitDestroy(BWAPI::Unit unit)		
{ 	
	ProductionManager::Instance().onUnitDestroy(unit);
	WorkerManager::Instance().onUnitDestroy(unit);
	InformationManager::Instance().onUnitDestroy(unit); 
}

void GameCommander::onUnitMorph(BWAPI::Unit unit)		
{ 
	InformationManager::Instance().onUnitMorph(unit);
	WorkerManager::Instance().onUnitMorph(unit);
}

BWAPI::Unit GameCommander::getClosestUnitToTarget(BWAPI::UnitType type, BWAPI::Position target)
{
	BWAPI::Unit closestUnit = nullptr;
	double closestDist = 100000;

	for (auto & unit : _validUnits)
	{
		if (unit->getType() == type)
		{
			double dist = unit->getDistance(target);
			if (!closestUnit || dist < closestDist)
			{
				closestUnit = unit;
				closestDist = dist;
			}
		}
	}

	return closestUnit;
}

BWAPI::Unit GameCommander::getClosestWorkerToTarget(BWAPI::Position target)
{
	BWAPI::Unit closestUnit = nullptr;
	double closestDist = 100000;

	for (auto & unit : _validUnits)
	{
		if (unit->getType().isWorker() && !unit->isCarryingMinerals())
		{
			double dist = unit->getDistance(target);
			if (!closestUnit || dist < closestDist)
			{
				closestUnit = unit;
				closestDist = dist;
			}
		}
	}

	return closestUnit;
}

void GameCommander::assignUnit(BWAPI::Unit unit, BWAPI::Unitset & set)
{
    if (_scoutUnits.contains(unit)) { _scoutUnits.erase(unit); }
    else if (_combatUnits.contains(unit)) { _combatUnits.erase(unit); }

    set.insert(unit);
}

void GameCommander::onGameStart()
{
	BWEM::Map::Instance().Initialize();
	BWEM::Map::Instance().EnableAutomaticPathAnalysis();
	bool startingLocationsOK = BWEM::Map::Instance().FindBasesForStartingLocations();
	assert(startingLocationsOK);

	_strategy = "SmartUnlock";
	_sversion = "2017-04-22-23-49";
	StratNo = 17;

	sunkensPerWorkerSupply = 0.1;

	strategy.expansionsitic = false;
	//if (rand() % 2 == 0)
	//{
	//	strategy.AllowAllIn = true;
	//}
	strategy.enableZergling = true;
	if (rand() % 3 == 0)
	{
		strategy.enableHydralisk = true;
	}
	if (rand() % 3 == 0)
	{
		strategy.enableLurker = true;
	}
	if (rand() % 2 == 0)
	{
		strategy.enableMutalisk = true;
	}
	if (rand() % 3 == 0)
	{
		strategy.enableUltralisk = true;
	}
	if (rand() % 3 == 0)
	{
		strategy.enableGuardian = true;
	}
	strategy.TechSpeed = rand() % 4 + 1;
	//strategy.WorkerSupplyBeforeUnits = rand() % 150;
	if (strategy.enableZergling == false)
	{
		strategy.WorkerSupplyBeforeUnits = std::max(strategy.WorkerSupplyBeforeUnits, 22);
	}
	if (BWAPI::Broodwar->enemy()->getRace() == BWAPI::Races::Zerg)
	{
		strategy.WorkerSupplyBeforeUnits = 12;
		strategy.droneToUnitRatio = 0.4;
		strategy.ForceToSaveForPoolAt = 18;
		strategy.attackWithoutLingSpeed = false;
	}
	if (strategy.AllowAllIn)
	{
		strategy.AllInWorkerSupply = std::max(rand() % 150, strategy.WorkerSupplyBeforeUnits);
	}
	//strategy.FreeStyleWorkerSupply = std::max(rand() % 150, strategy.WorkerSupplyBeforeUnits);
	//strategy.droneToUnitRatio = (rand() % 15 + 1) * 0.1;

	if (strategy.enableGuardian || strategy.enableUltralisk)
	{
		strategy.MutaliskDeflation = strategy.TechSpeed;
		strategy.LurkerDeflation = strategy.TechSpeed;
		strategy.HydraliskDeflation = strategy.TechSpeed;
		strategy.ZerglingDeflation = strategy.TechSpeed;
	}
	if (strategy.enableLurker || strategy.enableMutalisk)
	{
		strategy.HydraliskDeflation = strategy.TechSpeed;
		strategy.ZerglingDeflation = strategy.TechSpeed;
	}
	if (strategy.enableHydralisk)
	{
		strategy.ZerglingDeflation = strategy.TechSpeed;
	}
	//_strategy = strategyMap[StratNo];
	

	//if (BWAPI::Broodwar->enemy()->getRace() != BWAPI::Races::Unknown)
	//{
	//	std::string enemyName = BWAPI::Broodwar->enemy()->getName();
	//	std::string enemyRaceName = BWAPI::Broodwar->enemy()->getRace().getName();
	//	std::replace(enemyName.begin(), enemyName.end(), ' ', '_');

	//	std::string enemyFile = Config::Strategy::WriteDir + enemyName + _sversion + ".txt";
	//	std::string enemyRaceFile = Config::Strategy::WriteDir + enemyRaceName + _sversion + ".txt";
	//	std::string totalFile = Config::Strategy::WriteDir + _sversion + ".txt";

	//	std::ifstream file(enemyFile.c_str());
	//	if (!file.is_open())
	//	{
	//		file.open(enemyRaceFile.c_str());
	//	}
	//	if (!file.is_open())
	//	{
	//		file.open(totalFile.c_str());
	//	}
	//	if (file.is_open())
	//	{
	//		std::string line;
	//		int lastStrategy = 0;
	//		double lastMacroHeavyness = strategy.WorkerSupplyBeforeUnits;
	//		while (std::getline(file, line)) /* read a line */
	//		{
	//			if (line.find("lingScore") != std::string::npos)
	//			{
	//				InformationManager::Instance().lingScore = std::stod(line.substr(line.find_first_of("0123456789"), line.length() - line.find_first_of("0123456789")));
	//				if (line.find("-") != std::string::npos)
	//				{
	//					InformationManager::Instance().lingScore *= -1;
	//				}
	//			}
	//			if (line.find("hydraScore") != std::string::npos)
	//			{
	//				InformationManager::Instance().hydraScore = std::stod(line.substr(line.find_first_of("0123456789"), line.length() - line.find_first_of("0123456789")));
	//				if (line.find("-") != std::string::npos)
	//				{
	//					InformationManager::Instance().hydraScore *= -1;
	//				}
	//			}
	//			if (line.find("lurkerScore") != std::string::npos)
	//			{
	//				InformationManager::Instance().lurkerScore = std::stod(line.substr(line.find_first_of("0123456789"), line.length() - line.find_first_of("0123456789")));
	//				if (line.find("-") != std::string::npos)
	//				{
	//					InformationManager::Instance().lurkerScore *= -1;
	//				}
	//			}
	//			if (line.find("mutaScore") != std::string::npos)
	//			{
	//				InformationManager::Instance().mutaScore = std::stod(line.substr(line.find_first_of("0123456789"), line.length() - line.find_first_of("0123456789")));
	//				if (line.find("-") != std::string::npos)
	//				{
	//					InformationManager::Instance().mutaScore *= -1;
	//				}
	//			}
	//			if (line.find("ultraScore") != std::string::npos)
	//			{
	//				InformationManager::Instance().ultraScore = std::stod(line.substr(line.find_first_of("0123456789"), line.length() - line.find_first_of("0123456789")));
	//				if (line.find("-") != std::string::npos)
	//				{
	//					InformationManager::Instance().ultraScore *= -1;
	//				}
	//			}
	//			if (line.find("guardScore") != std::string::npos)
	//			{
	//				InformationManager::Instance().guardScore = std::stod(line.substr(line.find_first_of("0123456789"), line.length() - line.find_first_of("0123456789")));
	//				if (line.find("-") != std::string::npos)
	//				{
	//					InformationManager::Instance().guardScore *= -1;
	//				}
	//			}
	//			if (line.find("macroHeavyness") != std::string::npos)
	//			{
	//				lastMacroHeavyness = std::stod(line.substr(line.find_first_of("0123456789"), line.length() - line.find_first_of("0123456789")));
	//			}
	//			if (line.find("supplyISawAir") != std::string::npos)
	//			{
	//				InformationManager::Instance().supplyISawAir = std::stod(line.substr(line.find_first_of("0123456789"), line.length() - line.find_first_of("0123456789")));
	//			}
	//			if (line.find("strategy") != std::string::npos)
	//			{
	//				lastStrategy = std::stoi(line.substr(line.find_first_of("0123456789"), line.length() - line.find_first_of("0123456789")));
	//			}
	//			if (line.find("victory") != std::string::npos)
	//			{
	//				std::string victory = line.substr(line.find_first_of("=") + 1, line.length() - (line.find_first_of("=") + 1));
	//				if (victory == "true")
	//				{
	//					StratNo = lastStrategy;
	//					_strategy = strategyMap[StratNo];
	//					InformationManager::Instance().lingScore = 0;
	//					InformationManager::Instance().hydraScore = 0;
	//					InformationManager::Instance().lurkerScore = 0;
	//					InformationManager::Instance().mutaScore = 0;
	//					InformationManager::Instance().guardScore = 0;
	//					InformationManager::Instance().ultraScore = 0;
	//				}
	//				if (victory == "false")
	//				{
	//					strategy.WorkerSupplyBeforeUnits = lastMacroHeavyness;
	//				}
	//			}
	//		}
	//		file.close();
	//	}
	//}

	//double lowest = 0;
	//lowest = std::min(lowest, InformationManager::Instance().lingScore);
	//lowest = std::min(lowest, InformationManager::Instance().hydraScore);
	//lowest = std::min(lowest, InformationManager::Instance().lurkerScore);
	//lowest = std::min(lowest, InformationManager::Instance().mutaScore);
	//lowest = std::min(lowest, InformationManager::Instance().ultraScore);
	//lowest = std::min(lowest, InformationManager::Instance().guardScore);

	//double highest = 0;
	//highest = std::max(highest, InformationManager::Instance().lingScore);
	//highest = std::max(highest, InformationManager::Instance().hydraScore);
	//highest = std::max(highest, InformationManager::Instance().lurkerScore);
	//highest = std::max(highest, InformationManager::Instance().mutaScore);
	//highest = std::max(highest, InformationManager::Instance().ultraScore);
	//highest = std::max(highest, InformationManager::Instance().guardScore);

	//double offset = (lowest + highest) * -0.5;

	//InformationManager::Instance().lingScore += offset;
	//InformationManager::Instance().hydraScore += offset;
	//InformationManager::Instance().lurkerScore += offset;
	//InformationManager::Instance().mutaScore += offset;
	//InformationManager::Instance().ultraScore += offset;
	//InformationManager::Instance().guardScore += offset;

	//BWAPI::Broodwar->setLatCom(false); //this line made everything way worse than before!
	BWAPI::Broodwar->sendText("AILien 2017-07-22-21-41");
	BWAPI::Broodwar->sendText("Strategy: %s (#%d) Drones: %d", _strategy.c_str(), StratNo, int(strategy.WorkerSupplyBeforeUnits / 2.0 + 4));
	_startingstrategy = _strategy;
	macroHeavynessAtGameStart = macroHeavyness;
}

void GameCommander::onEnd(bool victory)
{
	std::string enemyName = BWAPI::Broodwar->enemy()->getName();
	std::string enemyRaceName = BWAPI::Broodwar->enemy()->getRace().getName();
	std::replace(enemyName.begin(), enemyName.end(), ' ', '_');

	std::string enemyFile = Config::Strategy::WriteDir + enemyName + _sversion + ".txt";
	std::string enemyRaceFile = Config::Strategy::WriteDir + enemyRaceName + _sversion + ".txt";
	std::string totalFile = Config::Strategy::WriteDir + _sversion + ".txt";

	std::stringstream ss;

	ss << "lingScore=" << InformationManager::Instance().lingScore << "\n";
	ss << "hydraScore=" << InformationManager::Instance().hydraScore << "\n";
	ss << "lurkerScore=" << InformationManager::Instance().lurkerScore << "\n";
	ss << "mutaScore=" << InformationManager::Instance().mutaScore << "\n";
	ss << "ultraScore=" << InformationManager::Instance().ultraScore << "\n";
	ss << "guardScore=" << InformationManager::Instance().guardScore << "\n";
	if (victory == true)
	{
		ss << "macroHeavyness=" << macroHeavynessAtGameStart << "\n";
	}
	else
	{
		if (macroHeavyness < macroHeavynessAtGameStart)
		{
			macroHeavyness = macroHeavynessAtGameStart * 0.8;
			ss << "macroHeavyness=" << macroHeavyness << "\n";
		}
		else
		{
			macroHeavyness = macroHeavynessAtGameStart * 1.25;
			ss << "macroHeavyness=" << macroHeavyness << "\n";
		}
	}
	ss << "supplyISawAir=" << InformationManager::Instance().supplyISawAir << "\n";
	ss << "strategy=" << StratNo << "\n";
	if (victory == true)
	{
		ss << "victory=true";
	}
	else
	{
		ss << "victory=false";
	}

	Logger::LogOverwriteToFile(enemyFile, ss.str());
	Logger::LogOverwriteToFile(enemyRaceFile, ss.str());
	Logger::LogOverwriteToFile(totalFile, ss.str());
}
